package com.applitools.eyes.android.common;

import com.applitools.eyes.android.common.utils.ArgumentGuard;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;

/**
 * Eyes test results.
 */
@JsonIgnoreProperties(value = {"$id", "isPassed", "$url"}, ignoreUnknown = true)
public class TestResults {
    @JsonProperty("Steps")
    private int steps;
    @JsonProperty("Matches")
    private int matches;
    @JsonProperty("Mismatches")
    private int mismatches;
    @JsonProperty("Missing")
    private int missing;
    @JsonProperty("ExactMatches")
    private int exactMatches;
    @JsonProperty("StrictMatches")
    private int strictMatches;
    @JsonProperty("ContentMatches")
    private int contentMatches;
    @JsonProperty("LayoutMatches")
    private int layoutMatches;
    @JsonProperty("NoneMatches")
    private int noneMatches;
    @JsonIgnore
    private String url;
    @JsonProperty("IsNew")
    private boolean isNew;
    @JsonProperty("IsDifferent")
    private boolean isDifferent;
    @JsonProperty("IsAborted")
    private boolean isAborted;
    @JsonProperty("Name")
    private String name;
    @JsonProperty("AppName")
    private String appName;

    /**
     * @return The total number of test steps.
     */
    public int getSteps() {
        return steps;
    }

    /**
     * @return The total number of test steps that matched the baseline.
     */
    public int getMatches() {
        return matches;
    }

    /**
     * @return The total number of test steps that did not match the baseline.
     */
    public int getMismatches() {
        return mismatches;
    }

    /**
     * @return The total number of baseline test steps that were missing in
     * the test.
     */
    public int getMissing() {
        return missing;
    }

    /**
     * @return The total number of test steps that exactly matched the baseline.
     */
    @SuppressWarnings("UnusedDeclaration")
    public int getExactMatches() {
        return exactMatches;
    }

    /**
     * @return The total number of test steps that strictly matched the
     * baseline.
     */
    @SuppressWarnings("UnusedDeclaration")
    public int getStrictMatches() {
        return strictMatches;
    }

    /**
     * @return The total number of test steps that matched the baseline by
     * content.
     */
    @SuppressWarnings("UnusedDeclaration")
    public int getContentMatches() {
        return contentMatches;
    }

    /**
     * @return The total number of test steps that matched the baseline by
     * layout.
     */
    @SuppressWarnings("UnusedDeclaration")
    public int getLayoutMatches() {
        return layoutMatches;
    }

    /**
     * @return The total number of test steps that matched the baseline without
     * performing any comparison.
     */
    @SuppressWarnings("UnusedDeclaration")
    public int getNoneMatches() {
        return noneMatches;
    }

    /**
     * @return The URL where test results can be viewed.
     */
    public String getUrl() {
        return url;
    }

    /**
     * @return Whether or not this is a new test.
     */
    public boolean isNew() {
        return isNew;
    }

    /**
     * @return Whether or not this test passed.
     */
    @SuppressWarnings("UnusedDeclaration")
    public boolean isPassed() {
        return (!isNew() && getMismatches() == 0 && getMissing() == 0);
    }

    /**
     * @param steps The number of visual checkpoints in the test.
     */
    @SuppressWarnings("UnusedDeclaration")
    void setSteps(int steps) {
        ArgumentGuard.greaterThanOrEqualToZero(steps, "steps");
        this.steps = steps;
    }

    /**
     * @param matches The number of visual matches in the test.
     */
    @SuppressWarnings("UnusedDeclaration")
    void setMatches(int matches) {
        ArgumentGuard.greaterThanOrEqualToZero(matches, "matches");
        this.matches = matches;
    }

    /**
     * @param mismatches The number of mismatches in the test.
     */
    @SuppressWarnings("UnusedDeclaration")
    void setMismatches(int mismatches) {
        ArgumentGuard.greaterThanOrEqualToZero(mismatches, "mismatches");
        this.mismatches = mismatches;
    }

    /**
     * @param missing The number of visual checkpoints that were available in
     *                the baseline but were not found in the current test.
     */
    @SuppressWarnings("UnusedDeclaration")
    void setMissing(int missing) {
        ArgumentGuard.greaterThanOrEqualToZero(missing, "missing");
        this.missing = missing;
    }

    /**
     * @param exactMatches The number of matches performed with match
     *                     level set to
     *                     {@link com.applitools.eyes.android.common.MatchLevel#EXACT}
     */
    @SuppressWarnings("UnusedDeclaration")
    void setExactMatches(int exactMatches) {
        ArgumentGuard.greaterThanOrEqualToZero(exactMatches, "exactMatches");
        this.exactMatches = exactMatches;
    }

    /**
     * @param strictMatches The number of matches performed with match
     *                     level set to
     *                     {@link com.applitools.eyes.android.common.MatchLevel#STRICT}
     */
    @SuppressWarnings("UnusedDeclaration")
    void setStrictMatches(int strictMatches) {
        ArgumentGuard.greaterThanOrEqualToZero(strictMatches, "strictMatches");
        this.strictMatches = strictMatches;
    }

    /**
     * @param contentMatches The number of matches performed with match
     *                     level set to
     *                     {@link com.applitools.eyes.android.common.MatchLevel#CONTENT}
     */
    @SuppressWarnings("UnusedDeclaration")
    void setContentMatches(int contentMatches) {
        ArgumentGuard.greaterThanOrEqualToZero(contentMatches, "contentMatches");
        this.contentMatches = contentMatches;
    }

    /**
     * @param layoutMatches The number of matches performed with match
     *                     level set to
     *                     {@link com.applitools.eyes.android.common.MatchLevel#LAYOUT}
     */
    @SuppressWarnings("UnusedDeclaration")
    void setLayoutMatches(int layoutMatches) {
        ArgumentGuard.greaterThanOrEqualToZero(layoutMatches, "layoutMatches");
        this.layoutMatches = layoutMatches;
    }

    /**
     * @param noneMatches The number of matches performed with match
     *                     level set to
     *                     {@link com.applitools.eyes.android.common.MatchLevel#NONE}
     */
    @SuppressWarnings("UnusedDeclaration")
    void setNoneMatches(int noneMatches) {
        ArgumentGuard.greaterThanOrEqualToZero(noneMatches, "noneMatches");
        this.noneMatches = noneMatches;
    }

    /**
     * @param url The URL of the test results.
     */
    public void setUrl(String url) {
        this.url = url;
    }

    /**
     * @param isNew Whether or not this test has an existing baseline.
     */
    public void setNew(boolean isNew) {
        this.isNew = isNew;
    }

    public boolean isDifferent() {
        return isDifferent;
    }

    public void setDifferent(boolean isDifferent) {
        this.isDifferent = isDifferent;
    }

    public boolean isAborted() {
        return isAborted;
    }

    public void setAborted(boolean isAborted) {
        this.isAborted = isAborted;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public String getAppName() {
        return appName;
    }

    public void setAppName(String appName) {
        this.appName = appName;
    }

    @Override
    public String toString() {
        String isNewTestStr = isNew ? "New test" : "Existing test";
        return isNewTestStr + " [ steps: " + getSteps()
                + ", matches: " + getMatches()
                + ", mismatches:" + getMismatches() + ", missing: "
                + getMissing() + "] , URL: " + getUrl();
    }
}
