package com.applitools.eyes.android.common.logger;

import com.applitools.eyes.android.common.utils.ArgumentGuard;

/**
 * Logs trace messages.
 */
public class Logger {

    public static final String LOGGER_NAME = "EyesLog";

    private LogHandler mLogHandler = null;

    public Logger() {
        mLogHandler = new NullLogHandler();
    }

    /**
     * @return The currently set log handler.
     */
    public LogHandler getLogHandler() {
        return mLogHandler;
    }

    /**
     * Sets the log handler.
     * @param handler The log handler to set. If you want a log handler which
     *                does nothing, use {@link
     *                com.applitools.eyes.android.common.logger.NullLogHandler}.
     */
    public void setLogHandler(LogHandler handler) {
        ArgumentGuard.notNull(handler, "handler");

        mLogHandler = handler;
    }

    /**
     * @return The name of the method which called the logger, if possible,
     * or an empty string.
     */
    private String getPrefix() {
        StackTraceElement[] stackTraceElements =
                Thread.currentThread().getStackTrace();

        String prefix = "";
        // getStackTrace()<-getPrefix()<-log()/verbose()<-"actual caller"
        if (stackTraceElements != null && stackTraceElements.length >= 4) {
            prefix = stackTraceElements[3].getMethodName() + "():";
        }

        return prefix;
    }

    /**
     * Writes a verbose write message.
     *
     * @param message The message to log as verbose.
     */
    public void verbose(String message) {
        mLogHandler.onMessage(true, getPrefix() + message);
    }

    /**
     * Writes a (non-verbose) write message.
     *
     * @param message The message to log.
     */
    public void log(String message) {
        mLogHandler.onMessage(false, getPrefix() + message);
    }
}
