package com.applitools.eyes.android.espresso;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.graphics.Bitmap;
import android.os.Build;
import android.view.View;

import com.applitools.eyes.android.common.AbstractProxySettings;
import com.applitools.eyes.android.common.EyesRunner;
import com.applitools.eyes.android.common.EyesScreenshot;
import com.applitools.eyes.android.common.IBatchCloser;
import com.applitools.eyes.android.common.ImageMatchSettings;
import com.applitools.eyes.android.common.Location;
import com.applitools.eyes.android.common.MatchLevel;
import com.applitools.eyes.android.common.Property;
import com.applitools.eyes.android.common.RectangleSize;
import com.applitools.eyes.android.common.TestResults;
import com.applitools.eyes.android.common.config.Configuration;
import com.applitools.eyes.android.common.config.IConfigurationGetter;
import com.applitools.eyes.android.common.config.IConfigurationProvider;
import com.applitools.eyes.android.common.config.IConfigurationSetter;
import com.applitools.eyes.android.common.utils.ArgumentGuard;
import com.applitools.eyes.android.components.IComponentsProvider;
import com.applitools.eyes.android.components.support.SupportComponentsProvider;
import com.applitools.eyes.android.core.EmptyRegionProvider;
import com.applitools.eyes.android.core.EyesBase;
import com.applitools.eyes.android.common.Region;
import com.applitools.eyes.android.core.ScreenshotProvider;
import com.applitools.eyes.android.core.TestFailedException;
import com.applitools.eyes.android.core.fluent.ICheckSettings;
import com.applitools.eyes.android.core.fluent.ICheckSettingsInternal;
import com.applitools.eyes.android.espresso.exceptions.GoogleMapScreenshotTimeoutException;
import com.applitools.eyes.android.espresso.fluent.IEspressoCheckTarget;
import com.applitools.eyes.android.espresso.fluent.Target;
import com.applitools.eyes.android.espresso.utils.ImageUtils;
import com.applitools.eyes.android.espresso.utils.LocationUtils;

import org.hamcrest.Matcher;

import java.net.URI;

public class Eyes extends EyesBase implements IConfigurationProvider, IBatchCloser {

    /**
     * Default timeout to get screenshot from GoogleMap
     */
    public static final int GOOGLE_MAP_SCREENSHOT_TIMEOUT = 15 * 1000;

    private boolean mStitchContent = false;

    private Configuration configuration = new Configuration();

    private IComponentsProvider mComponentsProvider;

    private EyesRunner mEyesRunner;

    public boolean shouldStitchContent() {
        return mStitchContent;
    }

    public Eyes(URI serverUrl) {
        super(serverUrl);
        setConfigurationProvider(this);
        if (mEyesRunner == null) {
            mEyesRunner = new ClassicRunner();
        }
    }

    public Eyes() {
        this(getDefaultServerUrl());
    }

    public Eyes(EyesRunner eyesRunner) {
        this(getDefaultServerUrl());
        mEyesRunner = eyesRunner == null ? new ClassicRunner() : eyesRunner;
    }

    @Override
    protected String getBaseAgentId() {
        String version = "4.5.3";
        return "eyes-espresso/" + version;
    }

    public boolean isHideCaret() {
        return configuration.getHideCaret();
    }

    public void setHideCaret(boolean hideCaret) {
        if (configuration != null) {
            configuration.setHideCaret(hideCaret);
        }
    }

    @Override
    protected RectangleSize getViewportSize() {
        verifyComponentsProvider();
        RectangleSize viewportSize = configuration.getViewportSize();
        if (viewportSize == null) {
            viewportSize = mComponentsProvider.getViewportSize();
        }
        return viewportSize;
    }

    @Override
    protected void setViewportSize(RectangleSize size) {
        configuration.setViewportSize(size);
    }

    /**
     * Forces a full page screenshot (by scrolling and stitching) if the
     * browser only supports viewport screenshots).
     *
     * @param shouldForce Whether to force a full page screenshot or not.
     */
    public void setForceFullPageScreenshot(boolean shouldForce) {
        configuration.setForceFullPageScreenshot(shouldForce);
    }

    /**
     * @return Whether Eyes should force a full page screenshot.
     */
    public boolean getForceFullPageScreenshot() {
        return configuration.getForceFullPageScreenshot();
    }

    /**
     * See {@link #open(String, String)}.
     */
    public void open(String testName) {
        verifyComponentsProvider();
        open(mComponentsProvider.getAppName(), testName);
    }

    /**
     * See {@link #open(String, String, RectangleSize)}.
     * {@code viewportSize} defaults to {@code null}.
     * {@code sessionType} defaults to {@code null}.
     */
    public void open(String appName, String testName) {
        open(appName, testName, null);
    }

    /**
     * Starts a test.
     *
     * @param appName        The name of the application under test.
     * @param testName       The test name.
     * @param viewportSize   The screen viewport size.
     */
    protected void open(String appName, String testName, RectangleSize viewportSize ) {
        if (getIsDisabled()) {
            mLogger.verbose("Ignored");
            return;
        }
        configuration.setAppName(appName);
        configuration.setTestName(testName);
        configuration.setViewportSize(viewportSize != null ? viewportSize : getViewportSize());
        configuration.setSessionType(null);

        openBase();

        mEyesRunner.addBatch(this.configuration.getBatch().getId(), this);
    }

    /**
     * Add property to the session info.
     * @param name          The name of property.
     * @param value         The value of property.
     */
    public void addProperty(String name, String value) {
        mProperties.add(new Property(name, value));
    }

    public void check(ICheckSettings checkSettings) {
        this.check(null, checkSettings);
    }

    public void check(String name, ICheckSettings checkSettings) {
        ArgumentGuard.notNull(checkSettings, "checkSettings");

        verifyComponentsProvider();

        mLogger.verbose(String.format("check(\"%s\", checkSettings) - begin", name));

        final ICheckSettingsInternal checkSettingsInternal = (ICheckSettingsInternal) checkSettings;
        IEspressoCheckTarget espressoCheckTarget = (checkSettings instanceof IEspressoCheckTarget) ? (IEspressoCheckTarget) checkSettings : null;

        this.mStitchContent = checkSettingsInternal.getStitchContent();
        boolean tmpForceFullPage = getForceFullPageScreenshot();
        if (checkSettingsInternal.getFullpageScreenshot() != null) {
            this.setForceFullPageScreenshot(checkSettingsInternal.getFullpageScreenshot());
        }

        final Region targetRegion = checkSettingsInternal.getTargetRegion();

        if (targetRegion != null) {
            this.checkWindowBase(new EmptyBitmapProvider() {
                @Override
                public Region getRegion() {
                    return targetRegion;
                }

                @Override
                public Location getLocation() {
                    return LocationUtils.getLocation(targetRegion);
                }
            }, name, false, checkSettings);
        } else if (espressoCheckTarget != null) {
            if (espressoCheckTarget.isGoogleMap()) {
                checkGoogleMapFragment(espressoCheckTarget.getId(), !espressoCheckTarget.isNotSupportMap(), name, GOOGLE_MAP_SCREENSHOT_TIMEOUT);
            } else if (espressoCheckTarget.isDialogView()) {
                View dialogView = mComponentsProvider.getDialogView();
                if (dialogView == null) {
                    mLogger.verbose("Dialog is not presented. Couldn't take the screenshot");
                    return;
                }
                this.checkWindowBase(new ViewRegionProvider(dialogView, ((ICheckSettingsInternal) checkSettings).getHideCaret(), mComponentsProvider), name, false, checkSettings);
            } else if (espressoCheckTarget.isPopupView()) {
                View popupView = mComponentsProvider.getPopupView();
                if (popupView == null) {
                    mLogger.verbose("Popup is not presented. Couldn't take the screenshot");
                    return;
                }
                this.checkWindowBase(new ViewRegionProvider(popupView, ((ICheckSettingsInternal) checkSettings).getHideCaret(), mComponentsProvider), name, false, checkSettings);
            } else {
                Matcher targetMatcher = espressoCheckTarget.getTargetMatcher();
                View targetElement = espressoCheckTarget.getTargetView();
                if (targetElement == null && targetMatcher != null) {
                    targetElement = mComponentsProvider.findView(targetMatcher);
                }
                if (targetElement != null) {
                    this.checkWindowBase(new ViewRegionProvider(targetElement, ((ICheckSettingsInternal) checkSettings).getHideCaret(), mComponentsProvider),
                            name, false, checkSettings);
                } else {
                    this.checkWindowBase(EmptyRegionProvider.INSTANCE, name, false, checkSettings);
                }
            }
        }

        this.mStitchContent = false;
        setForceFullPageScreenshot(tmpForceFullPage);

        mLogger.verbose("check - done!");
    }

    /**
     * See {@link #checkWindow(String)}.
     * <br>{@code tag} defaults to {@code null}.
     */
    public void checkWindow() {
        checkWindow(null);
    }

    /**
     * Takes a snapshot of the application under test and matches it with
     * the expected output.
     *
     * @param tag An optional tag to be associated with the snapshot.
     * @throws TestFailedException Thrown if a mismatch is detected and
     *                             immediate failure reports are enabled.
     */
    @SuppressLint("DefaultLocale")
    public void checkWindow(String tag) {

        if (getIsDisabled()) {
            mLogger.log(String.format("CheckWindow('%s'): Ignored", tag));
            return;
        }

        mLogger.log(String.format("CheckWindow('%s')", tag));

        this.check(tag, Target.window().withName(tag).hideCaret(configuration.getHideCaret()));
    }

    /**
     * See {@link #checkRegion(Region, String)}.
     * <br>{@code tag} defaults to {@code null}.
     */
    @SuppressWarnings("UnusedDeclaration")
    public void checkRegion(Region region) {
        checkRegion(region, null);
    }

    /**
     * Takes a snapshot of the application under test and matches a specific
     * region within it with the expected output.
     *
     * @param region       A non empty region representing the screen region to
     *                     check.
     * @param tag          An optional tag to be associated with the snapshot.
     * @throws TestFailedException Thrown if a mismatch is detected and
     *                             immediate failure reports are enabled.
     */
    @SuppressLint("DefaultLocale")
    public void checkRegion(final Region region, String tag) {
        if (getIsDisabled()) {
            mLogger.log(String.format("CheckRegion([%s], '%s'): Ignored", region, tag));
            return;
        }

        ArgumentGuard.notNull(region, "region");

        mLogger.verbose(String.format("CheckRegion([%s], '%s')", region, tag));

        this.check(tag, Target.region(region).withName(tag).hideCaret(configuration.getHideCaret()));
    }

    /**
     * See {@link #checkRegion(Matcher, String)}.
     * <br>{@code tag} defaults to {@code null}.
     */
    @SuppressWarnings("UnusedDeclaration")
    public void checkRegion(Matcher matcher) {
        checkRegion(matcher, null);
    }

    /**
     * Takes a snapshot of the application under test and matches a region of
     * a specific element with the expected region output.
     *
     * @param matcher Selects the view to check
     * @param tag     An optional tag to be associated with the snapshot.
     * @throws TestFailedException if a mismatch is detected and
     *                             immediate failure reports are enabled
     */
    @SuppressWarnings("UnusedDeclaration")
    public void checkRegion(Matcher matcher, String tag) {
        if (getIsDisabled()) {
            mLogger.log(String.format("CheckRegion([%s], '%s'): Ignored", matcher, tag));
            return;
        }

        ArgumentGuard.notNull(matcher, "matcher");

        mLogger.verbose(String.format("CheckRegion([%s], '%s')", matcher, tag));

        this.check(tag, Target.region(matcher).hideCaret(configuration.getHideCaret()));
    }

    /**
     * See {@link #checkElement(int, String)}.
     * {@code tag} defaults to {@code null}.
     */
    public void checkElement(int elementId) {
        checkElement(elementId, null);
    }

    /**
     * Takes a snapshot of the application under test and matches an element
     * specified by the given selector with the expected region output.
     *
     * @param elementId Selects the element to check.
     * @param tag     An optional tag to be associated with the screenshot.
     * @throws TestFailedException if a mismatch is detected and
     *                             immediate failure reports are enabled
     */
    public void checkElement(int elementId, String tag) {
        if (getIsDisabled()) {
            mLogger.log(String.format("CheckElement(elementId, '%s'): Ignored", tag));
            return;
        }

        verifyComponentsProvider();
        checkElement(mComponentsProvider.findView(elementId), tag);
    }

    /**
     * Takes a snapshot of the application under test and matches a specific
     * element with the expected region output.
     *
     * @param element      The element to check.
     * @param tag          An optional tag to be associated with the snapshot.
     * @throws TestFailedException if a mismatch is detected and
     *                             immediate failure reports are enabled
     */
    protected void checkElement(final View element, String tag) {
        if (getIsDisabled()) {
            mLogger.log(String.format("checkElement([%s], '%s'): Ignored", element, tag));
            return;
        }

        ArgumentGuard.notNull(element, "element");

        mLogger.verbose(String.format("checkElement([%s], '%s')", element, tag));

        this.check(tag, Target.region(element).withName(tag).hideCaret(configuration.getHideCaret()));
    }

    /**
     * See {@link #checkWindowAllLayers(String)}.
     * {@code tag} defaults to {@code null}.
     */
    public void checkWindowAllLayers() {
        checkWindowAllLayers(null);
    }

    /**
     * Takes a snapshot of all application layers under test
     * Should be used only when AlertDialog or Dialog presented on the screen.
     * Otherwise you test can be hanged for a 60 seconds.
     *
     * @param tag          An optional tag to be associated with the snapshot.
     * @throws TestFailedException if a mismatch is detected and
     *                             immediate failure reports are enabled
     */
    public void checkWindowAllLayers(String tag) {
        if (getIsDisabled()) {
            mLogger.log(String.format("CheckWindowAllLayers('%s'): Ignored", tag));
            return;
        }

        mLogger.log(String.format("CheckWindowAllLayers('%s')", tag));

        this.check(tag, Target.window().includeAllLayers().hideCaret(configuration.getHideCaret()));
    }

    /**
     * See {@link #checkGoogleMapFragment(int, boolean, String, long)}.
     * <br>{@code tag} defaults to {@code null}
     * <br>{@code timeout} defaults to {@link #GOOGLE_MAP_SCREENSHOT_TIMEOUT}
     */
    @SuppressWarnings("UnusedDeclaration")
    public void checkGoogleMapFragment(int fragmentId, boolean isSupportMapFragment) {
        checkGoogleMapFragment(fragmentId, isSupportMapFragment, null, GOOGLE_MAP_SCREENSHOT_TIMEOUT);
    }

    /**
     * See {@link #checkGoogleMapFragment(int, boolean, String, long)}.
     * <br>{@code tag} defaults to {@code null}
     */
    @SuppressWarnings("UnusedDeclaration")
    public void checkGoogleMapFragment(int fragmentId, boolean isSupportMapFragment, long timeout) {
        checkGoogleMapFragment(fragmentId, isSupportMapFragment, null, timeout);
    }

    /**
     * See {@link #checkGoogleMapFragment(int, boolean, String, long)}.
     * <br>{@code timeout} defaults to {@link #GOOGLE_MAP_SCREENSHOT_TIMEOUT}
     */
    @SuppressWarnings("UnusedDeclaration")
    public void checkGoogleMapFragment(int fragmentId, boolean isSupportMapFragment, String tag) {
        checkGoogleMapFragment(fragmentId, isSupportMapFragment, tag, GOOGLE_MAP_SCREENSHOT_TIMEOUT);
    }

    /**
     * Takes a snapshot of the application under test and matches a region of
     * a GoogleMap fragment with the expected region output.
     *
     * @param fragmentId           Indicates specific fragment.
     * @param isSupportMapFragment Needs to check for MapFragment from support library.
     * @param tag                  An optional tag to be associated with the snapshot.
     * @param timeout              An optional timeout for taking snapshot from GoogleMap
     * @throws TestFailedException if a mismatch is detected and
     *                             immediate failure reports are enabled
     */
    @SuppressLint("DefaultLocale")
    public void checkGoogleMapFragment(int fragmentId, boolean isSupportMapFragment, final String tag, long timeout) {
        if (getIsDisabled()) {
            mLogger.log(String.format("CheckGoogleMapFragment([%d], '%b', '%s'): Ignored", fragmentId, isSupportMapFragment, tag));
            return;
        }
        mLogger.verbose(String.format("CheckGoogleMapFragment([%d], '%b', '%s')", fragmentId, isSupportMapFragment, tag));

        verifyComponentsProvider();

        final Activity activity = mComponentsProvider.getCurrentActivity();

        final Bitmap bitmap = mComponentsProvider.getGoogleMapFragmentScreenshot(activity, fragmentId,
                isSupportMapFragment, timeout);

        if (bitmap == null) {
            throw new GoogleMapScreenshotTimeoutException("Can not get screenshot from GoogleMap");
        }

        Eyes.super.checkWindowBase(
                new BitmapProvider() {
                    @Override
                    public byte[] getImage() {
                        return ImageUtils.bitmapToBytes(bitmap);
                    }

                    @Override
                    public Location getLocation() {
                        return LocationUtils.empty();
                    }
                },
                tag,
                false,
                configuration.getHideCaret()
        );
    }

    /**
     * See {@link #checkGoogleMapView(int, String, long)}.
     * <br>{@code tag} defaults to {@code null}
     * <br>{@code timeout} defaults to {@link #GOOGLE_MAP_SCREENSHOT_TIMEOUT}
     */
    @SuppressWarnings("UnusedDeclaration")
    public void checkGoogleMapView(int mapViewId) {
        checkGoogleMapView(mapViewId, null, GOOGLE_MAP_SCREENSHOT_TIMEOUT);
    }

    /**
     * See {@link #checkGoogleMapView(int, String, long)}.
     * <br>{@code tag} defaults to {@code null}
     */
    @SuppressWarnings("UnusedDeclaration")
    public void checkGoogleMapView(int mapViewId, long timeout) {
        checkGoogleMapView(mapViewId, null, timeout);
    }

    /**
     * See {@link #checkGoogleMapView(int, String, long)}.
     * <br>{@code timeout} defaults to {@link #GOOGLE_MAP_SCREENSHOT_TIMEOUT}
     */
    @SuppressWarnings("UnusedDeclaration")
    public void checkGoogleMapView(int mapViewId, String tag) {
        checkGoogleMapView(mapViewId, tag, GOOGLE_MAP_SCREENSHOT_TIMEOUT);
    }

    /**
     * Takes a snapshot of the application under test and matches a region of
     * a GoogleMap with the expected region output.
     *
     * @param mapViewId Indicates a specific presented MapView.
     * @param tag       An optional tag to be associated with the snapshot.
     * @param timeout   An optional timeout for taking snapshot from GoogleMap
     * @throws TestFailedException if a mismatch is detected and
     *                             immediate failure reports are enabled
     */
    @SuppressLint("DefaultLocale")
    public void checkGoogleMapView(int mapViewId, final String tag, long timeout) {
        if (getIsDisabled()) {
            mLogger.log(String.format("CheckGoogleMapView([%d], '%s'): Ignored", mapViewId, tag));
            return;
        }
        mLogger.verbose(String.format("CheckGoogleMapView([%d], '%s')", mapViewId, tag));

        verifyComponentsProvider();

        Activity activity = mComponentsProvider.getCurrentActivity();

        final Bitmap bitmap = mComponentsProvider.getGoogleMapViewScreenshot(activity, mapViewId, timeout);

        if (bitmap == null) {
            throw new GoogleMapScreenshotTimeoutException("Can not get screenshot from GoogleMap");
        }

        Eyes.super.checkWindowBase(
                new BitmapProvider() {
                    @Override
                    public byte[] getImage() {
                        return ImageUtils.bitmapToBytes(bitmap);
                    }

                    @Override
                    public Location getLocation() {
                        return LocationUtils.empty();
                    }
                },
                tag,
                false,
                configuration.getHideCaret()
        );
    }

    @Override
    protected EyesScreenshot getScreenshot(boolean includeAllLayers, boolean hideCaret) {
        mLogger.verbose("getScreenshot()");
        EyesScreenshot result;

        boolean tmpForceFullPage = getForceFullPageScreenshot();
        boolean tmpStitchContent = mStitchContent;

        if (includeAllLayers) {
            setForceFullPageScreenshot(false);
            mStitchContent = false;
        }

        ScreenshotProvider imageProvider;
        verifyComponentsProvider();
        mLogger.log("Using " + mComponentsProvider.getClass().getSimpleName());

        if (getForceFullPageScreenshot() || mStitchContent) {
            mLogger.verbose("Full page screenshot requested");
            imageProvider = new FullPageScreenshotProvider(includeAllLayers, hideCaret,
                    mComponentsProvider, mLogger);
        } else {
            mLogger.verbose("Frame screenshot requested");
            imageProvider = new CurrentFrameScreenshotProvider(includeAllLayers, hideCaret,
                    mComponentsProvider);
        }

        setForceFullPageScreenshot(tmpForceFullPage);
        mStitchContent = tmpStitchContent;

        mLogger.verbose("Getting screenshot from ScreenshotProvider...");
        result = new EyesImageScreenshot(mLogger, imageProvider.getImage());
        mLogger.verbose("Screenshot received...");
        return result;
    }

    @Override
    protected EyesScreenshot getScreenshot(byte[] image) {
        mLogger.verbose("getScreenshot(Bitmap)");

        return new EyesImageScreenshot(mLogger, image);
    }

    @Override
    protected String getTitle() {
        verifyComponentsProvider();
        return mComponentsProvider.getCurrentScreenName();
    }

    @Override
    protected String getDeviceModel() {
        return Build.MODEL;
    }

    @Override
    protected String getOsName() {
        return "Android " + Build.VERSION.RELEASE;
    }

    @Override
    protected String getApplicationNameFromPackage() {
        verifyComponentsProvider();
        return mComponentsProvider.getAppName();
    }

    /**
     * Sets the user given agent id of the SDK. {@code null} is referred to
     * as no id.
     *
     * @param agentId The agent ID to set.
     */
    public void setAgentId(String agentId) {
        configuration.setAgentId(agentId);
    }

    /**
     * @return The user given agent id of the SDK.
     */
    public String getAgentId() {
        return configuration.getAgentId();
    }

    /**
     * Gets ignore caret.
     *
     * @return Whether to ignore or the blinking caret or not when comparing images.
     */
    public boolean getIgnoreCaret() {
        return configuration.getIgnoreCaret();
    }

    /**
     * Sets the ignore blinking caret value.
     *
     * @param value The ignore value.
     */
    public void setIgnoreCaret(boolean value) {
        configuration.setIgnoreCaret(value);
    }

    /**
     * Sets app name.
     *
     * @param appName The name of the application under test.
     */
    public void setAppName(String appName) {
        this.configuration.setAppName(appName);
    }

    /**
     * Gets app name.
     *
     * @return The name of the application under test.
     */
    public String getAppName() {
        return configuration.getAppName();
    }

    /**
     * Automatically save differences as a baseline.
     * @param saveDiffs Sets whether to automatically save differences as baseline.
     */
    public void setSaveDiffs(Boolean saveDiffs) {
        configuration.setSaveDiffs(saveDiffs);
    }

    /**
     * Returns whether to automatically save differences as a baseline.
     * @return Whether to automatically save differences as baseline.
     */
    public Boolean getSaveDiffs() {
        return configuration.getSaveDiffs();
    }

    /**
     * This function is deprecated. Please use
     * {@link #setDefaultMatchSettings} instead.
     * <p>
     * The test-wide match level to use when checking application screenshot
     * with the expected output.
     *
     * @param matchLevel The match level setting.
     * @see com.applitools.eyes.android.common.MatchLevel
     */
    public void setMatchLevel(MatchLevel matchLevel) {
        configuration.setMatchLevel(matchLevel);
    }

    /**
     * @deprecated  Please use{@link #getDefaultMatchSettings} instead.
     * @return The test-wide match level.
     */
    public MatchLevel getMatchLevel() {
        return configuration.getMatchLevel();
    }

    /**
     * Updates the match settings to be used for the session.
     *
     * @param defaultMatchSettings The match settings to be used for the
     *                             session.
     */
    public void setDefaultMatchSettings(ImageMatchSettings defaultMatchSettings) {
        ArgumentGuard.notNull(defaultMatchSettings, "defaultMatchSettings");
        configuration.setDefaultMatchSettings(defaultMatchSettings);
    }

    /**
     *
     * @return The match settings used for the session.
     */
    public ImageMatchSettings getDefaultMatchSettings() {
        return configuration.getDefaultMatchSettings();
    }

    /**
     * Set whether or not new tests are saved by default.
     *
     * @param saveNewTests True if new tests should be saved by default.
     *                     False otherwise.
     */
    public void setSaveNewTests(boolean saveNewTests) {
        configuration.setSaveNewTests(saveNewTests);
    }

    /**
     * @return True if new tests are saved by default.
     */
    public boolean getSaveNewTests() {
        return configuration.getSaveNewTests();
    }

    /**
     * Set whether or not failed tests are saved by default.
     *
     * @param saveFailedTests True if failed tests should be saved by
     *                        default, false otherwise.
     */
    public void setSaveFailedTests(boolean saveFailedTests) {
        configuration.setSaveFailedTests(saveFailedTests);
    }

    /**
     * @return True if failed tests are saved by default.
     */
    public boolean getSaveFailedTests() {
        return configuration.getSaveFailedTests();
    }

    /**
     * Sets the maximum time (in ms) a match operation tries to perform a match.
     *
     * @param ms Total number of ms to wait for a match.
     */
    public void setMatchTimeout(int ms) {
        configuration.setMatchTimeout(ms);
    }

    /**
     * Gets match timeout.
     *
     * @return The maximum time in ms waits for a match.
     */
    public int getMatchTimeout() {
        return configuration.getMatchTimeout();
    }

    /**
     * Sets the branch in which the baseline for subsequent test runs resides.
     * If the branch does not already exist it will be created under the
     * specified parent branch (see {@link #setParentBranchName}).
     * Changes to the baseline or model of a branch do not propagate to other
     * branches.
     *
     * @param branchName Branch name or {@code null} to specify the default
     *                   branch.
     */
    public void setBranchName(String branchName) {
        configuration.setBranchName(branchName);
    }

    /**
     *
     * @return The current branch (see {@link #setBranchName(String)}).
     */
    public String getBranchName() {
        return configuration.getBranchName();
    }

    /**
     * Sets the branch under which new branches are created. (see {@link
     * #setBranchName(String)}.
     *
     * @param branchName Branch name or {@code null} to specify the default
     *                   branch.
     */
    public void setParentBranchName(String branchName) {
        configuration.setParentBranchName(branchName);
    }

    /**
     *
     * @return The name of the current parent branch under which new branches
     * will be created. (see {@link #setParentBranchName(String)}).
     */
    public String getParentBranchName() {
        return configuration.getParentBranchName();
    }

    /**
     * Sets the branch under which new branches are created. (see {@link
     * #setBranchName(String)}.
     * @param branchName Branch name or {@code null} to specify the default branch.
     */
    public void setBaselineBranchName(String branchName) {
        configuration.setBaselineBranchName(branchName);
    }

    /**
     * @return The name of the current parent branch under which new branches
     * will be created. (see {@link #setBaselineBranchName(String)}).
     */
    public String getBaselineBranchName() {
        return configuration.getBaselineBranchName();
    }

    /**
     * Sets baseline name.
     *
     * @param baselineName If specified, determines the baseline to compare with and disables automatic baseline inference.
     * @deprecated Only available for backward compatibility. See {@link #setBaselineEnvName(String)}.
     */
    @Deprecated
    public void setBaselineName(String baselineName) {
        setBaselineEnvName(baselineName);
    }

    /**
     * Gets baseline name.
     *
     * @return The baseline name, if specified.
     * @deprecated Only available for backward compatibility. See {@link #getBaselineEnvName()}.
     */
    @Deprecated
    public String getBaselineName() {
        return getBaselineEnvName();
    }

    /**
     * If not {@code null}, determines the name of the environment of the baseline.
     *
     * @param baselineEnvName The name of the baseline's environment.
     */
    public void setBaselineEnvName(String baselineEnvName) {
        configuration.setBaselineEnvName(baselineEnvName);
    }

    /**
     * If not {@code null}, determines the name of the environment of the baseline.
     *
     * @return The name of the baseline's environment, or {@code null} if no such name was set.
     */
    public String getBaselineEnvName() {
        return configuration.getBaselineEnvName();
    }

    /**
     * Sets the proxy settings to be used by the rest client.
     *
     * @param proxySettings The proxy settings to be used by the rest client.
     *                      If {@code null} then no proxy is set.
     */
    public void setProxy(ProxySettings proxySettings) {
        if (configuration != null) {
            configuration.setProxy(proxySettings);
        }
        super.setProxy(proxySettings);
    }

    public Configuration getConfiguration() {
        return new Configuration(configuration);
    }

    public void setConfiguration(Configuration configuration) {
        ArgumentGuard.notNull(configuration, "configuration");
        String apiKey = configuration.getApiKey();
        if (apiKey != null) {
            setApiKey(apiKey);
        }
        URI serverUrl = configuration.getServerUrl();
        if (serverUrl != null) {
            setServerUrl(serverUrl);
        }
        AbstractProxySettings proxy = configuration.getProxy();
        if (proxy != null) {
            setProxy(proxy);
        }
        this.configuration = new Configuration(configuration);
        if (mComponentsProvider != null) {
            mComponentsProvider.setFeatures(this.configuration.getFeatures());
        }
    }

    @Override
    public IConfigurationGetter get() {
        return configuration;
    }

    @Override
    public IConfigurationSetter set() {
        return configuration;
    }

    public void setComponentsProvider(IComponentsProvider componentsProvider) {
        this.mComponentsProvider = componentsProvider;
        this.mComponentsProvider.setLogger(mLogger);
        componentsProvider.setFeatures(configuration.getFeatures());
    }

    public IComponentsProvider getComponentsProvider() {
        verifyComponentsProvider();
        return mComponentsProvider;
    }

    @Override
    public TestResults close(boolean throwEx) {
        TestResults results = null;
        try {
            results = super.close(throwEx);
        } catch (Throwable e) {
            if (throwEx) {
                throw e;
            }
        }
        if (mEyesRunner != null) {
            mEyesRunner.aggregateResult(results);
        }
        return results;
    }

    @Override
    public void closeBatch(String batchId) {
        doCloseBatch(batchId);
    }

    private void verifyComponentsProvider() {
        if (mComponentsProvider == null) {
            mComponentsProvider = new SupportComponentsProvider();
            mComponentsProvider.setLogger(mLogger);
        }
    }

    @Override
    public IConfigurationProvider getConfigurationProvider() {
        return this;
    }
}
