package com.applitools.eyes.android.espresso;

import android.app.Activity;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.view.View;

import com.applitools.eyes.android.common.logger.Logger;
import com.applitools.eyes.android.components.IComponentsProvider;
import com.applitools.eyes.android.espresso.utils.ImageUtils;
import com.applitools.eyes.android.common.Region;
import com.applitools.eyes.android.core.ScreenshotProvider;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

class FullPageScreenshotProvider implements ScreenshotProvider {

    private boolean mIncludeAllLayers;
    private boolean mHideCaret;
    private IComponentsProvider mComponentsProvider;

    private Logger mLogger;
    private static final String TAG = "";

    public FullPageScreenshotProvider(boolean includeAllLayers, boolean hideCaret,
                                      IComponentsProvider componentsProvider, Logger logger) {
        this.mIncludeAllLayers = includeAllLayers;
        this.mHideCaret = hideCaret;
        this.mComponentsProvider = componentsProvider;
        this.mLogger = logger;
    }

    @Override
    public byte[] getImage() {
        Bitmap bitmap;

        Activity activity = mComponentsProvider.getCurrentActivity();
        try {
            bitmap = getFullPageScreenshot(activity);
        } catch (Exception e) {
            throw new RuntimeException("Unable to capture screenshot.", e);
        }

        return ImageUtils.bitmapToBytes(bitmap);
    }

    private Bitmap getFullPageScreenshot(Activity activity) throws InterruptedException, IOException {
        View scrollableView = mComponentsProvider.findScrollableView(activity);

        if (scrollableView != null) {
            mLogger.log("Found scrollableView: " + scrollableView.getClass().getName() +
                    " | height: " + scrollableView.getHeight() + " | visibility: " + scrollableView.getVisibility());
        }
        return getFullScreenshot(activity, scrollableView);
    }

    private Bitmap getFullScreenshot(final Activity activity, final View scrollableView) throws InterruptedException, IOException {
        Bitmap result = null;
        if (scrollableView != null && scrollableView.getHeight() != 0) {
            mLogger.log(TAG + "Scrollable view instance of " + scrollableView.getClass().getSimpleName());

            int maxScrollableHeight = mComponentsProvider.getFullScrollableHeight(activity, scrollableView);

            mLogger.log(TAG + "Max scrollable height = " + maxScrollableHeight);

            mLogger.log(TAG + "Scrollable view height = " + scrollableView.getHeight());
            if (scrollableView.getHeight() == 0 ||
                    (maxScrollableHeight <= scrollableView.getHeight())) {
                return mComponentsProvider.captureViewportScreenshot(activity, false, true,
                        mIncludeAllLayers, mHideCaret);
            }

            int behaviorOffset = mComponentsProvider.getBehaviorOffset(activity, scrollableView);
            mLogger.log(TAG + "BehaviorOffset = " + behaviorOffset);
            boolean behaviorScrolled = false;

            scrollableView.setVerticalScrollBarEnabled(false);
            scrollableView.setHorizontalScrollBarEnabled(false);

            int oneScrollStep = scrollableView.getHeight() - scrollableView.getPaddingBottom() - scrollableView.getPaddingTop() - behaviorOffset;
            int maxScrollSteps = maxScrollableHeight / oneScrollStep;

            mLogger.log(TAG + "One scroll step = " + oneScrollStep);
            mLogger.log(TAG + "maxScrollSteps = " + (double) maxScrollableHeight / oneScrollStep);

            List<Bitmap> imagesList = new ArrayList<>();
            int[] location = new int[2];
            scrollableView.getLocationInWindow(location);

            mLogger.log(TAG + "Scrollable view location [" + location[0] + ", " + location[1] + "]");
            mLogger.log(TAG + "Scrollable view size [" + scrollableView.getWidth() + ", " + scrollableView.getHeight() + "]");

            Bitmap bitmap;
            if ((location[1] - mComponentsProvider.getStatusBarHeight(activity)) != 0) {
                 bitmap = mComponentsProvider.captureFrame(activity,
                        new Region(Region.FULL_WIDTH,
                                location[1] - mComponentsProvider.getStatusBarHeight(activity),
                                mComponentsProvider.getStatusBarHeight(activity),
                                0), mHideCaret);
                imagesList.add(bitmap);
            }

            for (int step = 0; step <= maxScrollSteps; step++) {
                Bitmap scrollableViewBitmap = mComponentsProvider.captureFrame(activity,
                        new Region(Region.FULL_WIDTH,
                                scrollableView.getHeight() - scrollableView.getPaddingBottom() - (step != 0 ? scrollableView.getPaddingTop() : 0) - behaviorOffset,
                                location[1] + (step != 0 ? scrollableView.getPaddingTop() : 0),
                                0), mHideCaret);

                if (step == maxScrollSteps && maxScrollSteps != 0) {
                    int cropTo = maxScrollableHeight - (oneScrollStep * (step));
                    int cropFrom = oneScrollStep - cropTo;
                    scrollableViewBitmap = Bitmap.createBitmap(scrollableViewBitmap, 0, cropFrom, scrollableViewBitmap.getWidth(), cropTo);
                }

                imagesList.add(scrollableViewBitmap);

                mComponentsProvider.scroll(activity, scrollableView, oneScrollStep, step);

                if (step == maxScrollSteps - 1 && behaviorOffset > 0) {
                    mComponentsProvider.scrollBehavior(activity, scrollableView, behaviorOffset);
                    behaviorScrolled = true;
                }

                Thread.sleep(500);
            }
            int activityRealHeight = activity.getWindow().getDecorView().getHeight();

            if ((mComponentsProvider.getScreenHeight(activity) +
                    mComponentsProvider.getBottomNavigationBarHeight(activity) -
                    activity.getWindow().getDecorView().getHeight()) == 0) {
                activityRealHeight = mComponentsProvider.getScreenHeight(activity);
            }

            int remainingHeight = activityRealHeight + behaviorOffset - (location[1] + scrollableView.getHeight() - scrollableView.getPaddingBottom());
            mLogger.log(TAG + "Remaining height = " + remainingHeight);
            if (remainingHeight > 0) {
                try {
                    bitmap = mComponentsProvider.captureFrame(activity,
                            new Region(Region.FULL_WIDTH,
                                    remainingHeight,
                                    location[1] + scrollableView.getHeight() - scrollableView.getPaddingBottom() - behaviorOffset,
                                    0), mHideCaret);
                    imagesList.add(bitmap);
                } catch (IllegalArgumentException ex) {
                    mLogger.log(TAG + "IllegalArgumentException while taking bitmap with remaining height: " + ex.getMessage());
                }
            }

            Bitmap bitmap_1 = imagesList.get(0);
            for (int i = 1; i < imagesList.size(); i++) {
                Bitmap bitmap_2 = imagesList.get(i);
                if (result == null) {
                    result = combineImages(bitmap_1, bitmap_2);
                } else {
                    result = combineImages(result, bitmap_2);
                }
            }

            mComponentsProvider.scrollToTop(activity, scrollableView);

            if (behaviorScrolled) {
                mComponentsProvider.scrollBehavior(activity, scrollableView, -behaviorOffset);
            }
        } else {
            mLogger.log(TAG + "Scrollable view is not presented");
            result = mComponentsProvider.captureViewportScreenshot(activity, false, true,
                    mIncludeAllLayers, mHideCaret);
        }
        return result;
    }

    private Bitmap combineImages(Bitmap bitmap_1, Bitmap bitmap_2) throws InterruptedException {
        Bitmap combinedBitmap;

        int width = bitmap_1.getWidth();
        int height = bitmap_1.getHeight() + bitmap_2.getHeight();

        combinedBitmap = Bitmap.createBitmap(width, height, Bitmap.Config.ARGB_8888);

        Canvas comboImage = new Canvas(combinedBitmap);
        comboImage.drawBitmap(bitmap_1, 0f, 0f, null);
        comboImage.drawBitmap(bitmap_2, 0f, bitmap_1.getHeight(), null);

        return combinedBitmap;
    }
}
