package com.applitools.eyes.android.espresso.logger;

import android.os.Environment;

import com.applitools.eyes.android.common.exceptions.EyesException;
import com.applitools.eyes.android.common.logger.LogHandler;
import com.applitools.eyes.android.common.utils.ArgumentGuard;

import java.io.File;
import java.io.IOException;
import java.util.logging.FileHandler;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;

/**
 * Writes log messages to a file.
 */
public class FileLogHandler implements LogHandler {

    private Logger mLogger;
    private boolean mIsVerbose;

    /**
     * Creates a new FileHandler instance.
     * @param fileName The file in which to save the logs.
     * @param append Whether to append the logs if the current file exists,
     *               or to overwrite the existing file.
     * @param isVerbose Whether to handle or ignore verbose log messages.
     */
    public FileLogHandler(String fileName, boolean append, boolean isVerbose) {
        ArgumentGuard.notNullOrEmpty(fileName, "fileName");
        mLogger = Logger.getLogger(com.applitools.eyes.android.common.logger.Logger.LOGGER_NAME);
        mIsVerbose = isVerbose;
        FileHandler fileHandler;
        try {
            fileHandler = new FileHandler(getFileLogDir().getAbsolutePath() + "/" + fileName, append);
            mLogger.addHandler(fileHandler);
            SimpleFormatter formatter = new SimpleFormatter();
            fileHandler.setFormatter(formatter);

        } catch (IOException e) {
            throw new EyesException("FileLogHandler: ", e);
        }
    }

    /**
     * See {@link #FileLogHandler(String, boolean, boolean)}.
     * {@code fileName} defaults to {@code eyes.log}, append defaults to
     * {@code true}.
     *
     * @param isVerbose Whether to handle or ignore verbose log messages.
     */
    @SuppressWarnings("UnusedDeclaration")
    public FileLogHandler(boolean isVerbose) {
        this("eyes.log", true, isVerbose);
    }

    @Override
    public void onMessage(boolean verbose, String logString) {
        if (mLogger != null && (!verbose || mIsVerbose)) {
            mLogger.info(logString);
        }
    }

    @Override
    public void close() {

    }

    @Override
    public boolean isOpen() {
        return true; //TODO check this
    }

    @Override
    public void open() {

    }

    private File getFileLogDir() throws IOException {
        File dir = new File(Environment.getExternalStorageDirectory().getPath() + "/" + "applitools" + "/logs");
        File directory = initDir(dir);

        if (directory == null) {
            throw new IOException("Can not create directory for logs. If you use API 23 and higher you need permission for access to storage.");
        }
        return directory;
    }

    private static File initDir(File directory) throws IOException {
        File result = null;
        try {
            createDir(directory);
            if (directory.isDirectory() && directory.canWrite()) {
                result = directory;
            }
        } catch (IOException ignored) {
        }
        return result;
    }

    private static void createDir(File dir) throws IOException {
        File parent = dir.getParentFile();
        if (!parent.exists()) {
            createDir(parent);
        }
        if (!dir.exists() && !dir.mkdirs()) {
            throw new IOException("Can not create output dir: " + dir.getAbsolutePath());
        }
        CommandLineUtils.cmdPermissionRWX(dir);
    }
}
