package com.applovin.sdk;

import android.content.Intent;

import java.util.Map;

/**
 * Service used for tracking various analytical events.
 */
public interface AppLovinEventService
{
    /**
     * Set a super property to be recorded with all future events.
     * <p>
     * If the super property is set to {@code null}, will remove the super property from being recorded with all future events.
     *
     * @param superProperty The super property value for the given super property key.
     *                      Valid types include {@link String}, {@link Integer}, {@link Long}, {@link Double}, {@link Float}, {@link java.util.Date}, {@link android.net.Uri}, {@link java.util.List}, {@link Map}.
     *                      Setting it to {@code null} will remove that super property from being recorded with all future events.
     * @param key           The super property key for the given super property.
     */
    void setSuperProperty(final Object superProperty, final String key);

    /**
     * Returns a map representing the currently set super properties that are passed up on events.
     */
    Map<String, Object> getSuperProperties();

    /**
     * Track an event without additional data.
     * <p>
     * Where applicable, it is suggested to use one of the predefined strings provided in {@link AppLovinEventTypes} for the event and parameter key.
     *
     * @param eventName A string representing the event to track.
     */
    void trackEvent(final String eventName);

    /**
     * Track an event with additional data.
     * <p>
     * Where applicable, it is suggested to use one of the predefined strings provided in {@link AppLovinEventTypes} for the event and parameter key.
     *
     * @param eventName  A string representing the event to track.
     * @param parameters A map containing key-value pairs further describing this event.
     */
    void trackEvent(final String eventName, final Map<String, String> parameters);

    /**
     * Track an in app purchase.
     *
     * @param purchaseIntent Intent returned to you by Google Play.
     * @param parameters     A map containing key-value pairs further describing this event. You should provide, at a minimum, {@link AppLovinEventParameters#PRODUCT_IDENTIFIER}, {@link AppLovinEventParameters#REVENUE_AMOUNT} and {@link AppLovinEventParameters#REVENUE_CURRENCY}.
     */
    void trackInAppPurchase(final Intent purchaseIntent, final Map<String, String> parameters);

    /**
     * Track a checkout / standard purchase.
     *
     * @param transactionId An optional unique identifier for this transaction, as generated by you.
     * @param parameters    A map containing key-value pairs further describing this event. You should provide, at a minimum, {@link AppLovinEventParameters#PRODUCT_IDENTIFIER}, {@link AppLovinEventParameters#REVENUE_AMOUNT} and {@link AppLovinEventParameters#REVENUE_CURRENCY}.
     */
    void trackCheckout(final String transactionId, final Map<String, String> parameters);
}
