package com.applovin.mediation.adapters;

import android.app.Activity;
import android.content.SharedPreferences;
import android.os.Bundle;
import android.preference.PreferenceManager;
import android.text.TextUtils;

import com.applovin.mediation.MaxAdFormat;
import com.applovin.mediation.MaxReward;
import com.applovin.mediation.adapter.MaxAdViewAdapter;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.MaxInterstitialAdapter;
import com.applovin.mediation.adapter.MaxRewardedAdapter;
import com.applovin.mediation.adapter.listeners.MaxAdViewAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxRewardedAdapterListener;
import com.applovin.mediation.adapter.parameters.MaxAdapterInitializationParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterResponseParameters;
import com.applovin.sdk.AppLovinSdk;
import com.applovin.sdk.AppLovinSdkUtils;
import com.smaato.sdk.banner.ad.AutoReloadInterval;
import com.smaato.sdk.banner.ad.BannerAdSize;
import com.smaato.sdk.banner.widget.BannerError;
import com.smaato.sdk.banner.widget.BannerView;
import com.smaato.sdk.core.Config;
import com.smaato.sdk.core.SmaatoSdk;
import com.smaato.sdk.core.log.LogLevel;
import com.smaato.sdk.core.repository.AdRequestParams;
import com.smaato.sdk.iahb.InAppBid;
import com.smaato.sdk.iahb.InAppBiddingException;
import com.smaato.sdk.iahb.SmaatoSdkInAppBidding;
import com.smaato.sdk.interstitial.Interstitial;
import com.smaato.sdk.interstitial.InterstitialAd;
import com.smaato.sdk.interstitial.InterstitialError;
import com.smaato.sdk.interstitial.InterstitialRequestError;
import com.smaato.sdk.rewarded.RewardedError;
import com.smaato.sdk.rewarded.RewardedInterstitial;
import com.smaato.sdk.rewarded.RewardedInterstitialAd;
import com.smaato.sdk.rewarded.RewardedRequestError;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import androidx.annotation.NonNull;

/**
 * Created by Christopher Cong on March 11 2019
 */
public class SmaatoMediationAdapter
        extends MediationAdapterBase
        implements MaxAdViewAdapter, MaxInterstitialAdapter, MaxRewardedAdapter
{
    private static final SmaatoMediationAdapterRouter ROUTER;
    private static final AtomicBoolean                INITIALIZED = new AtomicBoolean();

    // Used by the mediation adapter router
    private String placementId;

    // Ad Objects
    private BannerView             adView;
    private InterstitialAd         interstitialAd;
    private RewardedInterstitialAd rewardedAd;

    static
    {
        if ( AppLovinSdk.VERSION_CODE >= 90802 )
        {
            ROUTER = (SmaatoMediationAdapterRouter) MediationAdapterRouter.getSharedInstance( SmaatoMediationAdapterRouter.class );
        }
        else
        {
            ROUTER = new SmaatoMediationAdapterRouter();
        }
    }

    public SmaatoMediationAdapter(final AppLovinSdk sdk) { super( sdk ); }

    //region MaxAdapter

    @Override
    public String getSdkVersion()
    {
        return SmaatoSdk.getVersion();
    }

    @Override
    public String getAdapterVersion()
    {
        return com.applovin.mediation.adapters.smaato.BuildConfig.VERSION_NAME;
    }

    @Override
    public void onDestroy()
    {
        if ( adView != null )
        {
            adView.setEventListener( null );
            adView.destroy();
            adView = null;
        }

        interstitialAd = null;
        rewardedAd = null;

        ROUTER.removeAdapter( this, placementId );
    }

    @Override
    public void initialize(final MaxAdapterInitializationParameters parameters, final Activity activity, final OnCompletionListener onCompletionListener)
    {
        if ( INITIALIZED.compareAndSet( false, true ) )
        {
            final String pubId = parameters.getServerParameters().getString( "pub_id", "" );
            log( "Initializing Smaato SDK with publisher id: " + pubId + "..." );

            removeUnsupportedUserConsent( activity );

            final Config config = Config.builder()
                    .setLogLevel( parameters.isTesting() ? LogLevel.DEBUG : LogLevel.ERROR )
                    .setHttpsOnly( parameters.getServerParameters().getBoolean( "https_only" ) )
                    .build();

            SmaatoSdk.init( activity.getApplication(), config, pubId );

            // Smaato: Call all other APIs after `SmaatoSdk.init(...)`
            updateAgeRestrictedUser( parameters );
        }

        onCompletionListener.onCompletion( InitializationStatus.DOES_NOT_APPLY, null );
    }

    //endregion

    //region MaxAdViewAdapter

    @Override
    public void loadAdViewAd(final MaxAdapterResponseParameters parameters, final MaxAdFormat adFormat, final Activity activity, final MaxAdViewAdapterListener listener)
    {
        final String bidResponse = parameters.getBidResponse();
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading " + ( AppLovinSdkUtils.isValidString( bidResponse ) ? "bidding " : "" ) + adFormat.getLabel() + " ad for placement: " + placementId + "..." );

        updateAgeRestrictedUser( parameters );

        adView = new BannerView( activity );
        adView.setAutoReloadInterval( AutoReloadInterval.DISABLED );

        adView.setEventListener( new BannerView.EventListener()
        {
            @Override
            public void onAdLoaded(@NonNull final BannerView bannerView)
            {
                log( "AdView loaded" );

                // Passing extra info such as creative id supported in 9.15.0+
                if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( bannerView.getCreativeId() ) )
                {
                    Bundle extraInfo = new Bundle( 1 );
                    extraInfo.putString( "creative_id", bannerView.getCreativeId() );

                    listener.onAdViewAdLoaded( adView, extraInfo );
                }
                else
                {
                    listener.onAdViewAdLoaded( adView );
                }
            }

            @Override
            public void onAdFailedToLoad(@NonNull final BannerView bannerView, @NonNull final BannerError bannerError)
            {
                log( "AdView load failed to load with error: " + bannerError );

                final MaxAdapterError error = toMaxError( bannerError );
                listener.onAdViewAdLoadFailed( error );
            }

            @Override
            public void onAdImpression(@NonNull final BannerView bannerView)
            {
                log( "AdView displayed" );
                listener.onAdViewAdDisplayed();
            }

            @Override
            public void onAdClicked(@NonNull final BannerView bannerView)
            {
                log( "AdView clicked" );
                listener.onAdViewAdClicked();
            }

            @Override
            public void onAdTTLExpired(@NonNull final BannerView bannerView)
            {
                log( "AdView ad expired" );
            }
        } );

        adView.loadAd( placementId, toAdSize( adFormat ), createBiddingAdRequestParams( bidResponse ) );
    }

    //endregion

    //region MaxInterstitialAdapter

    @Override
    public void loadInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        final String bidResponse = parameters.getBidResponse();
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading " + ( AppLovinSdkUtils.isValidString( bidResponse ) ? "bidding " : "" ) + "interstitial ad for placement: " + placementId + "..." );

        updateAgeRestrictedUser( parameters );

        ROUTER.addInterstitialAdapter( this, listener, placementId );

        final InterstitialAd loadedAd = ROUTER.getInterstitialAd( placementId );
        if ( loadedAd != null && loadedAd.isAvailableForPresentation() )
        {
            log( "Interstitial already loaded for placement: " + placementId + "..." );
            listener.onInterstitialAdLoaded();
        }
        else
        {
            Interstitial.loadAd( placementId, ROUTER, createBiddingAdRequestParams( bidResponse ) );
        }
    }

    @Override
    public void showInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Showing interstitial ad for placement: " + placementId + "..." );

        ROUTER.addShowingAdapter( this );

        interstitialAd = ROUTER.getInterstitialAd( placementId );
        if ( interstitialAd != null && interstitialAd.isAvailableForPresentation() )
        {
            interstitialAd.showAd( activity );
        }
        else
        {
            log( "Interstitial not ready." );
            ROUTER.onAdDisplayFailed( placementId, MaxAdapterError.AD_NOT_READY );
        }
    }

    //endregion

    //region MaxRewardedAdapter

    @Override
    public void loadRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        final String bidResponse = parameters.getBidResponse();
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading " + ( AppLovinSdkUtils.isValidString( bidResponse ) ? "bidding " : "" ) + "rewarded ad for placement: " + placementId + "..." );

        updateAgeRestrictedUser( parameters );

        ROUTER.addRewardedAdapter( this, listener, placementId );

        final RewardedInterstitialAd loadedAd = ROUTER.getRewardedAd( placementId );
        if ( loadedAd != null && loadedAd.isAvailableForPresentation() )
        {
            log( "Rewarded ad already loaded for placement: " + placementId + "..." );
            listener.onRewardedAdLoaded();
        }
        else
        {
            RewardedInterstitial.loadAd( placementId, ROUTER, createBiddingAdRequestParams( bidResponse ) );
        }
    }

    @Override
    public void showRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Showing rewarded ad for placement: " + placementId + "..." );

        ROUTER.addShowingAdapter( this );

        rewardedAd = ROUTER.getRewardedAd( placementId );
        if ( rewardedAd != null && rewardedAd.isAvailableForPresentation() )
        {
            // Configure userReward from server.
            configureReward( parameters );

            rewardedAd.showAd();
        }
        else
        {
            log( "Rewarded ad not ready." );
            ROUTER.onAdDisplayFailed( placementId, MaxAdapterError.AD_NOT_READY );
        }
    }

    //endregion

    //region GDPR

    private void removeUnsupportedUserConsent(final Activity activity)
    {
        // For more GDPR info: https://github.com/InteractiveAdvertisingBureau/GDPR-Transparency-and-Consent-Framework/blob/master/Mobile%20In-App%20Consent%20APIs%20v1.0%20Final.md#cmp-internal-structure-defined-api-

        //
        // Previous version of adapters could have set the values for IAB Consent, which are no longer supported. Remove them:
        // https://app.asana.com/0/615971567602282/1189834741836833
        //

        final SharedPreferences preferences = PreferenceManager.getDefaultSharedPreferences( activity );
        preferences.edit()
                .remove( "IABConsent_SubjectToGDPR" )
                .remove( "IABConsent_ConsentString" )
                .apply();
    }

    //endregion

    //region Helper Methods

    private void updateAgeRestrictedUser(final MaxAdapterParameters parameters)
    {
        Boolean isAgeRestrictedUser = getPrivacySetting( "isAgeRestrictedUser", parameters );
        if ( isAgeRestrictedUser != null )
        {
            SmaatoSdk.setCoppa( isAgeRestrictedUser );
        }
    }

    private Boolean getPrivacySetting(final String privacySetting, final MaxAdapterParameters parameters)
    {
        try
        {
            // Use reflection because compiled adapters have trouble fetching `boolean` from old SDKs and `Boolean` from new SDKs (above 9.14.0)
            Class<?> parametersClass = parameters.getClass();
            Method privacyMethod = parametersClass.getMethod( privacySetting );
            return (Boolean) privacyMethod.invoke( parameters );
        }
        catch ( Exception exception )
        {
            log( "Error getting privacy setting " + privacySetting + " with exception: ", exception );
            return ( AppLovinSdk.VERSION_CODE >= 9140000 ) ? null : false;
        }
    }

    private BannerAdSize toAdSize(final MaxAdFormat adFormat)
    {
        if ( adFormat == MaxAdFormat.BANNER )
        {
            return BannerAdSize.XX_LARGE_320x50;
        }
        else if ( adFormat == MaxAdFormat.MREC )
        {
            return BannerAdSize.MEDIUM_RECTANGLE_300x250;
        }
        else if ( adFormat == MaxAdFormat.LEADER )
        {
            return BannerAdSize.LEADERBOARD_728x90;
        }
        else
        {
            throw new IllegalArgumentException( "Unsupported ad format: " + adFormat );
        }
    }

    private MaxAdapterError toMaxError(final BannerError bannerError)
    {
        final MaxAdapterError adapterError;
        if ( bannerError == BannerError.NO_AD_AVAILABLE )
        {
            adapterError = MaxAdapterError.NO_FILL;
        }
        else if ( bannerError == BannerError.INVALID_REQUEST )
        {
            adapterError = MaxAdapterError.INVALID_CONFIGURATION;
        }
        else if ( bannerError == BannerError.NETWORK_ERROR )
        {
            adapterError = MaxAdapterError.NO_CONNECTION;
        }
        else if ( bannerError == BannerError.INTERNAL_ERROR )
        {
            adapterError = MaxAdapterError.INTERNAL_ERROR;
        }
        else if ( bannerError == BannerError.CREATIVE_RESOURCE_EXPIRED )
        {
            adapterError = MaxAdapterError.AD_EXPIRED;
        }
        else if ( bannerError == BannerError.AD_UNLOADED )
        {
            adapterError = MaxAdapterError.INVALID_LOAD_STATE;
        }
        else
        {
            adapterError = MaxAdapterError.UNSPECIFIED;
        }
        return new MaxAdapterError( adapterError.getErrorCode(), adapterError.getErrorMessage(), bannerError.ordinal(), bannerError.name() );
    }

    private AdRequestParams createBiddingAdRequestParams(final String bidResponse)
    {
        if ( TextUtils.isEmpty( bidResponse ) ) return null;

        final String token;
        try
        {
            final InAppBid inAppBid = InAppBid.create( bidResponse );
            token = SmaatoSdkInAppBidding.saveBid( inAppBid );
        }
        catch ( final InAppBiddingException exception )
        {
            log( "Error occurred in saving pre-bid: " + exception );
            return null;
        }

        return AdRequestParams.builder().setUBUniqueId( token ).build();
    }

    //endregion

    //region SmaatoMediationAdapterRouter

    /**
     * Router for interstitial/rewarded ad events.
     * Ads are removed on ad displayed/expired, as Smaato will allow a new ad load for the same adSpaceId.
     */
    private static class SmaatoMediationAdapterRouter
            extends MediationAdapterRouter
            implements com.smaato.sdk.interstitial.EventListener, com.smaato.sdk.rewarded.EventListener
    {
        // Interstitial
        private final Map<String, InterstitialAd> interstitialAds     = new HashMap<>();
        private final Object                      interstitialAdsLock = new Object();

        // Rewarded
        private final Map<String, RewardedInterstitialAd> rewardedAds     = new HashMap<>();
        private final Object                              rewardedAdsLock = new Object();

        private boolean hasGrantedReward;

        @Override
        void initialize(final MaxAdapterInitializationParameters parameters, final Activity activity, final OnCompletionListener onCompletionListener) {}

        public InterstitialAd getInterstitialAd(final String placementId)
        {
            synchronized ( interstitialAdsLock )
            {
                return interstitialAds.get( placementId );
            }
        }

        public RewardedInterstitialAd getRewardedAd(final String placementId)
        {
            synchronized ( rewardedAdsLock )
            {
                return rewardedAds.get( placementId );
            }
        }

        //region Interstitial listener

        @Override
        public void onAdLoaded(final InterstitialAd interstitialAd)
        {
            final String placementId = interstitialAd.getAdSpaceId();

            synchronized ( interstitialAdsLock )
            {
                interstitialAds.put( placementId, interstitialAd );
            }

            log( "Interstitial loaded for placement: " + placementId + "..." );
            onAdLoaded( placementId, interstitialAd.getCreativeId() );
        }

        @Override
        public void onAdFailedToLoad(final InterstitialRequestError interstitialRequestError)
        {
            final String placementId = interstitialRequestError.getAdSpaceId();

            log( "Interstitial failed to load for placement: " + placementId + "...with error: " + interstitialRequestError.getInterstitialError() );

            onAdLoadFailed( placementId, toMaxError( interstitialRequestError.getInterstitialError() ) );
        }

        @Override
        public void onAdError(@NonNull final InterstitialAd interstitialAd, @NonNull final InterstitialError interstitialError)
        {
            log( "Interstitial failed to display with error: " + interstitialError );

            if ( interstitialAd != null )
            {
                final String placementId = interstitialAd.getAdSpaceId();

                synchronized ( interstitialAdsLock )
                {
                    interstitialAds.remove( placementId );
                }

                onAdDisplayFailed( placementId, toMaxError( interstitialError ) );
            }
        }

        @Override
        public void onAdImpression(final InterstitialAd interstitialAd)
        {
            final String placementId = interstitialAd.getAdSpaceId();

            // Allow the next rewarded ad to load
            synchronized ( interstitialAdsLock )
            {
                interstitialAds.remove( placementId );
            }

            log( "Interstitial displayed" );
            onAdDisplayed( placementId );
        }

        @Override
        public void onAdOpened(@NonNull final InterstitialAd interstitialAd) {}

        @Override
        public void onAdClicked(final InterstitialAd interstitialAd)
        {
            log( "Interstitial clicked" );
            onAdClicked( interstitialAd.getAdSpaceId() );
        }

        @Override
        public void onAdClosed(final InterstitialAd interstitialAd)
        {
            log( "Interstitial hidden" );
            onAdHidden( interstitialAd.getAdSpaceId() );
        }

        @Override
        public void onAdTTLExpired(final InterstitialAd interstitialAd)
        {
            log( "Interstitial expired" );

            synchronized ( interstitialAdsLock )
            {
                interstitialAds.remove( interstitialAd.getAdSpaceId() );
            }
        }

        private MaxAdapterError toMaxError(final InterstitialError interstitialError)
        {
            final MaxAdapterError adapterError;
            if ( interstitialError == InterstitialError.NO_AD_AVAILABLE )
            {
                adapterError = MaxAdapterError.NO_FILL;
            }
            else if ( interstitialError == InterstitialError.INVALID_REQUEST )
            {
                adapterError = MaxAdapterError.INVALID_CONFIGURATION;
            }
            else if ( interstitialError == InterstitialError.NETWORK_ERROR )
            {
                adapterError = MaxAdapterError.NO_CONNECTION;
            }
            else if ( interstitialError == InterstitialError.INTERNAL_ERROR )
            {
                adapterError = MaxAdapterError.INTERNAL_ERROR;
            }
            else if ( interstitialError == InterstitialError.CREATIVE_RESOURCE_EXPIRED )
            {
                adapterError = MaxAdapterError.AD_EXPIRED;
            }
            else if ( interstitialError == InterstitialError.AD_UNLOADED )
            {
                adapterError = MaxAdapterError.INVALID_LOAD_STATE;
            }
            else
            {
                adapterError = MaxAdapterError.UNSPECIFIED;
            }

            return new MaxAdapterError( adapterError.getErrorCode(), adapterError.getErrorMessage(), interstitialError.ordinal(), interstitialError.name() );
        }

        //endregion

        //region Rewarded listener

        @Override
        public void onAdLoaded(final RewardedInterstitialAd rewardedInterstitialAd)
        {
            final String placementId = rewardedInterstitialAd.getAdSpaceId();

            synchronized ( rewardedAdsLock )
            {
                rewardedAds.put( placementId, rewardedInterstitialAd );
            }

            log( "Rewarded ad loaded for placement: " + placementId + "..." );
            onAdLoaded( placementId, rewardedInterstitialAd.getCreativeId() );
        }

        @Override
        public void onAdFailedToLoad(final RewardedRequestError rewardedRequestError)
        {
            final String placementId = rewardedRequestError.getAdSpaceId();

            log( "Rewarded ad failed to load for placement: " + placementId + "...with error: " + rewardedRequestError.getRewardedError() );
            onAdLoadFailed( placementId, toMaxError( rewardedRequestError.getRewardedError() ) );
        }

        @Override
        public void onAdError(@NonNull final RewardedInterstitialAd rewardedInterstitialAd, @NonNull final RewardedError rewardedError)
        {
            log( "Rewarded ad failed to display with error: " + rewardedError );

            if ( rewardedInterstitialAd != null )
            {
                final String placementId = rewardedInterstitialAd.getAdSpaceId();

                synchronized ( rewardedAdsLock )
                {
                    rewardedAds.remove( placementId );
                }

                onAdDisplayFailed( placementId, toMaxError( rewardedError ) );
            }
        }

        @Override
        public void onAdStarted(final RewardedInterstitialAd rewardedInterstitialAd)
        {
            final String placementId = rewardedInterstitialAd.getAdSpaceId();

            // Allow the next rewarded ad to load
            synchronized ( rewardedAdsLock )
            {
                rewardedAds.remove( placementId );
            }

            log( "Rewarded ad displayed" );
            onAdDisplayed( placementId );
            onRewardedAdVideoStarted( placementId );
        }

        @Override
        public void onAdClicked(final RewardedInterstitialAd rewardedInterstitialAd)
        {
            log( "Rewarded ad clicked" );
            onAdClicked( rewardedInterstitialAd.getAdSpaceId() );
        }

        @Override
        public void onAdReward(final RewardedInterstitialAd rewardedInterstitialAd)
        {
            log( "Rewarded ad video completed" );
            onRewardedAdVideoCompleted( rewardedInterstitialAd.getAdSpaceId() );

            hasGrantedReward = true;
        }

        @Override
        public void onAdClosed(final RewardedInterstitialAd rewardedInterstitialAd)
        {
            final String placementId = rewardedInterstitialAd.getAdSpaceId();

            if ( hasGrantedReward || shouldAlwaysRewardUser( placementId ) )
            {
                final MaxReward reward = getReward( placementId );
                log( "Rewarded user with reward: " + reward );
                onUserRewarded( placementId, reward );
            }

            log( "Rewarded ad hidden" );
            onAdHidden( placementId );
        }

        @Override
        public void onAdTTLExpired(final RewardedInterstitialAd rewardedInterstitialAd)
        {
            log( "Rewarded ad expired" );

            synchronized ( rewardedAdsLock )
            {
                rewardedAds.remove( rewardedInterstitialAd.getAdSpaceId() );
            }
        }

        private void onAdLoaded(final String creativeId, final String placementId)
        {
            // Passing extra info such as creative id supported in 9.15.0+
            if ( AppLovinSdk.VERSION_CODE >= 9150000 && !TextUtils.isEmpty( creativeId ) )
            {
                Bundle extraInfo = new Bundle( 1 );
                extraInfo.putString( "creative_id", creativeId );

                onAdLoaded( placementId, extraInfo );
            }
            else
            {
                onAdLoaded( placementId );
            }
        }

        private MaxAdapterError toMaxError(final RewardedError rewardedError)
        {
            final int adapterErrorCode;
            if ( rewardedError == RewardedError.NO_AD_AVAILABLE )
            {
                adapterErrorCode = MaxAdapterError.ERROR_CODE_NO_FILL;
            }
            else if ( rewardedError == RewardedError.INVALID_REQUEST )
            {
                adapterErrorCode = MaxAdapterError.ERROR_CODE_INVALID_CONFIGURATION;
            }
            else if ( rewardedError == RewardedError.NETWORK_ERROR )
            {
                adapterErrorCode = MaxAdapterError.ERROR_CODE_NO_CONNECTION;
            }
            else if ( rewardedError == RewardedError.INTERNAL_ERROR )
            {
                adapterErrorCode = MaxAdapterError.ERROR_CODE_INTERNAL_ERROR;
            }
            else if ( rewardedError == RewardedError.CREATIVE_RESOURCE_EXPIRED )
            {
                adapterErrorCode = MaxAdapterError.ERROR_CODE_AD_EXPIRED;
            }
            else
            {
                adapterErrorCode = MaxAdapterError.ERROR_CODE_UNSPECIFIED;
            }

            return new MaxAdapterError( adapterErrorCode, rewardedError.name() );
        }

        //endregion
    }

    //endregion
}
