package com.applovin.mediation.adapters;

import android.app.Activity;
import android.os.Bundle;
import android.view.View;

import com.applovin.mediation.MaxAdFormat;
import com.applovin.mediation.MaxReward;
import com.applovin.mediation.adapter.MaxAdViewAdapter;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.MaxInterstitialAdapter;
import com.applovin.mediation.adapter.MaxRewardedAdapter;
import com.applovin.mediation.adapter.MaxSignalProvider;
import com.applovin.mediation.adapter.listeners.MaxAdViewAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxRewardedAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxSignalCollectionListener;
import com.applovin.mediation.adapter.parameters.MaxAdapterInitializationParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterResponseParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterSignalCollectionParameters;
import com.applovin.mediation.adapters.vungle.BuildConfig;
import com.applovin.sdk.AppLovinSdk;
import com.applovin.sdk.AppLovinSdkConfiguration;
import com.vungle.warren.AdConfig;
import com.vungle.warren.Banners;
import com.vungle.warren.InitCallback;
import com.vungle.warren.LoadAdCallback;
import com.vungle.warren.PlayAdCallback;
import com.vungle.warren.Plugin;
import com.vungle.warren.Vungle;
import com.vungle.warren.VungleApiClient;
import com.vungle.warren.VungleBanner;
import com.vungle.warren.VungleNativeAd;
import com.vungle.warren.VungleSettings;
import com.vungle.warren.error.VungleException;

import java.lang.reflect.Method;
import java.util.concurrent.atomic.AtomicBoolean;

public class VungleMediationAdapter
        extends MediationAdapterBase
        implements MaxSignalProvider, MaxInterstitialAdapter, MaxRewardedAdapter, MaxAdViewAdapter
{
    // Bid token constant
    private static final int BID_TOKEN_SIZE_LIMIT = 10; // Vungle recommends setting this to 10

    private static final AtomicBoolean INITIALIZED = new AtomicBoolean();

    private static InitializationStatus sStatus;

    private VungleBanner   bannerAd;
    private VungleNativeAd mrecAd;
    private View           mrecAdView;

    // Explicit default constructor declaration
    public VungleMediationAdapter(final AppLovinSdk sdk) { super( sdk ); }

    @Override
    public void initialize(final MaxAdapterInitializationParameters parameters, final Activity activity, final OnCompletionListener onCompletionListener)
    {
        // Check existence of SDK classes
        checkExistence( Vungle.class );

        updateUserPrivacySettings( parameters );

        if ( INITIALIZED.compareAndSet( false, true ) )
        {
            final String appId = parameters.getServerParameters().getString( "app_id", null );
            log( "Initializing Vungle SDK with app id: " + appId + "..." );

            sStatus = InitializationStatus.INITIALIZING;

            Plugin.addWrapperInfo( VungleApiClient.WrapperFramework.max, getAdapterVersion() );

            VungleSettings settings = new VungleSettings.Builder().disableBannerRefresh().build();
            InitCallback initCallback = new InitCallback()
            {
                @Override
                public void onSuccess()
                {
                    log( "Vungle SDK initialized" );

                    sStatus = InitializationStatus.INITIALIZED_SUCCESS;
                    onCompletionListener.onCompletion( sStatus, null );
                }

                @Override
                public void onError(final VungleException exception)
                {
                    log( "Vungle SDK failed to initialize with error", exception );

                    sStatus = InitializationStatus.INITIALIZED_FAILURE;
                    onCompletionListener.onCompletion( sStatus, exception.getLocalizedMessage() );
                }

                @Override
                public void onAutoCacheAdAvailable(final String id)
                {
                    log( "Auto-cached ad " + id );
                }
            };

            // Note: Vungle requires the Application Context
            Vungle.init( appId, activity.getApplicationContext(), initCallback, settings );
        }
        else
        {
            log( "Vungle SDK already initialized" );

            onCompletionListener.onCompletion( sStatus, null );
        }
    }

    @Override
    public String getSdkVersion()
    {
        return getVersionString( com.vungle.warren.BuildConfig.class, "VERSION_NAME" );
    }

    @Override
    public String getAdapterVersion()
    {
        return BuildConfig.VERSION_NAME;
    }

    @Override
    public void onDestroy()
    {
        if ( bannerAd != null )
        {
            bannerAd.destroyAd();
            bannerAd = null;
        }

        if ( mrecAdView != null && mrecAd != null )
        {
            // finishDisplayingAd only dismisses MREC and In-Feed Ads, not fullscreen ads.
            mrecAd.finishDisplayingAd();
            mrecAd = null;
            mrecAdView = null;
        }
    }

    //region Signal Collection

    @Override
    public void collectSignal(final MaxAdapterSignalCollectionParameters parameters, final Activity activity, final MaxSignalCollectionListener callback)
    {
        log( "Collecting signal..." );

        if ( !Vungle.isInitialized() )
        {
            callback.onSignalCollectionFailed( "Vungle SDK not successfully initialized" );
            return;
        }

        String signal = Vungle.getAvailableBidTokens( activity.getApplicationContext(), BID_TOKEN_SIZE_LIMIT );
        callback.onSignalCollected( signal );
    }

    //endregion

    //region MaxInterstitialAdapter

    @Override
    public void loadInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading interstitial ad for placement: " + placementId + "..." );

        if ( !Vungle.isInitialized() )
        {
            log( "Vungle SDK not successfully initialized: failing interstitial ad load..." );
            listener.onInterstitialAdLoadFailed( MaxAdapterError.NOT_INITIALIZED );

            return;
        }

        if ( Vungle.canPlayAd( placementId ) )
        {
            log( "Interstitial ad loaded" );
            listener.onInterstitialAdLoaded();
        }
        else if ( isValidPlacement( parameters ) )
        {
            updateUserPrivacySettings( parameters );
            Vungle.loadAd( placementId, new LoadAdCallback()
            {
                @Override
                public void onAdLoad(final String id)
                {
                    log( "Interstitial ad loaded" );
                    listener.onInterstitialAdLoaded();
                }

                @Override
                public void onError(final String id, final VungleException exception)
                {
                    final MaxAdapterError error = toMaxError( exception );
                    log( "Interstitial ad for placement " + id + " failed to load with error: " + error );

                    listener.onInterstitialAdLoadFailed( error );
                }
            } );
        }
        else
        {
            log( "Interstitial ad failed to load due to an invalid placement id: " + placementId );
            listener.onInterstitialAdLoadFailed( MaxAdapterError.INVALID_CONFIGURATION );
        }
    }

    @Override
    public void showInterstitialAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxInterstitialAdapterListener listener)
    {
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Showing interstitial ad for placement: " + placementId + "..." );

        if ( Vungle.canPlayAd( placementId ) )
        {
            Vungle.playAd( placementId, retrieveAdConfig( parameters.getServerParameters() ), new InterstitialAdListener( listener ) );
        }
        else
        {
            log( "Interstitial ad not ready" );
            listener.onInterstitialAdDisplayFailed( MaxAdapterError.AD_NOT_READY );
        }
    }

    //endregion

    //region MaxRewardedAdapter

    @Override
    public void loadRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading rewarded ad for placement: " + placementId + "..." );

        if ( !Vungle.isInitialized() )
        {
            log( "Vungle SDK not successfully initialized: failing rewarded ad load..." );
            listener.onRewardedAdLoadFailed( MaxAdapterError.NOT_INITIALIZED );

            return;
        }

        if ( Vungle.canPlayAd( placementId ) )
        {
            log( "Rewarded ad loaded" );
            listener.onRewardedAdLoaded();
        }
        else if ( isValidPlacement( parameters ) )
        {
            updateUserPrivacySettings( parameters );
            Vungle.loadAd( placementId, new LoadAdCallback()
            {
                @Override
                public void onAdLoad(final String id)
                {
                    log( "Rewarded ad loaded" );
                    listener.onRewardedAdLoaded();
                }

                @Override
                public void onError(final String id, final VungleException exception)
                {
                    final MaxAdapterError error = toMaxError( exception );
                    log( "Rewarded ad for placement " + id + " failed to load with error: " + error );

                    listener.onRewardedAdLoadFailed( error );
                }
            } );
        }
        else
        {
            log( "Rewarded ad failed to load due to an invalid placement id: " + placementId );
            listener.onRewardedAdLoadFailed( MaxAdapterError.INVALID_CONFIGURATION );
        }
    }

    @Override
    public void showRewardedAd(final MaxAdapterResponseParameters parameters, final Activity activity, final MaxRewardedAdapterListener listener)
    {
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Showing rewarded ad for placement: " + placementId + "..." );

        if ( Vungle.canPlayAd( placementId ) )
        {
            // Configure reward from server.
            configureReward( parameters );

            Vungle.playAd( placementId, retrieveAdConfig( parameters.getServerParameters() ), new RewardedAdListener( listener ) );
        }
        else
        {
            log( "Rewarded ad not ready" );
            listener.onRewardedAdDisplayFailed( MaxAdapterError.AD_NOT_READY );
        }
    }

    //endregion

    //region MaxAdViewAdapter

    @Override
    public void loadAdViewAd(final MaxAdapterResponseParameters parameters, final MaxAdFormat adFormat, final Activity activity, final MaxAdViewAdapterListener listener)
    {
        final String adFormatLabel = adFormat.getLabel();
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Loading " + adFormatLabel + " adView ad for placement: " + placementId + "..." );

        if ( !Vungle.isInitialized() )
        {
            log( "Vungle SDK not successfully initialized: failing rewarded ad load..." );
            listener.onAdViewAdLoadFailed( MaxAdapterError.NOT_INITIALIZED );

            return;
        }

        final PlayAdCallback playAdCallback = new AdViewAdListener( adFormatLabel, listener );
        if ( MaxAdFormat.BANNER == adFormat || MaxAdFormat.LEADER == adFormat )
        {
            loadBannerAd( parameters, adFormat, listener, playAdCallback );
        }
        else // MaxAdFormat.MREC
        {
            loadMRecAd( parameters, listener, playAdCallback );
        }
    }

    private void loadBannerAd(final MaxAdapterResponseParameters parameters, final MaxAdFormat adFormat, final MaxAdViewAdapterListener listener, final PlayAdCallback playAdCallback)
    {
        final AdConfig.AdSize bannerAdSize = vungleBannerAdSize( adFormat );
        final String placementId = parameters.getThirdPartyAdPlacementId();
        if ( Banners.canPlayAd( placementId, bannerAdSize ) )
        {
            showBannerAd( placementId, adFormat, bannerAdSize, listener, playAdCallback );
        }
        else if ( isValidPlacement( parameters ) )
        {
            updateUserPrivacySettings( parameters );

            // loadBanner() will fail with "VungleException: Operation was canceled" if a banner's placement ID is already in use
            Banners.loadBanner( placementId, bannerAdSize, new LoadAdCallback()
            {
                @Override
                public void onAdLoad(final String id)
                {
                    showBannerAd( placementId, adFormat, bannerAdSize, listener, playAdCallback );
                }

                @Override
                public void onError(final String id, final VungleException exception)
                {
                    final MaxAdapterError error = toMaxError( exception );
                    log( adFormat.getLabel() + " ad for placement " + id + " failed to load with error: " + error );

                    listener.onAdViewAdLoadFailed( error );
                }
            } );
        }
        else
        {
            log( adFormat.getLabel() + " ad failed to load due to an invalid placement id: " + placementId );
            listener.onAdViewAdLoadFailed( MaxAdapterError.INVALID_CONFIGURATION );
        }
    }

    private void showBannerAd(final String placementId, final MaxAdFormat adFormat, final AdConfig.AdSize bannerAdSize, final MaxAdViewAdapterListener listener, final PlayAdCallback playAdCallback)
    {
        final String adFormatLabel = adFormat.getLabel();
        log( "Showing " + adFormatLabel + " ad for placement: " + placementId + "..." );

        // Don't need to set AdConfig for banners because Vungle's getBanner() method will create a new one anyways.
        bannerAd = Banners.getBanner( placementId, bannerAdSize, playAdCallback );
        if ( bannerAd != null )
        {
            log( adFormatLabel + " ad loaded" );
            listener.onAdViewAdLoaded( bannerAd );
        }
        else
        {
            final MaxAdapterError error = MaxAdapterError.INVALID_LOAD_STATE;
            log( adFormatLabel + " ad failed to load: " + error );

            listener.onAdViewAdLoadFailed( error );
        }
    }

    private void loadMRecAd(final MaxAdapterResponseParameters parameters, final MaxAdViewAdapterListener listener, final PlayAdCallback playAdCallback)
    {
        final String placementId = parameters.getThirdPartyAdPlacementId();

        if ( Vungle.canPlayAd( placementId ) )
        {
            showMRecAd( parameters, listener, playAdCallback );
        }
        else if ( isValidPlacement( parameters ) )
        {
            updateUserPrivacySettings( parameters );

            Vungle.loadAd( placementId, new LoadAdCallback()
            {
                @Override
                public void onAdLoad(final String id)
                {
                    showMRecAd( parameters, listener, playAdCallback );
                }

                @Override
                public void onError(final String id, final VungleException exception)
                {
                    final MaxAdapterError error = toMaxError( exception );
                    log( "MREC ad for placement " + id + " failed to load with error: " + error );

                    listener.onAdViewAdLoadFailed( error );
                }
            } );
        }
        else
        {
            log( "MREC ad failed to load due to an invalid placement id: " + placementId );
            listener.onAdViewAdLoadFailed( MaxAdapterError.INVALID_CONFIGURATION );
        }
    }

    private void showMRecAd(final MaxAdapterResponseParameters parameters, final MaxAdViewAdapterListener listener, final PlayAdCallback playAdCallback)
    {
        final String placementId = parameters.getThirdPartyAdPlacementId();
        log( "Showing MREC ad for placement: " + placementId + "..." );

        final AdConfig config = retrieveAdConfig( parameters.getServerParameters() );
        config.setAdSize( AdConfig.AdSize.VUNGLE_MREC );

        // Note: Vungle MRECs require an additional step to load. A failed "renderNativeView" would be considered a failed load.
        mrecAd = Vungle.getNativeAd( placementId, config, playAdCallback );
        if ( mrecAd != null )
        {
            mrecAdView = mrecAd.renderNativeView();
            if ( mrecAdView != null )
            {
                log( "MREC ad loaded" );
                listener.onAdViewAdLoaded( mrecAdView );
            }
            else
            {
                final MaxAdapterError error = MaxAdapterError.NO_FILL;
                log( "MREC ad failed to load: " + error );

                listener.onAdViewAdLoadFailed( error );
            }
        }
        else
        {
            // getNativeAd will return null if we try to get another ad when the placement id is already in use.
            final MaxAdapterError error = MaxAdapterError.INVALID_LOAD_STATE;
            log( "MREC ad failed to load: " + error );

            listener.onAdViewAdLoadFailed( error );
        }
    }

    //endregion

    //region Helper Methods

    private boolean isValidPlacement(final MaxAdapterResponseParameters parameters)
    {
        return Vungle.getValidPlacements().contains( parameters.getThirdPartyAdPlacementId() ) || parameters.isTesting();
    }

    private void updateUserPrivacySettings(final MaxAdapterParameters parameters)
    {
        if ( getWrappingSdk().getConfiguration().getConsentDialogState() == AppLovinSdkConfiguration.ConsentDialogState.APPLIES )
        {
            Boolean hasUserConsent = getPrivacySetting( "hasUserConsent", parameters );
            if ( hasUserConsent != null )
            {
                Vungle.Consent consentStatus = hasUserConsent ? Vungle.Consent.OPTED_IN : Vungle.Consent.OPTED_OUT;
                Vungle.updateConsentStatus( consentStatus, "" );
            }
        }

        if ( AppLovinSdk.VERSION_CODE >= 91100 )
        {
            Boolean isDoNotSell = getPrivacySetting( "isDoNotSell", parameters );
            if ( isDoNotSell != null )
            {
                Vungle.Consent ccpaStatus = isDoNotSell ? Vungle.Consent.OPTED_OUT : Vungle.Consent.OPTED_IN;
                Vungle.updateCCPAStatus( ccpaStatus );
            }
        }
    }

    private Boolean getPrivacySetting(final String privacySetting, final MaxAdapterParameters parameters)
    {
        try
        {
            // Use reflection because compiled adapters have trouble fetching `boolean` from old SDKs and `Boolean` from new SDKs (above 9.14.0)
            Class<?> parametersClass = parameters.getClass();
            Method privacyMethod = parametersClass.getMethod( privacySetting );
            return (Boolean) privacyMethod.invoke( parameters );
        }
        catch ( Exception exception )
        {
            log( "Error getting privacy setting " + privacySetting + " with exception: ", exception );
            return ( AppLovinSdk.VERSION_CODE >= 9140000 ) ? null : false;
        }
    }

    private AdConfig retrieveAdConfig(final Bundle serverParameters)
    {
        final AdConfig config = new AdConfig();

        if ( serverParameters.containsKey( "ordinal" ) )
        {
            config.setOrdinal( serverParameters.getInt( "ordinal" ) );
        }

        if ( serverParameters.containsKey( "immersive_mode" ) )
        {
            config.setImmersiveMode( serverParameters.getBoolean( "immersive_mode" ) );
        }

        if ( serverParameters.containsKey( "is_muted" ) ) // Introduced in 9.10.0
        {
            config.setMuted( serverParameters.getBoolean( "is_muted" ) );
        }

        return config;
    }

    private AdConfig.AdSize vungleBannerAdSize(final MaxAdFormat adFormat)
    {
        if ( adFormat == MaxAdFormat.BANNER )
        {
            return AdConfig.AdSize.BANNER;
        }
        else if ( adFormat == MaxAdFormat.LEADER )
        {
            return AdConfig.AdSize.BANNER_LEADERBOARD;
        }
        else
        {
            throw new IllegalArgumentException( "Unsupported banner ad format: " + adFormat );
        }
    }

    private static MaxAdapterError toMaxError(final VungleException exception)
    {
        final int vungleErrorCode = exception.getExceptionCode();
        final MaxAdapterError adapterError;

        if ( vungleErrorCode == VungleException.NO_SERVE )
        {
            adapterError = MaxAdapterError.NO_FILL;
        }
        else if ( vungleErrorCode == VungleException.UNKNOWN_ERROR )
        {
            adapterError = MaxAdapterError.UNSPECIFIED;
        }
        else if ( vungleErrorCode == VungleException.CONFIGURATION_ERROR )
        {
            adapterError = MaxAdapterError.INVALID_CONFIGURATION;
        }
        else if ( vungleErrorCode == VungleException.AD_EXPIRED )
        {
            adapterError = MaxAdapterError.INVALID_LOAD_STATE;
        }
        else if ( vungleErrorCode == VungleException.MISSING_REQUIRED_ARGUMENTS_FOR_INIT )
        {
            adapterError = MaxAdapterError.NOT_INITIALIZED;
        }
        else if ( vungleErrorCode == VungleException.APPLICATION_CONTEXT_REQUIRED )
        {
            adapterError = MaxAdapterError.NOT_INITIALIZED;
        }
        else if ( vungleErrorCode == VungleException.VUNGLE_NOT_INTIALIZED )
        {
            adapterError = MaxAdapterError.NOT_INITIALIZED;
        }
        else if ( vungleErrorCode == VungleException.AD_UNABLE_TO_PLAY )
        {
            adapterError = MaxAdapterError.INTERNAL_ERROR;
        }
        else if ( vungleErrorCode == VungleException.AD_FAILED_TO_DOWNLOAD )
        {
            adapterError = MaxAdapterError.NO_CONNECTION;
        }
        else if ( vungleErrorCode == VungleException.PLACEMENT_NOT_FOUND )
        {
            adapterError = MaxAdapterError.INVALID_CONFIGURATION;
        }
        else if ( vungleErrorCode == VungleException.SERVER_RETRY_ERROR )
        {
            adapterError = MaxAdapterError.SERVER_ERROR;
        }
        else
        {
            adapterError = MaxAdapterError.UNSPECIFIED;
        }

        return new MaxAdapterError( adapterError.getErrorCode(), adapterError.getErrorMessage(), vungleErrorCode, exception.getLocalizedMessage() );
    }

    //endregion

    private class InterstitialAdListener
            implements PlayAdCallback
    {
        private final MaxInterstitialAdapterListener listener;

        InterstitialAdListener(final MaxInterstitialAdapterListener listener)
        {
            this.listener = listener;
        }

        @Override
        public void onAdStart(final String id)
        {
            // Old CIMP location. Caused discrepancies with Vungle.
            log( "Interstitial ad started" );
        }

        @Override
        public void onAdViewed(final String id)
        {
            log( "Interstitial ad displayed" );
            listener.onInterstitialAdDisplayed();
        }

        @Override
        public void onAdClick(final String id)
        {
            log( "Interstitial ad clicked" );
            listener.onInterstitialAdClicked();
        }

        @Override
        public void onAdLeftApplication(final String id)
        {
            log( "Interstitial ad left application" );
        }

        @Override
        public void onAdEnd(final String id)
        {
            log( "Interstitial ad hidden" );
            listener.onInterstitialAdHidden();
        }

        @Override
        public void onError(final String id, final VungleException exception)
        {
            log( "Interstitial ad failed to display" );
            listener.onInterstitialAdDisplayFailed( toMaxError( exception ) );
        }

        @Override
        public void onAdRewarded(final String id)
        {
            // Interstitial ad listener does not use this method
        }

        @Override
        public void onAdEnd(final String id, final boolean completed, final boolean isCTAClicked)
        {
            // Deprecated callback
        }
    }

    private class RewardedAdListener
            implements PlayAdCallback
    {
        private final MaxRewardedAdapterListener listener;

        private boolean hasGrantedReward;

        RewardedAdListener(final MaxRewardedAdapterListener listener)
        {
            this.listener = listener;
        }

        @Override
        public void onAdStart(final String id)
        {
            // Old CIMP location. Caused discrepancies with Vungle.
            log( "Rewarded ad started" );
        }

        @Override
        public void onAdViewed(final String id)
        {
            log( "Rewarded ad displayed" );

            listener.onRewardedAdDisplayed();
            listener.onRewardedAdVideoStarted();
        }

        @Override
        public void onAdClick(final String id)
        {
            log( "Rewarded ad clicked" );
            listener.onRewardedAdClicked();
        }

        @Override
        public void onAdRewarded(final String id)
        {
            log( "Rewarded ad user did earn reward" );
            hasGrantedReward = true;
        }

        @Override
        public void onAdLeftApplication(final String id)
        {
            log( "Rewarded ad left application" );
        }

        @Override
        public void onAdEnd(final String id)
        {
            log( "Rewarded ad video completed" );
            listener.onRewardedAdVideoCompleted();

            if ( hasGrantedReward || shouldAlwaysRewardUser() )
            {
                final MaxReward reward = getReward();
                log( "Rewarded user with reward: " + reward );
                listener.onUserRewarded( reward );
            }

            log( "Rewarded ad hidden" );
            listener.onRewardedAdHidden();
        }

        @Override
        public void onError(final String id, final VungleException exception)
        {
            log( "Rewarded ad failed to display" );
            listener.onRewardedAdDisplayFailed( toMaxError( exception ) );
        }

        @Override
        public void onAdEnd(final String id, final boolean completed, final boolean isCTAClicked)
        {
            // Deprecated callback
        }
    }

    private class AdViewAdListener
            implements PlayAdCallback
    {
        private final String                   adFormatLabel;
        private final MaxAdViewAdapterListener listener;

        AdViewAdListener(final String adFormatLabel, final MaxAdViewAdapterListener listener)
        {
            this.adFormatLabel = adFormatLabel;
            this.listener = listener;
        }

        @Override
        public void onAdStart(final String id)
        {
            // Old CIMP location. Caused discrepancies with Vungle.
            log( adFormatLabel + " ad started" );
        }

        @Override
        public void onAdViewed(final String id)
        {
            log( adFormatLabel + " ad displayed" );
            listener.onAdViewAdDisplayed();
        }

        @Override
        public void onAdClick(final String id)
        {
            log( adFormatLabel + " ad clicked" );
            listener.onAdViewAdClicked();
        }

        @Override
        public void onAdLeftApplication(final String id)
        {
            log( adFormatLabel + " ad left application" );
        }

        @Override
        public void onAdEnd(final String id)
        {
            log( adFormatLabel + " ad hidden" );
            listener.onAdViewAdHidden();
        }

        @Override
        public void onError(final String id, final VungleException exception)
        {
            log( adFormatLabel + " ad display failed" );
            listener.onAdViewAdDisplayFailed( toMaxError( exception ) );
        }

        @Override
        public void onAdRewarded(final String id)
        {
            // Ad view ad listener does not use this method
        }

        @Override
        public void onAdEnd(final String id, final boolean completed, final boolean isCTAClicked)
        {
            // Deprecated callback
        }
    }
}
