package com.applovin.mediation.adapters;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.NonNull;

import com.applovin.mediation.MaxAdFormat;
import com.applovin.mediation.MaxReward;
import com.applovin.mediation.adapter.MaxAdViewAdapter;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.MaxInterstitialAdapter;
import com.applovin.mediation.adapter.MaxNativeAdAdapter;
import com.applovin.mediation.adapter.MaxRewardedAdapter;
import com.applovin.mediation.adapter.MaxSignalProvider;
import com.applovin.mediation.adapter.listeners.MaxAdViewAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxNativeAdAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxRewardedAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxSignalCollectionListener;
import com.applovin.mediation.adapter.parameters.MaxAdapterInitializationParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterResponseParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterSignalCollectionParameters;
import com.applovin.mediation.nativeAds.MaxNativeAd;
import com.applovin.mediation.nativeAds.MaxNativeAdView;
import com.applovin.sdk.AppLovinSdk;
import com.applovin.sdk.AppLovinSdkConfiguration;
import com.applovin.sdk.AppLovinSdkUtils;

import java.lang.ref.WeakReference;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import sg.bigo.ads.BigoAdSdk;
import sg.bigo.ads.ConsentOptions;
import sg.bigo.ads.api.AdConfig;
import sg.bigo.ads.api.AdError;
import sg.bigo.ads.api.AdInteractionListener;
import sg.bigo.ads.api.AdLoadListener;
import sg.bigo.ads.api.AdOptionsView;
import sg.bigo.ads.api.AdSize;
import sg.bigo.ads.api.AdTag;
import sg.bigo.ads.api.BannerAd;
import sg.bigo.ads.api.BannerAdLoader;
import sg.bigo.ads.api.BannerAdRequest;
import sg.bigo.ads.api.InterstitialAd;
import sg.bigo.ads.api.InterstitialAdLoader;
import sg.bigo.ads.api.InterstitialAdRequest;
import sg.bigo.ads.api.MediaView;
import sg.bigo.ads.api.NativeAd;
import sg.bigo.ads.api.NativeAdLoader;
import sg.bigo.ads.api.NativeAdRequest;
import sg.bigo.ads.api.RewardAdInteractionListener;
import sg.bigo.ads.api.RewardVideoAd;
import sg.bigo.ads.api.RewardVideoAdLoader;
import sg.bigo.ads.api.RewardVideoAdRequest;

/**
 * 1. 在外部自行初始化BigoAdSdk
 * 2. 依赖adapter
 * 3. 平台配置
 */
public class BigoAdsMediationAdapter extends MediationAdapterBase implements MaxInterstitialAdapter,
        MaxRewardedAdapter, MaxAdViewAdapter, MaxNativeAdAdapter, MaxSignalProvider {

    private static final String TAG = "BigoAdsMediationAdapter";

    private static final String DEBUG_KEY   = "debug";
    private static final String APP_KEY     = "app_key";
    private static final String CHANNEL_KEY = "channel";

    private static final AtomicBoolean        initialized = new AtomicBoolean();
    private static       InitializationStatus status;

    private InterstitialAd interstitialAd;
    private RewardVideoAd  rewardedAd;
    private BannerAd       bannerAd;
    private NativeAd       nativeAd;

    public BigoAdsMediationAdapter(final AppLovinSdk appLovinSdk) {
        super(appLovinSdk);
    }

    //region init and destroy
    @Override
    public void initialize(MaxAdapterInitializationParameters parameters, Activity activity, OnCompletionListener onCompletionListener) {
        if (initialized.compareAndSet(false, true)) {
            status = InitializationStatus.INITIALIZING;

            // GDPR options
            if (getWrappingSdk().getConfiguration().getConsentDialogState() == AppLovinSdkConfiguration.ConsentDialogState.APPLIES) {
                Boolean hasUserConsent = getPrivacySetting("hasUserConsent", parameters);
                if (hasUserConsent != null) {
                    BigoAdSdk.setUserConsent(activity, ConsentOptions.GDPR, hasUserConsent);
                }
            }
            // CCPA options
            if (AppLovinSdk.VERSION_CODE >= 91100) {
                Boolean isDoNotSell = getPrivacySetting("isDoNotSell", parameters);
                if (isDoNotSell != null) {
                    BigoAdSdk.setUserConsent(activity, ConsentOptions.CCPA, !isDoNotSell); // isDoNotSell means user has opted out of selling data.
                }
            }

            final Bundle serverParameters = parameters.getServerParameters();
            final String appId = serverParameters.getString(APP_KEY);
            if (TextUtils.isEmpty(appId)) {
                if (!BigoAdSdk.isInitialized()) {
                    e("Failed to init bigo ads sdk due to empty app key.");
                }
                return;
            }
            final String channel = serverParameters.getString(CHANNEL_KEY);
            final String debug = serverParameters.getString(DEBUG_KEY);
            final boolean isDebug = "true".equals(debug);
            log("Initializing Bigo ads SDK with app id: " + appId);

            Boolean isAgeRestrictedUser = getPrivacySetting("isAgeRestrictedUser", parameters);
            if (isAgeRestrictedUser != null) {
                // coppa(isAgeRestrictedUser ? 1 : 0);
            }

            if (activity == null) {
                e("Failed to init bigo ads sdk due to empty activity.");
                return;
            }
            AdConfig.Builder adConfigBuilder = new AdConfig.Builder();
            adConfigBuilder.setAppId(appId);
            adConfigBuilder.setChannel(channel);
            adConfigBuilder.setDebug(isDebug);
            AdConfig adConfig = adConfigBuilder.build();

            BigoAdSdk.initialize(activity.getApplicationContext(), adConfig, new BigoAdSdk.InitListener() {
                @Override
                public void onInitialized() {
                    log("Bigo ads sdk initialized");
                    status = InitializationStatus.INITIALIZED_SUCCESS;
                    onCompletionListener.onCompletion(status, null);
                }
            });
        } else {
            log("Bigo ads sdk attempted initialization already - marking initialization as completed");
            onCompletionListener.onCompletion(status, null);
        }
    }

    @Override
    public String getSdkVersion() {
        return BigoAdSdk.getSDKVersion();
    }

    @Override
    public String getAdapterVersion() {
        return BuildConfig.ADAPTER_VERSION;
    }

    @Override
    public void onDestroy() {
        if (interstitialAd != null) {
            interstitialAd.destroy();
            interstitialAd = null;
            log("Destroy interstitial ad.");
        }
        if (rewardedAd != null) {
            rewardedAd.destroy();
            rewardedAd = null;
            log("Destroy reward video ad.");
        }
        if (bannerAd != null) {
            bannerAd.destroy();
            bannerAd = null;
            log("Destroy banner ad.");
        }
        if (nativeAd != null) {
            nativeAd.destroy();
            nativeAd = null;
            log("Destroy native ad.");
        }
    }
    // endregion

    @Override
    public void collectSignal(MaxAdapterSignalCollectionParameters maxAdapterSignalCollectionParameters, Activity activity, MaxSignalCollectionListener maxSignalCollectionListener) {
    }

    //region load and show
    @Override
    public void loadInterstitialAd(MaxAdapterResponseParameters parameters, Activity activity, MaxInterstitialAdapterListener maxInterstitialAdapterListener) {
        String slotId = parameters.getThirdPartyAdPlacementId();
        if (TextUtils.isEmpty(slotId)) {
            if (maxInterstitialAdapterListener != null)
                maxInterstitialAdapterListener.onInterstitialAdLoadFailed(toMaxError(AdError.ERROR_CODE_INTERNAL_ERROR, "empty interstitial slot."));
        }
        String bidResponse = parameters.getBidResponse();
        log("Loading " + (AppLovinSdkUtils.isValidString(bidResponse) ? "bidding " : "") + "interstitial ad with slot id " + slotId);
        InterstitialAdRequest.Builder adRequestBuilder = new InterstitialAdRequest.Builder();
        adRequestBuilder.withSlotId(slotId);

        /*if (AppLovinSdkUtils.isValidString(bidResponse)) {
            adRequestBuilder.withBid(bidResponse);
        }*/
        InterstitialAdLoader interstitialAdLoader = new InterstitialAdLoader.Builder()
                .withAdLoadListener(new InterstitialAdListener(slotId, maxInterstitialAdapterListener)).build();
        interstitialAdLoader.loadAd(adRequestBuilder.build());

    }

    @Override
    public void showInterstitialAd(MaxAdapterResponseParameters parameters, Activity activity, MaxInterstitialAdapterListener maxInterstitialAdapterListener) {
        InterstitialAd ad = interstitialAd;
        if (ad == null) {
            if (maxInterstitialAdapterListener != null)
                maxInterstitialAdapterListener.onInterstitialAdDisplayFailed(toMaxError(AdError.ERROR_CODE_INTERNAL_ERROR, "empty interstitial display ad."));
            return;
        }
        if (ad.isExpired()) {
            if (maxInterstitialAdapterListener != null)
                maxInterstitialAdapterListener.onInterstitialAdDisplayFailed(toMaxError(AdError.ERROR_CODE_AD_EXPIRED, "expired interstitial display ad."));
            return;
        }
        log("Showing interstitial ad with slot id " + parameters.getThirdPartyAdPlacementId());
        ad.show();
    }

    @Override
    public void loadRewardedAd(MaxAdapterResponseParameters parameters, Activity activity, MaxRewardedAdapterListener maxRewardedAdapterListener) {
        String slotId = parameters.getThirdPartyAdPlacementId();
        if (TextUtils.isEmpty(slotId)) {
            if (maxRewardedAdapterListener != null)
                maxRewardedAdapterListener.onRewardedAdLoadFailed(toMaxError(AdError.ERROR_CODE_INTERNAL_ERROR, "empty reward video slot."));
        }
        String bidResponse = parameters.getBidResponse();
        log("Loading " + (AppLovinSdkUtils.isValidString(bidResponse) ? "bidding " : "") + "rewarded ad with slot id " + slotId);

        // NOTE: No privacy APIs to toggle before ad load
        /*if (AppLovinSdkUtils.isValidString(bidResponse)) {
            adSlotBuilder.withBid(bidResponse);
        }*/

        RewardVideoAdRequest rewardVideoAdRequest = new RewardVideoAdRequest.Builder()
                .withSlotId(slotId).build();
        RewardVideoAdLoader rewardVideoAdLoader = new RewardVideoAdLoader.Builder()
                .withAdLoadListener(new RewardedAdListener(slotId, maxRewardedAdapterListener))
                .build();
        rewardVideoAdLoader.loadAd(rewardVideoAdRequest);
    }

    @Override
    public void showRewardedAd(MaxAdapterResponseParameters parameters, Activity activity, MaxRewardedAdapterListener maxRewardedAdapterListener) {
        RewardVideoAd ad = rewardedAd;
        if (ad == null) {
            if (maxRewardedAdapterListener != null)
                maxRewardedAdapterListener.onRewardedAdDisplayFailed(toMaxError(AdError.ERROR_CODE_INTERNAL_ERROR, "empty reward video display ad."));
            return;
        }
        if (ad.isExpired()) {
            if (maxRewardedAdapterListener != null)
                maxRewardedAdapterListener.onRewardedAdDisplayFailed(toMaxError(AdError.ERROR_CODE_AD_EXPIRED, "expired reward video display ad."));
            return;
        }
        log("Showing rewarded ad for slot id " + parameters.getThirdPartyAdPlacementId());

        // Configure userReward from server.
        configureReward(parameters);
        ad.show();
    }

    @Override
    public void loadNativeAd(MaxAdapterResponseParameters parameters, Activity activity, MaxNativeAdAdapterListener maxNativeAdAdapterListener) {
        String slotId = parameters.getThirdPartyAdPlacementId();
        if (TextUtils.isEmpty(slotId)) {
            if (maxNativeAdAdapterListener != null)
                maxNativeAdAdapterListener.onNativeAdLoadFailed(toMaxError(AdError.ERROR_CODE_INTERNAL_ERROR, "empty native slot."));
        }
        log("Loading native ad with slot id \"" + slotId + "\"...");
        NativeAdRequest nativeAdRequest = new NativeAdRequest.Builder().withSlotId(slotId).build();
        NativeAdLoader nativeAdLoader = new NativeAdLoader.Builder()
                .withAdLoadListener(new NativeAdListener(parameters, activity, maxNativeAdAdapterListener))
                .build();
        nativeAdLoader.loadAd(nativeAdRequest);
    }

    @Override
    public void loadAdViewAd(MaxAdapterResponseParameters parameters, MaxAdFormat adFormat, Activity activity, MaxAdViewAdapterListener maxAdViewAdapterListener) {
        String bidResponse = parameters.getBidResponse();
        String slotId = parameters.getThirdPartyAdPlacementId();
        if (TextUtils.isEmpty(slotId)) {
            if (maxAdViewAdapterListener != null)
                maxAdViewAdapterListener.onAdViewAdLoadFailed(toMaxError(AdError.ERROR_CODE_INTERNAL_ERROR, "empty adView slot."));
        }
        log("Loading " + (AppLovinSdkUtils.isValidString(bidResponse) ? "bidding " : "") + adFormat.getLabel() + " ad for slot id \"" + slotId + "\"...");

        AppLovinSdkUtils.Size adSize = adFormat.getSize();
        BannerAdRequest.Builder adRequestBuilder = new BannerAdRequest.Builder();
        /*if (AppLovinSdkUtils.isValidString(bidResponse)) {
            adRequestBuilder.withBid(bidResponse);
        }*/
        adRequestBuilder.withSlotId(slotId);
        int width = adSize.getWidth(); // dp
        int height = adSize.getHeight();
        adRequestBuilder.withAdSizes(fromBigoAdSize(width, height));

        // Implement a SampleAdListener and forward callbacks to mediation. The callback forwarding
        // is handled by SampleBannerEventFowarder.
        BannerAdLoader adLoader = new BannerAdLoader.Builder()
                .withAdLoadListener(new AdViewListener(slotId, adFormat, maxAdViewAdapterListener)).build();
        // Make an ad request.
        adLoader.loadAd(adRequestBuilder.build());
    }
    //endregion

    //region Helper Methods
    private Boolean getPrivacySetting(final String privacySetting, final MaxAdapterParameters parameters) {
        try {
            // Use reflection because compiled adapters have trouble fetching `boolean` from old SDKs and `Boolean` from new SDKs (above 9.14.0)
            Class<?> parametersClass = parameters.getClass();
            Method privacyMethod = parametersClass.getMethod(privacySetting);
            return (Boolean) privacyMethod.invoke(parameters);
        } catch (Exception exception) {
            log("Error getting privacy setting " + privacySetting + " with exception: ", exception);
            return (AppLovinSdk.VERSION_CODE >= 9140000) ? null : false;
        }
    }

    private AdSize fromBigoAdSize(int widthInDp, int heightInDp) {
        if (heightInDp >= AdSize.LARGE_RECTANGLE.getHeight()) {
            return AdSize.LARGE_RECTANGLE;
        } else if (heightInDp >= AdSize.MEDIUM_RECTANGLE.getHeight()) {
            return AdSize.MEDIUM_RECTANGLE;
        } else if (heightInDp >= AdSize.LARGE_BANNER.getHeight()) {
            return AdSize.LARGE_BANNER;
        } else {
            // Default to standard banner size
            return AdSize.BANNER;
        }
    }

    private static MaxAdapterError toMaxError(final int errorCode, final String bigoAdsErrorMessage) {
        MaxAdapterError adapterError = MaxAdapterError.UNSPECIFIED;
        switch (errorCode) {
            case AdError.ERROR_CODE_UNINITIALIZED:
                adapterError = MaxAdapterError.NOT_INITIALIZED;
                break;
            case AdError.ERROR_CODE_INVALID_REQUEST:
                adapterError = MaxAdapterError.BAD_REQUEST;
                break;
            case AdError.ERROR_CODE_NETWORK_ERROR:
                adapterError = MaxAdapterError.NO_CONNECTION;
                break;
            case AdError.ERROR_CODE_NO_FILL:
                adapterError = MaxAdapterError.NO_FILL;
                break;
            case AdError.ERROR_CODE_ASSETS_ERROR:
                adapterError = MaxAdapterError.MISSING_REQUIRED_NATIVE_AD_ASSETS;
                break;
            case AdError.ERROR_CODE_AD_EXPIRED:
                adapterError = MaxAdapterError.AD_EXPIRED;
                break;
            case AdError.ERROR_CODE_INTERNAL_ERROR:
            case AdError.ERROR_CODE_AD_DISABLE:
            case AdError.ERROR_CODE_FULLSCREEN_AD_FAILED_TO_SHOW:
            case AdError.ERROR_CODE_VIDEO_ERROR:
            case AdError.ERROR_CODE_NATIVE_VIEW_MISSING:
                adapterError = MaxAdapterError.INTERNAL_ERROR;
                break;
        }
        return new MaxAdapterError(adapterError.getErrorCode(), adapterError.getErrorMessage(), errorCode, bigoAdsErrorMessage);
    }
    //endregion

    //region InterstitialAdListener
    private final class InterstitialAdListener implements AdLoadListener<InterstitialAd> {

        private final String                         slotId;
        private final MaxInterstitialAdapterListener listener;

        InterstitialAdListener(final String slotId, final MaxInterstitialAdapterListener listener) {
            this.slotId = slotId;
            this.listener = listener;
        }

        @Override
        public void onError(@NonNull AdError adError) {
            MaxAdapterError adapterError = toMaxError(adError.getCode(), adError.getMessage());
            log("Interstitial ad (" + slotId + ") failed to load with error: " + adapterError);
            listener.onInterstitialAdLoadFailed(adapterError);
        }

        @Override
        public void onAdLoaded(@NonNull InterstitialAd interstitialAd) {
            interstitialAd.setAdInteractionListener(new AdInteractionListener() {
                @Override
                public void onAdError(@NonNull AdError adError) {
                    onError(adError);
                }

                @Override
                public void onAdImpression() {
                    log("Bigo ads interstitial ad impression.");
                    listener.onInterstitialAdDisplayed();
                }

                @Override
                public void onAdClicked() {
                    log("Bigo ads interstitial ad clicked.");
                    listener.onInterstitialAdClicked();
                }

                @Override
                public void onAdOpened() {
                    log("Bigo ads interstitial ad opened.");
                }

                @Override
                public void onAdClosed() {
                    log("Bigo ads interstitial ad closed.");
                    listener.onInterstitialAdHidden();
                }
            });

            BigoAdsMediationAdapter.this.interstitialAd = interstitialAd;
            log("Interstitial ad loaded: " + slotId);
            listener.onInterstitialAdLoaded();
        }
    }
    //endregion

    //region RewardedAdListener

    /**
     * The reward callback in Applovin adapter invoked when reward video dismiss.
     */
    private class RewardedAdListener implements AdLoadListener<RewardVideoAd> {
        private final String                     slotId;
        private final MaxRewardedAdapterListener listener;

        private boolean hasGrantedReward;

        RewardedAdListener(final String slotId, final MaxRewardedAdapterListener listener) {
            this.slotId = slotId;
            this.listener = listener;
        }

        @Override
        public void onError(@NonNull AdError adError) {
            MaxAdapterError adapterError = toMaxError(adError.getCode(), adError.getMessage());
            log("Rewarded ad (" + slotId + ") failed to load with error: " + adapterError);
            listener.onRewardedAdLoadFailed(adapterError);
        }

        @Override
        public void onAdLoaded(@NonNull RewardVideoAd rewardVideoAd) {

            rewardVideoAd.setAdInteractionListener(new RewardAdInteractionListener() {
                @Override
                public void onAdRewarded() {
                    log("Rewarded ad video completed: " + slotId);
                    listener.onRewardedAdVideoCompleted();

                    log("Rewarded user with reward.");
                    hasGrantedReward = true;
                }

                @Override
                public void onAdError(@NonNull AdError adError) {
                    log("Rewarded ad failed to display: " + slotId);
                    listener.onRewardedAdDisplayFailed(MaxAdapterError.UNSPECIFIED);
                }

                @Override
                public void onAdImpression() {
                    log("Rewarded ad displayed: " + slotId);

                    listener.onRewardedAdDisplayed();
                    listener.onRewardedAdVideoStarted();
                }

                @Override
                public void onAdClicked() {
                    log("Rewarded ad clicked: " + slotId);
                    listener.onRewardedAdClicked();
                }

                @Override
                public void onAdOpened() {

                }

                @Override
                public void onAdClosed() {
                    log("Rewarded ad hidden: " + slotId);
                    if (hasGrantedReward || shouldAlwaysRewardUser()) {
                        final MaxReward reward = getReward();
                        log("Rewarded user with reward: " + reward);
                        listener.onUserRewarded(reward);
                    }

                    listener.onRewardedAdHidden();
                }
            });
            BigoAdsMediationAdapter.this.rewardedAd = rewardVideoAd;

            log("Rewarded ad loaded: " + slotId);
            listener.onRewardedAdLoaded();
        }
    }
    //endregion

    //region AdViewListener
    private class AdViewListener implements AdLoadListener<BannerAd> {
        private final String                   slotId;
        private final MaxAdFormat              adFormat;
        private final MaxAdViewAdapterListener listener;

        AdViewListener(final String slotId, final MaxAdFormat adFormat, final MaxAdViewAdapterListener listener) {
            this.slotId = slotId;
            this.adFormat = adFormat;
            this.listener = listener;
        }

        @Override
        public void onError(@NonNull AdError adError) {
            MaxAdapterError adapterError = toMaxError(adError.getCode(), adError.getMessage());
            log(adFormat.getLabel() + " ad (" + slotId + ") failed to load with error: " + adapterError);
            listener.onAdViewAdLoadFailed(adapterError);
        }

        @Override
        public void onAdLoaded(@NonNull BannerAd bannerAd) {
            log(adFormat.getLabel() + " ad (" + slotId + ") loaded.");

            bannerAd.setAdInteractionListener(new AdInteractionListener() {
                @Override
                public void onAdError(@NonNull AdError adError) {
                    onError(adError);
                }

                @Override
                public void onAdImpression() {
                    log(adFormat.getLabel() + " ad shown: " + slotId);
                    listener.onAdViewAdDisplayed();
                }

                @Override
                public void onAdClicked() {

                    log(adFormat.getLabel() + " ad clicked: " + slotId);
                    listener.onAdViewAdClicked();
                }

                @Override
                public void onAdOpened() {

                }

                @Override
                public void onAdClosed() {

                }
            });
            BigoAdsMediationAdapter.this.bannerAd = bannerAd;
            listener.onAdViewAdLoaded(bannerAd.adView());
        }
    }
    //endregion

    //region NativeAdListener
    private class NativeAdListener implements AdLoadListener<NativeAd>, AdInteractionListener {
        final String                     slotId;
        final Bundle                     serverParameters;
        final WeakReference<Activity>    activityRef;
        final MaxNativeAdAdapterListener listener;

        NativeAdListener(final MaxAdapterResponseParameters parameters,
                         final Activity activity,
                         final MaxNativeAdAdapterListener listener) {
            this.slotId = parameters.getThirdPartyAdPlacementId();
            this.serverParameters = parameters.getServerParameters();
            this.activityRef = new WeakReference<>(activity);
            this.listener = listener;
        }

        @Override
        public void onError(@NonNull AdError adError) {
            log("Native ad (" + slotId + ") failed to load with error code (" + adError.getCode() + ") and message: " + adError.getMessage());

            MaxAdapterError adapterError = toMaxError(adError.getCode(), adError.getMessage());
            listener.onNativeAdLoadFailed(adapterError);
        }

        @Override
        public void onAdLoaded(@NonNull NativeAd nativeAd) {
            final Activity activity = activityRef.get();
            if (activity == null) {
                log("Native ad (" + slotId + ") failed to load: activity reference is null when ad is loaded");
                listener.onNativeAdLoadFailed(MaxAdapterError.INVALID_LOAD_STATE);
                return;
            }
            nativeAd.setAdInteractionListener(this);
            log("Native ad loaded: " + slotId + ". Preparing assets...");
            BigoAdsMediationAdapter.this.nativeAd = nativeAd;

            // Create MaxNativeAd after images are loaded from remote URLs
            AppLovinSdkUtils.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    Context context = activity;
                    final MediaView mediaView = new MediaView(context);
                    final AdOptionsView adOptionsView = new AdOptionsView(context);

                    log("Creating native ad with assets");

                    MaxNativeAd.Builder builder = new MaxNativeAd.Builder()
                            .setAdFormat(MaxAdFormat.NATIVE)
                            .setTitle(nativeAd.getTitle())
                            .setBody(nativeAd.getDescription())
                            .setCallToAction(nativeAd.getCallToAction())
                            // .setIcon(icon)
                            .setMediaView(mediaView)
                            .setOptionsView(adOptionsView);
                    MaxNativeAd maxNativeAd = new MaxBigoNativeAd(builder);

                    log("Native ad fully loaded: " + slotId);
                    listener.onNativeAdLoaded(maxNativeAd, null);
                }
            });
        }

        @Override
        public void onAdError(@NonNull AdError adError) {
            onError(adError);
        }

        @Override
        public void onAdImpression() {
            log("Native ad displayed: " + slotId);
            listener.onNativeAdDisplayed(null);
        }

        @Override
        public void onAdClicked() {
            // This callback is never called
            log("Native ad clicked: " + slotId);
            listener.onNativeAdClicked();
        }

        @Override
        public void onAdOpened() {

        }

        @Override
        public void onAdClosed() {

        }
    }
    //endregion

    //region MaxBigoNativeAd
    private class MaxBigoNativeAd
            extends MaxNativeAd {
        public MaxBigoNativeAd(final Builder builder) {
            super(builder);
        }

        @Override
        public void prepareViewForInteraction(final MaxNativeAdView maxNativeAdView) {
            NativeAd nativeAd = BigoAdsMediationAdapter.this.nativeAd;
            if (nativeAd == null) {
                e("Failed to register native ad view due to empty native ad.");
                return;
            }
            if (maxNativeAdView == null) {
                e("Failed to register native ad view due to empty ad view.");
                return;
            }

            List<View> clickableViews = new ArrayList<>();
            if (AppLovinSdkUtils.isValidString(getTitle()) && maxNativeAdView.getTitleTextView() != null) {
                maxNativeAdView.getTitleTextView().setTag(AdTag.TITLE);
                clickableViews.add(maxNativeAdView.getTitleTextView());
            }
            if (AppLovinSdkUtils.isValidString(getBody()) && maxNativeAdView.getBodyTextView() != null) {
                maxNativeAdView.getBodyTextView().setTag(AdTag.DESCRIPTION);
                clickableViews.add(maxNativeAdView.getBodyTextView());
            }
            if (getIcon() != null && maxNativeAdView.getIconImageView() != null) {
                clickableViews.add(maxNativeAdView.getIconImageView());
            }
            if (getMediaView() != null && maxNativeAdView.getMediaContentViewGroup() != null) {
                clickableViews.add(maxNativeAdView.getMediaContentViewGroup());
            }
            if (maxNativeAdView.getCallToActionButton() != null) {
                maxNativeAdView.getCallToActionButton().setTag(AdTag.CALL_TO_ACTION);
                clickableViews.add(maxNativeAdView.getCallToActionButton());
            }

            View mediaViewObject = getMediaView();
            View optionsViewObject = getOptionsView();
            MediaView mediaView = mediaViewObject instanceof MediaView ? (MediaView) mediaViewObject : null;
            AdOptionsView adOptionsView = optionsViewObject instanceof AdOptionsView ? (AdOptionsView) optionsViewObject : null;
            if (mediaView == null) {
                e("Warn: media view is null or unmatched sg.bigo.ads.api.MediaView type, can not render media.");
            }
            if (adOptionsView == null) {
                e("Warn: options view is null or unmatched sg.bigo.ads.api.AdOptionsView type, can not render options view.");
            }
            nativeAd.registerViewForInteraction(maxNativeAdView, mediaView, maxNativeAdView.getIconImageView(), adOptionsView, clickableViews);
        }
    }
    //endregion
}
