package com.applovin.mediation.adapters;

import android.app.Activity;
import android.content.Context;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.text.TextUtils;
import android.view.View;
import android.view.ViewGroup;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.applovin.mediation.MaxAdFormat;
import com.applovin.mediation.MaxReward;
import com.applovin.mediation.adapter.MaxAdViewAdapter;
import com.applovin.mediation.adapter.MaxAdapterError;
import com.applovin.mediation.adapter.MaxInterstitialAdapter;
import com.applovin.mediation.adapter.MaxNativeAdAdapter;
import com.applovin.mediation.adapter.MaxRewardedAdapter;
import com.applovin.mediation.adapter.MaxSignalProvider;
import com.applovin.mediation.adapter.listeners.MaxAdViewAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxInterstitialAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxNativeAdAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxRewardedAdapterListener;
import com.applovin.mediation.adapter.listeners.MaxSignalCollectionListener;
import com.applovin.mediation.adapter.parameters.MaxAdapterInitializationParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterResponseParameters;
import com.applovin.mediation.adapter.parameters.MaxAdapterSignalCollectionParameters;
import com.applovin.mediation.adapters.bigo.BuildConfig;
import com.applovin.mediation.nativeAds.MaxNativeAd;
import com.applovin.mediation.nativeAds.MaxNativeAdView;
import com.applovin.sdk.AppLovinSdk;
import com.applovin.sdk.AppLovinSdkConfiguration;
import com.applovin.sdk.AppLovinSdkUtils;

import org.json.JSONObject;

import java.lang.ref.WeakReference;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import sg.bigo.ads.BigoAdSdk;
import sg.bigo.ads.ConsentOptions;
import sg.bigo.ads.api.AdConfig;
import sg.bigo.ads.api.AdError;
import sg.bigo.ads.api.AdInteractionListener;
import sg.bigo.ads.api.AdLoadListener;
import sg.bigo.ads.api.AdOptionsView;
import sg.bigo.ads.api.AdSize;
import sg.bigo.ads.api.AdTag;
import sg.bigo.ads.api.BannerAd;
import sg.bigo.ads.api.BannerAdLoader;
import sg.bigo.ads.api.BannerAdRequest;
import sg.bigo.ads.api.InterstitialAd;
import sg.bigo.ads.api.InterstitialAdLoader;
import sg.bigo.ads.api.InterstitialAdRequest;
import sg.bigo.ads.api.MediaView;
import sg.bigo.ads.api.NativeAd;
import sg.bigo.ads.api.NativeAdLoader;
import sg.bigo.ads.api.NativeAdRequest;
import sg.bigo.ads.api.RewardAdInteractionListener;
import sg.bigo.ads.api.RewardVideoAd;
import sg.bigo.ads.api.RewardVideoAdLoader;
import sg.bigo.ads.api.RewardVideoAdRequest;

/**
 * 1. 在外部自行初始化BigoAdSdk
 * 2. 依赖adapter
 * 3. 平台配置
 */
public class BigoAdsMediationAdapter extends MediationAdapterBase implements MaxInterstitialAdapter,
        MaxRewardedAdapter, MaxAdViewAdapter, MaxNativeAdAdapter, MaxSignalProvider {

    private static final String TAG = "BigoAdsMediationAdapter";
    private static final String KEY_CREATIVE_ID = "creative_id";

    private static       String extString;

    private Context        applicationContext;
    private InterstitialAd interstitialAd;
    private RewardVideoAd  rewardedAd;
    private BannerAd       bannerAd;
    private NativeAd       nativeAd;

    public BigoAdsMediationAdapter(final AppLovinSdk appLovinSdk) {
        super(appLovinSdk);
    }

    //region init and destroy
    @Override
    public void initialize(MaxAdapterInitializationParameters parameters, Activity activity, OnCompletionListener onCompletionListener) {
        if (applicationContext == null && activity != null) {
            applicationContext = activity.getApplicationContext();
        }
        if (applicationContext != null) {
            updatePrivacyAndEnsureExt(applicationContext, parameters);
        }
        if (onCompletionListener != null) {
            onCompletionListener.onCompletion(InitializationStatus.INITIALIZED_FAILURE, "empty context");
        }
    }

    private void makeSureInitWithSlotRequest(String slotId, Activity activity, Runnable loadTask) {
        boolean waitForInitialization = !BigoAdSdk.isInitialized();
        String appId = "";
        Context context = activity;
        if (waitForInitialization) {
            // check context
            if (context == null) {
                context = applicationContext;
            }
            waitForInitialization = (context != null);

            if (waitForInitialization) {
                // using app id in server config.
                // using app id from slot id if app id in server config is empty.
                if (slotId != null && !slotId.isEmpty()) {
                    String[] slotIdSplit = slotId.split("-");
                    if (slotIdSplit != null && slotIdSplit.length >= 2 && slotIdSplit[0] != null && !slotIdSplit[0].isEmpty()) {
                        appId = slotIdSplit[0];
                    }
                }
                // check app id
                // Try to init bigo sdk whatever app id is
                // waitForInitialization = appId != null && !appId.isEmpty();
            }
        }
        if (waitForInitialization) {
            d("Initialize bigo sdk before requesting ad.");
            AtomicBoolean run = new AtomicBoolean(false);
            // make a timer to execute load task after 1000ms if onCompletion callback can not invoke expectedly
            Handler handler = new Handler(Looper.getMainLooper());
            handler.postDelayed(() -> {
                if (run.compareAndSet(false, true)) {
                    loadTask.run();
                }
            }, 1000);
            initInternal(appId, null, context, (initializationStatus, s) -> {
                // to execute load task whatever initializationStatus is
                if (initializationStatus == InitializationStatus.INITIALIZED_SUCCESS) {
                    if (run.compareAndSet(false, true)) {
                        loadTask.run();
                    }
                }
            });
        } else {
            loadTask.run();
        }
    }

    private void initInternal(String appId, @Nullable MaxAdapterInitializationParameters parameters, @NonNull Context context, @Nullable OnCompletionListener onCompletionListener) {
        // The parameters is always null in this case.
        updatePrivacy(context, parameters);

        try {
            AdConfig.Builder adConfigBuilder = new AdConfig.Builder();
            adConfigBuilder.setAppId(appId);
            AdConfig adConfig = adConfigBuilder.build();

            BigoAdSdk.initialize(context.getApplicationContext(), adConfig, new BigoAdSdk.InitListener() {
                @Override
                public void onInitialized() {
                    log("Bigo ads sdk initialized");
                    if (onCompletionListener != null) {
                        onCompletionListener.onCompletion(InitializationStatus.INITIALIZED_SUCCESS, null);
                    }
                }
            });
        } catch (Exception e) {
            if (onCompletionListener != null) {
                onCompletionListener.onCompletion(InitializationStatus.INITIALIZED_FAILURE, String.valueOf(e));
            }
            e.printStackTrace();
        }
    }

    private void updatePrivacyAndEnsureExt(@NonNull Context context, MaxAdapterInitializationParameters parameters) {
        JSONObject extJSON = updatePrivacy(context, parameters);
        if (extString == null) {
            if (extJSON == null) {
                extJSON = new JSONObject();
            }
            try {
                String appLovinSdkVersion = AppLovinSdk.VERSION;
                String adapterVersion = getAdapterVersion();
                String sdkVersion = getSdkVersion();
                if (!TextUtils.isEmpty(appLovinSdkVersion)) {
                    extJSON.putOpt("maxVersion", appLovinSdkVersion);
                }
                if (!TextUtils.isEmpty(adapterVersion)) {
                    extJSON.putOpt("adapterVersion", adapterVersion);
                }
                if (!TextUtils.isEmpty(sdkVersion)) {
                    extJSON.putOpt("sdkVersion", sdkVersion);
                }
                extString = extJSON.toString();
            } catch (Throwable e) {
                e.printStackTrace();
            }
        }
    }

    private JSONObject updatePrivacy(@NonNull Context context, @Nullable MaxAdapterInitializationParameters parameters) {
        if (parameters == null) {
            return null;
        }
        try {
            JSONObject jsonObject = null;
            if (extString == null) {
                jsonObject = new JSONObject();
            }
            // GDPR options
            Boolean hasUserConsent = null;
            if (getWrappingSdk().getConfiguration().getConsentDialogState() == AppLovinSdkConfiguration.ConsentDialogState.APPLIES) {
                hasUserConsent = getPrivacySetting("hasUserConsent", parameters);
                if (hasUserConsent != null) {
                    // BigoAdSdk.setUserConsent(context, ConsentOptions.GDPR, hasUserConsent);
                    if (!hasUserConsent) {
                        BigoAdSdk.setUserConsent(context, ConsentOptions.GDPR, true);
                    }
                }
            }
            if (jsonObject != null) {
                jsonObject.putOpt("GDPR", hasUserConsent == null ? "null" : hasUserConsent ? 1 : 0);
            }
            // CCPA options
            if (AppLovinSdk.VERSION_CODE >= 91100) {
                Boolean isDoNotSell = getPrivacySetting("isDoNotSell", parameters);
                if (isDoNotSell != null) {
                    // BigoAdSdk.setUserConsent(context, ConsentOptions.CCPA, !isDoNotSell); // isDoNotSell means user has opted out of selling data.
                    if (isDoNotSell) {
                        BigoAdSdk.setUserConsent(context, ConsentOptions.CCPA, true);
                    }
                }
                if (jsonObject != null) {
                    jsonObject.putOpt("isDoNotSell", isDoNotSell == null ? "null" : isDoNotSell ? 1 : 0);
                }
            }
            Boolean isAgeRestrictedUser = getPrivacySetting("isAgeRestrictedUser", parameters);
            if (isAgeRestrictedUser != null) {
                // coppa(isAgeRestrictedUser ? 1 : 0);
            }
            if (jsonObject != null) {
                jsonObject.putOpt("isAgeRestrictedUser", isAgeRestrictedUser == null ? "null" : isAgeRestrictedUser ? 1 : 0);
            }
            return jsonObject;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public String getSdkVersion() {
        return BigoAdSdk.getSDKVersion();
    }

    @Override
    public String getAdapterVersion() {
        return BuildConfig.ADAPTER_VERSION;
    }

    @Override
    public void onDestroy() {
        if (interstitialAd != null) {
            interstitialAd.destroy();
            interstitialAd = null;
            log("Destroy interstitial ad.");
        }
        if (rewardedAd != null) {
            rewardedAd.destroy();
            rewardedAd = null;
            log("Destroy reward video ad.");
        }
        if (bannerAd != null) {
            bannerAd.destroy();
            bannerAd = null;
            log("Destroy banner ad.");
        }
        if (nativeAd != null) {
            nativeAd.destroy();
            nativeAd = null;
            log("Destroy native ad.");
        }
    }
    // endregion

    @Override
    public void collectSignal(MaxAdapterSignalCollectionParameters maxAdapterSignalCollectionParameters, Activity activity, MaxSignalCollectionListener maxSignalCollectionListener) {
    }

    //region load and show
    @Override
    public void loadInterstitialAd(MaxAdapterResponseParameters parameters, Activity activity, MaxInterstitialAdapterListener maxInterstitialAdapterListener) {
        String slotId = parameters.getThirdPartyAdPlacementId();
        String bidResponse = parameters.getBidResponse();
        log("Loading " + (AppLovinSdkUtils.isValidString(bidResponse) ? "bidding " : "") + "interstitial ad with slot id " + slotId);

        makeSureInitWithSlotRequest(slotId, activity, () -> {
            InterstitialAdRequest.Builder adRequestBuilder = new InterstitialAdRequest.Builder();
            adRequestBuilder.withSlotId(slotId);
            /*if (AppLovinSdkUtils.isValidString(bidResponse)) {
                adRequestBuilder.withBid(bidResponse);
            }*/
            InterstitialAdLoader.Builder interstitialAdLoaderBuiler = new InterstitialAdLoader.Builder()
                    .withAdLoadListener(new InterstitialAdListener(slotId, maxInterstitialAdapterListener));
            if (!TextUtils.isEmpty(extString)) {
                interstitialAdLoaderBuiler.withExt(extString);
            }
            InterstitialAdLoader interstitialAdLoader = interstitialAdLoaderBuiler.build();
            interstitialAdLoader.loadAd(adRequestBuilder.build());
        });

    }

    @Override
    public void showInterstitialAd(MaxAdapterResponseParameters parameters, Activity activity, MaxInterstitialAdapterListener maxInterstitialAdapterListener) {
        InterstitialAd ad = interstitialAd;
        if (ad == null) {
            if (maxInterstitialAdapterListener != null)
                maxInterstitialAdapterListener.onInterstitialAdDisplayFailed(toMaxError(AdError.ERROR_CODE_INTERNAL_ERROR, "empty interstitial display ad."));
            return;
        }
        if (ad.isExpired()) {
            if (maxInterstitialAdapterListener != null)
                maxInterstitialAdapterListener.onInterstitialAdDisplayFailed(toMaxError(AdError.ERROR_CODE_AD_EXPIRED, "expired interstitial display ad."));
            return;
        }
        log("Showing interstitial ad with slot id " + parameters.getThirdPartyAdPlacementId());
        ad.show();
    }

    @Override
    public void loadRewardedAd(MaxAdapterResponseParameters parameters, Activity activity, MaxRewardedAdapterListener maxRewardedAdapterListener) {
        String slotId = parameters.getThirdPartyAdPlacementId();
        String bidResponse = parameters.getBidResponse();
        log("Loading " + (AppLovinSdkUtils.isValidString(bidResponse) ? "bidding " : "") + "rewarded ad with slot id " + slotId);

        makeSureInitWithSlotRequest(slotId, activity, () -> {
            // NOTE: No privacy APIs to toggle before ad load
            /*if (AppLovinSdkUtils.isValidString(bidResponse)) {
                adSlotBuilder.withBid(bidResponse);
            }*/
            RewardVideoAdRequest rewardVideoAdRequest = new RewardVideoAdRequest.Builder()
                    .withSlotId(slotId).build();
            RewardVideoAdLoader.Builder rewardVideoAdLoaderBuilder = new RewardVideoAdLoader.Builder()
                    .withAdLoadListener(new RewardedAdListener(slotId, maxRewardedAdapterListener));
            if (!TextUtils.isEmpty(extString)) {
                rewardVideoAdLoaderBuilder.withExt(extString);
            }
            RewardVideoAdLoader rewardVideoAdLoader = rewardVideoAdLoaderBuilder.build();
            rewardVideoAdLoader.loadAd(rewardVideoAdRequest);
        });
    }

    @Override
    public void showRewardedAd(MaxAdapterResponseParameters parameters, Activity activity, MaxRewardedAdapterListener maxRewardedAdapterListener) {
        RewardVideoAd ad = rewardedAd;
        if (ad == null) {
            if (maxRewardedAdapterListener != null)
                maxRewardedAdapterListener.onRewardedAdDisplayFailed(toMaxError(AdError.ERROR_CODE_INTERNAL_ERROR, "empty reward video display ad."));
            return;
        }
        if (ad.isExpired()) {
            if (maxRewardedAdapterListener != null)
                maxRewardedAdapterListener.onRewardedAdDisplayFailed(toMaxError(AdError.ERROR_CODE_AD_EXPIRED, "expired reward video display ad."));
            return;
        }
        log("Showing rewarded ad for slot id " + parameters.getThirdPartyAdPlacementId());

        // Configure userReward from server.
        configureReward(parameters);
        ad.show();
    }

    @Override
    public void loadNativeAd(MaxAdapterResponseParameters parameters, Activity activity, MaxNativeAdAdapterListener maxNativeAdAdapterListener) {
        String slotId = parameters.getThirdPartyAdPlacementId();
        log("Loading native ad with slot id \"" + slotId + "\"...");

        makeSureInitWithSlotRequest(slotId, activity, () -> {
            NativeAdRequest nativeAdRequest = new NativeAdRequest.Builder().withSlotId(slotId).build();
            NativeAdLoader.Builder nativeAdLoaderBuilder = new NativeAdLoader.Builder()
                    .withAdLoadListener(new NativeAdListener(parameters, activity == null ? null : activity.getApplicationContext(), activity, maxNativeAdAdapterListener));
            if (!TextUtils.isEmpty(extString)) {
                nativeAdLoaderBuilder.withExt(extString);
            }
            NativeAdLoader nativeAdLoader = nativeAdLoaderBuilder.build();
            nativeAdLoader.loadAd(nativeAdRequest);
        });
    }

    @Override
    public void loadAdViewAd(MaxAdapterResponseParameters parameters, MaxAdFormat adFormat, Activity activity, MaxAdViewAdapterListener maxAdViewAdapterListener) {
        String bidResponse = parameters.getBidResponse();
        String slotId = parameters.getThirdPartyAdPlacementId();
        log("Loading " + (AppLovinSdkUtils.isValidString(bidResponse) ? "bidding " : "") + adFormat.getLabel() + " ad for slot id \"" + slotId + "\"...");

        makeSureInitWithSlotRequest(slotId, activity, () -> {
            AppLovinSdkUtils.Size adSize = adFormat.getSize();
            BannerAdRequest.Builder adRequestBuilder = new BannerAdRequest.Builder();
            /*if (AppLovinSdkUtils.isValidString(bidResponse)) {
                adRequestBuilder.withBid(bidResponse);
            }*/
            adRequestBuilder.withSlotId(slotId);
            int width = adSize.getWidth(); // dp
            int height = adSize.getHeight();
            adRequestBuilder.withAdSizes(fromBigoAdSize(width, height));

            // Implement a SampleAdListener and forward callbacks to mediation. The callback forwarding
            // is handled by SampleBannerEventFowarder.
            BannerAdLoader.Builder bannerAdLoaderBuilder = new BannerAdLoader.Builder()
                    .withAdLoadListener(new AdViewListener(slotId, adFormat, maxAdViewAdapterListener));
            if (!TextUtils.isEmpty(extString)) {
                bannerAdLoaderBuilder.withExt(extString);
            }
            BannerAdLoader bannerAdLoader = bannerAdLoaderBuilder.build();
            // Make an ad request.
            bannerAdLoader.loadAd(adRequestBuilder.build());
        });
    }
    //endregion

    //region Helper Methods
    private Boolean getPrivacySetting(final String privacySetting, @Nullable final MaxAdapterParameters parameters) {
        if (parameters == null) {
            return null;
        }
        try {
            // Use reflection because compiled adapters have trouble fetching `boolean` from old SDKs and `Boolean` from new SDKs (above 9.14.0)
            Class<?> parametersClass = parameters.getClass();
            Method privacyMethod = parametersClass.getMethod(privacySetting);
            return (Boolean) privacyMethod.invoke(parameters);
        } catch (Exception exception) {
            log("Error getting privacy setting " + privacySetting + " with exception: ", exception);
            return null;
        }
    }

    private AdSize fromBigoAdSize(int widthInDp, int heightInDp) {
        if (heightInDp >= AdSize.LARGE_RECTANGLE.getHeight()) {
            return AdSize.LARGE_RECTANGLE;
        } else if (heightInDp >= AdSize.MEDIUM_RECTANGLE.getHeight()) {
            return AdSize.MEDIUM_RECTANGLE;
        } else if (heightInDp >= AdSize.LARGE_BANNER.getHeight()) {
            return AdSize.LARGE_BANNER;
        } else {
            // Default to standard banner size
            return AdSize.BANNER;
        }
    }

    private static MaxAdapterError toMaxError(final int errorCode, final String bigoAdsErrorMessage) {
        MaxAdapterError adapterError = MaxAdapterError.UNSPECIFIED;
        switch (errorCode) {
            case AdError.ERROR_CODE_UNINITIALIZED:
                adapterError = MaxAdapterError.NOT_INITIALIZED;
                break;
            case AdError.ERROR_CODE_INVALID_REQUEST:
                adapterError = MaxAdapterError.BAD_REQUEST;
                break;
            case AdError.ERROR_CODE_NETWORK_ERROR:
                adapterError = MaxAdapterError.NO_CONNECTION;
                break;
            case AdError.ERROR_CODE_NO_FILL:
                adapterError = MaxAdapterError.NO_FILL;
                break;
            /* max 10.3.5 not support
            case AdError.ERROR_CODE_ASSETS_ERROR:
                adapterError = MaxAdapterError.MISSING_REQUIRED_NATIVE_AD_ASSETS;
                break;
             */
            case AdError.ERROR_CODE_AD_EXPIRED:
                adapterError = MaxAdapterError.AD_EXPIRED;
                break;
            case AdError.ERROR_CODE_INTERNAL_ERROR:
            case AdError.ERROR_CODE_AD_DISABLE:
            case AdError.ERROR_CODE_FULLSCREEN_AD_FAILED_TO_SHOW:
            case AdError.ERROR_CODE_VIDEO_ERROR:
            case AdError.ERROR_CODE_NATIVE_VIEW_MISSING:
                adapterError = MaxAdapterError.INTERNAL_ERROR;
                break;
        }
        return new MaxAdapterError(adapterError.getErrorCode(), adapterError.getErrorMessage(), errorCode, bigoAdsErrorMessage);
    }
    //endregion

    //region InterstitialAdListener
    private final class InterstitialAdListener implements AdLoadListener<InterstitialAd> {

        private final String                         slotId;
        private final MaxInterstitialAdapterListener listener;

        InterstitialAdListener(final String slotId, final MaxInterstitialAdapterListener listener) {
            this.slotId = slotId;
            this.listener = listener;
        }

        @Override
        public void onError(@NonNull AdError adError) {
            MaxAdapterError adapterError = toMaxError(adError.getCode(), adError.getMessage());
            log("Interstitial ad (" + slotId + ") failed to load with error: " + stringError(adapterError));
            listener.onInterstitialAdLoadFailed(adapterError);
        }

        @Override
        public void onAdLoaded(@NonNull InterstitialAd interstitialAd) {
            interstitialAd.setAdInteractionListener(new AdInteractionListener() {
                @Override
                public void onAdError(@NonNull AdError adError) {
                    onError(adError);
                }

                @Override
                public void onAdImpression() {
                    log("Bigo ads interstitial ad impression.");
                    listener.onInterstitialAdDisplayed();
                }

                @Override
                public void onAdClicked() {
                    log("Bigo ads interstitial ad clicked.");
                    listener.onInterstitialAdClicked();
                }

                @Override
                public void onAdOpened() {
                    log("Bigo ads interstitial ad opened.");
                }

                @Override
                public void onAdClosed() {
                    log("Bigo ads interstitial ad closed.");
                    listener.onInterstitialAdHidden();
                }
            });

            Bundle extraInfo = new Bundle(1);
            extraInfo.putString(KEY_CREATIVE_ID, interstitialAd.getCreativeId());
            BigoAdsMediationAdapter.this.interstitialAd = interstitialAd;
            log("Interstitial ad loaded: " + slotId);
            listener.onInterstitialAdLoaded(extraInfo);
        }
    }
    //endregion

    //region RewardedAdListener

    /**
     * The reward callback in Applovin adapter invoked when reward video dismiss.
     */
    private class RewardedAdListener implements AdLoadListener<RewardVideoAd> {
        private final String                     slotId;
        private final MaxRewardedAdapterListener listener;

        private boolean hasGrantedReward;

        RewardedAdListener(final String slotId, final MaxRewardedAdapterListener listener) {
            this.slotId = slotId;
            this.listener = listener;
        }

        @Override
        public void onError(@NonNull AdError adError) {
            MaxAdapterError adapterError = toMaxError(adError.getCode(), adError.getMessage());
            log("Rewarded ad (" + slotId + ") failed to load with error: " + stringError(adapterError));
            listener.onRewardedAdLoadFailed(adapterError);
        }

        @Override
        public void onAdLoaded(@NonNull RewardVideoAd rewardVideoAd) {

            rewardVideoAd.setAdInteractionListener(new RewardAdInteractionListener() {
                @Override
                public void onAdRewarded() {
                    log("Rewarded ad video completed: " + slotId);
                    listener.onRewardedAdVideoCompleted();

                    log("Rewarded user with reward.");
                    hasGrantedReward = true;
                }

                @Override
                public void onAdError(@NonNull AdError adError) {
                    MaxAdapterError adapterError = toMaxError(adError.getCode(), adError.getMessage());
                    log("Rewarded ad failed to display: " + slotId + ", error: " + stringError(adapterError));
                    listener.onRewardedAdDisplayFailed(adapterError);
                }

                @Override
                public void onAdImpression() {
                    log("Rewarded ad displayed: " + slotId);

                    listener.onRewardedAdDisplayed();
                    listener.onRewardedAdVideoStarted();
                }

                @Override
                public void onAdClicked() {
                    log("Rewarded ad clicked: " + slotId);
                    listener.onRewardedAdClicked();
                }

                @Override
                public void onAdOpened() {

                }

                @Override
                public void onAdClosed() {
                    log("Rewarded ad hidden: " + slotId);
                    if (hasGrantedReward || shouldAlwaysRewardUser()) {
                        final MaxReward reward = getReward();
                        log("Rewarded user with reward: " + reward);
                        listener.onUserRewarded(reward);
                    }

                    listener.onRewardedAdHidden();
                }
            });
            Bundle extraInfo = new Bundle(1);
            extraInfo.putString(KEY_CREATIVE_ID, rewardVideoAd.getCreativeId());
            BigoAdsMediationAdapter.this.rewardedAd = rewardVideoAd;

            log("Rewarded ad loaded: " + slotId);
            listener.onRewardedAdLoaded(extraInfo);
        }
    }
    //endregion

    //region AdViewListener
    private class AdViewListener implements AdLoadListener<BannerAd> {
        private final String                   slotId;
        private final MaxAdFormat              adFormat;
        private final MaxAdViewAdapterListener listener;

        AdViewListener(final String slotId, final MaxAdFormat adFormat, final MaxAdViewAdapterListener listener) {
            this.slotId = slotId;
            this.adFormat = adFormat;
            this.listener = listener;
        }

        @Override
        public void onError(@NonNull AdError adError) {
            MaxAdapterError adapterError = toMaxError(adError.getCode(), adError.getMessage());
            log(adFormat.getLabel() + " ad (" + slotId + ") failed to load with error: " + stringError(adapterError));
            listener.onAdViewAdLoadFailed(adapterError);
        }

        @Override
        public void onAdLoaded(@NonNull BannerAd bannerAd) {
            log(adFormat.getLabel() + " ad (" + slotId + ") loaded.");

            bannerAd.setAdInteractionListener(new AdInteractionListener() {
                @Override
                public void onAdError(@NonNull AdError adError) {
                    onError(adError);
                }

                @Override
                public void onAdImpression() {
                    log(adFormat.getLabel() + " ad shown: " + slotId);
                    listener.onAdViewAdDisplayed();
                }

                @Override
                public void onAdClicked() {

                    log(adFormat.getLabel() + " ad clicked: " + slotId);
                    listener.onAdViewAdClicked();
                }

                @Override
                public void onAdOpened() {

                }

                @Override
                public void onAdClosed() {

                }
            });
            Bundle extraInfo = new Bundle(1);
            extraInfo.putString(KEY_CREATIVE_ID, bannerAd.getCreativeId());
            // adapter to auto-refresh.
            BannerAd oldBannerAd = BigoAdsMediationAdapter.this.bannerAd;
            BigoAdsMediationAdapter.this.bannerAd = bannerAd;
            if (oldBannerAd != null && oldBannerAd != bannerAd) {
                oldBannerAd.destroy();
            }
            listener.onAdViewAdLoaded(bannerAd.adView(), extraInfo);
        }
    }
    //endregion

    //region NativeAdListener
    private class NativeAdListener implements AdLoadListener<NativeAd>, AdInteractionListener {
        final String                     slotId;
        final Bundle                     serverParameters;
        final Context                    applicationContext;
        final WeakReference<Activity>    activityRef;
        final MaxNativeAdAdapterListener listener;

        NativeAdListener(final MaxAdapterResponseParameters parameters,
                         final Context applicationContext,
                         final Activity activity,
                         final MaxNativeAdAdapterListener listener) {
            this.slotId = parameters.getThirdPartyAdPlacementId();
            this.serverParameters = parameters.getServerParameters();
            this.applicationContext = applicationContext;
            this.activityRef = new WeakReference<>(activity);
            this.listener = listener;
        }

        @Override
        public void onError(@NonNull AdError adError) {
            MaxAdapterError adapterError = toMaxError(adError.getCode(), adError.getMessage());
            log("Native ad (" + slotId + ") failed to load with error: " + stringError(adapterError));
            listener.onNativeAdLoadFailed(adapterError);
        }

        @Override
        public void onAdLoaded(@NonNull NativeAd nativeAd) {
            Context context = activityRef.get();
            if (context == null) {
                context = applicationContext;
            }
            nativeAd.setAdInteractionListener(this);
            log("Native ad loaded: " + slotId + ". Preparing assets...");
            final Bundle extraInfo = new Bundle(1);
            extraInfo.putString(KEY_CREATIVE_ID, nativeAd.getCreativeId());
            BigoAdsMediationAdapter.this.nativeAd = nativeAd;

            // Create MaxNativeAd after images are loaded from remote URLs
            Context finalContext = context;
            AppLovinSdkUtils.runOnUiThread(new Runnable() {
                @Override
                public void run() {
                    MediaView mediaView = null;
                    AdOptionsView adOptionsView = null;
                    if (finalContext != null) {
                        mediaView = new MediaView(finalContext);
                        adOptionsView = new AdOptionsView(finalContext);
                    }

                    log("Creating native ad with assets");

                    MaxNativeAd.Builder builder = new MaxNativeAd.Builder()
                            .setAdFormat(MaxAdFormat.NATIVE)
                            .setTitle(nativeAd.getTitle())
                            .setBody(nativeAd.getDescription())
                            .setCallToAction(nativeAd.getCallToAction());
                    // .setIcon(icon)
                    if (mediaView != null) {
                        builder.setMediaView(mediaView);
                    }
                    if (adOptionsView != null) {
                        builder.setOptionsView(adOptionsView);
                    }
                    MaxNativeAd maxNativeAd = new MaxBigoNativeAd(builder);

                    log("Native ad fully loaded: " + slotId);
                    listener.onNativeAdLoaded(maxNativeAd, extraInfo);
                }
            });
        }

        @Override
        public void onAdError(@NonNull AdError adError) {
            onError(adError);
        }

        @Override
        public void onAdImpression() {
            log("Native ad displayed: " + slotId);
            listener.onNativeAdDisplayed(null);
        }

        @Override
        public void onAdClicked() {
            // This callback is never called
            log("Native ad clicked: " + slotId);
            listener.onNativeAdClicked();
        }

        @Override
        public void onAdOpened() {

        }

        @Override
        public void onAdClosed() {

        }
    }
    //endregion

    //region MaxBigoNativeAd
    private class MaxBigoNativeAd
            extends MaxNativeAd {
        public MaxBigoNativeAd(final Builder builder) {
            super(builder);
        }

        @Override
        public void prepareViewForInteraction(final MaxNativeAdView maxNativeAdView) {
            NativeAd nativeAd = BigoAdsMediationAdapter.this.nativeAd;
            if (nativeAd == null) {
                e("Failed to register native ad view due to empty native ad.");
                return;
            }
            if (maxNativeAdView == null) {
                e("Failed to register native ad view due to empty ad view.");
                return;
            }

            List<View> clickableViews = new ArrayList<>();
            if (AppLovinSdkUtils.isValidString(getTitle()) && maxNativeAdView.getTitleTextView() != null) {
                maxNativeAdView.getTitleTextView().setTag(AdTag.TITLE);
                clickableViews.add(maxNativeAdView.getTitleTextView());
            }
            if (AppLovinSdkUtils.isValidString(getBody()) && maxNativeAdView.getBodyTextView() != null) {
                maxNativeAdView.getBodyTextView().setTag(AdTag.DESCRIPTION);
                clickableViews.add(maxNativeAdView.getBodyTextView());
            }
            if (getIcon() != null && maxNativeAdView.getIconImageView() != null) {
                clickableViews.add(maxNativeAdView.getIconImageView());
            }
            if (getMediaView() != null && maxNativeAdView.getMediaContentViewGroup() != null) {
                clickableViews.add(maxNativeAdView.getMediaContentViewGroup());
            }
            if (maxNativeAdView.getCallToActionButton() != null) {
                maxNativeAdView.getCallToActionButton().setTag(AdTag.CALL_TO_ACTION);
                clickableViews.add(maxNativeAdView.getCallToActionButton());
            }

            ViewGroup mediaContentViewGroup = maxNativeAdView.getMediaContentViewGroup();
            ViewGroup optionsContentViewGroup = maxNativeAdView.getOptionsContentViewGroup();

            View mediaViewObject = getMediaView();
            View optionsViewObject = getOptionsView();
            MediaView mediaView = mediaViewObject instanceof MediaView ? (MediaView) mediaViewObject : null;
            AdOptionsView adOptionsView = optionsViewObject instanceof AdOptionsView ? (AdOptionsView) optionsViewObject : null;
            if (mediaView != null) {
                if (mediaContentViewGroup == null) {
                    mediaView = null; // Do not use media view if media content view group is null.
                    e("Warn: media view is null, can not render media.");
                } else { // mediaContentViewGroup != null && mediaView != null.
                    if (mediaView.getParent() == null) { // make sure media view in mediaContentViewGroup.
                        mediaContentViewGroup.removeAllViews();
                        mediaContentViewGroup.addView(mediaView);
                    }
                }
            }
            if (adOptionsView != null) {
                if (optionsContentViewGroup == null) {
                    adOptionsView = null; // Do not use options view if options content view group is null.
                    e("Warn: options view is null, can not render options view.");
                } else { // optionsContentViewGroup != null && adOptionsView != null.
                    if (adOptionsView.getParent() == null) {
                        optionsContentViewGroup.removeAllViews();
                        optionsContentViewGroup.addView(adOptionsView);
                    }
                }
            }
            nativeAd.registerViewForInteraction(maxNativeAdView, mediaView, maxNativeAdView.getIconImageView(), adOptionsView, clickableViews);
        }
    }
    //endregion

    //region public method
    private String stringError(MaxAdapterError error) {
        if (error == null) return null;

        int mediatedNetworkErrorCode = 0;
        String mediatedNetworkErrorMessage = "";
        try {
            mediatedNetworkErrorCode = error.getMediatedNetworkErrorCode();
            mediatedNetworkErrorMessage = error.getMediatedNetworkErrorMessage();
        } catch (Throwable t) { // May cause java.lang.NoSuchMethodError due to version unmatched
            t.printStackTrace();
        }
        return "MaxError{code=" + error.getCode() + ", message=\"" + error.getMessage() + "\"" + ", mediatedNetworkErrorCode=" + mediatedNetworkErrorCode +
                ", mediatedNetworkErrorMessage=\"" + mediatedNetworkErrorMessage + "\"" +
                "}";
    }
    //endregion
}
