package com.hyprmx.android.sdk.activity

import android.os.Bundle
import android.view.View
import android.view.ViewGroup
import android.widget.RelativeLayout
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.LifecycleObserver
import com.hyprmx.android.R
import com.hyprmx.android.sdk.analytics.EventController
import com.hyprmx.android.sdk.api.data.WebTrafficAd
import com.hyprmx.android.sdk.footer.FooterContract
import com.hyprmx.android.sdk.footer.FooterFragment
import com.hyprmx.android.sdk.footer.FooterPresenter
import com.hyprmx.android.sdk.fullscreen.FullScreenSharedInterface
import com.hyprmx.android.sdk.header.WebTrafficHeaderContract
import com.hyprmx.android.sdk.header.WebTrafficHeaderFragment
import com.hyprmx.android.sdk.header.WebTrafficHeaderPresenter
import com.hyprmx.android.sdk.network.NetworkConnectionMonitor
import com.hyprmx.android.sdk.om.OpenMeasurementController
import com.hyprmx.android.sdk.powersavemode.PowerSaveModeListener
import com.hyprmx.android.sdk.presentation.ActivityResultListener
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.ImageCacheManagerIf
import com.hyprmx.android.sdk.utility.InternetConnectionDialog
import com.hyprmx.android.sdk.webview.HyprMXWebView
import com.hyprmx.android.sdk.webview.WebViewFactoryIf
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import java.util.Locale

/**
 * HyprMXWebTrafficViewController is responsible for displaying web-traffic offers
 */
internal class HyprMXWebTrafficViewController(
  activity: AppCompatActivity,
  savedInstanceState: Bundle?,
  val distributorId: String,
  private val ad: WebTrafficAd,
  viewControllerListener: HyprMXBaseViewControllerListener,
  val eventController: EventController,
  val imageCacheManager: ImageCacheManagerIf,
  webViewFactory: WebViewFactoryIf,
  activityResultListener: ActivityResultListener,
  openMeasurementController: OpenMeasurementController?,
  powerSaveMode: PowerSaveModeListener,
  scope: CoroutineScope,
  networkConnectionMonitor: NetworkConnectionMonitor,
  internetConnectionDialog: InternetConnectionDialog,
  eventPublisher: PresentationEventPublisher,
  fullScreenSharedConnector: FullScreenSharedInterface,
) :
  HyprMXBaseViewController(
    activity,
    savedInstanceState = savedInstanceState,
    hyprMXBaseViewControllerListener = viewControllerListener,
    activityResultListener = activityResultListener,
    powerSaveMode = powerSaveMode,
    webViewFactory = webViewFactory,
    openMeasurementController = openMeasurementController,
    baseAd = ad,
    scope = scope,
    networkConnectionMonitor = networkConnectionMonitor,
    internetConnectionDialog = internetConnectionDialog,
    eventPublisher = eventPublisher,
    fullScreenSharedConnector = fullScreenSharedConnector,
  ),
  FooterContract.URLPresenter,
  FooterContract.NavigationPresenter,
  WebTrafficHeaderContract.NavigationPresenter,
  LifecycleObserver {

  private var delayWTForOM = false
  private lateinit var footerFragment: FooterFragment
  private lateinit var footerPresenter: FooterContract.Presenter
  private lateinit var webTrafficHeaderFragment: WebTrafficHeaderFragment
  private lateinit var webTrafficHeaderPresenter: WebTrafficHeaderContract.Presenter
  private lateinit var webTrafficContainer: RelativeLayout
  private lateinit var offerContainer: RelativeLayout
  private lateinit var fullScreenVideoContainer: RelativeLayout

  private lateinit var webTrafficLayout: RelativeLayout

  private var countDownRunnable: Job? = null

  private var timeLeft: Int = 0

  /**
   * Attempts to restore the webtraffic offer if the view has been recreated.
   * Either load the thank you page if available or exit with ad cancel.
   */
  override fun restoreState(savedInstanceState: Bundle) {
    super.restoreState(savedInstanceState)
    restoreState()
  }

  private fun setupWebTrafficPresenters() {
    footerFragment = activity.supportFragmentManager
      .findFragmentById(R.id.hyprmx_footer_fragment) as FooterFragment
    webTrafficHeaderFragment = activity.supportFragmentManager
      .findFragmentById(R.id.header_fragment) as WebTrafficHeaderFragment

    footerPresenter = FooterPresenter(
      this,
      this,
      ad.footer,
      footerFragment,
      true,
      imageCacheManager = imageCacheManager,
    )
    webTrafficHeaderPresenter = WebTrafficHeaderPresenter(
      ad.webTrafficHeader,
      webTrafficHeaderFragment,
      closable,
      this,
    )
  }

  override fun getOfferRootLayout(): ViewGroup {
    return webTrafficContainer
  }

  override fun onBackPressed() {
    when {
      webView.canGoBack() && !payoutComplete -> {
        webView.navigateBack()
      }
      else -> {
        onBackButtonPressed()
      }
    }
  }

  override fun onDestroy() {
    if (webView.parent != null) {
      webTrafficContainer.removeView(webView)
    }

    super.onDestroy()
  }

  override fun executeJS(script: String) {
    webView.loadUrl("javascript:$script")
  }

  override suspend fun setForwardButtonEnabled(enabled: Boolean): Unit =
    withContext(Dispatchers.Main) {
      HyprMXLog.d("Updating forward navigation to ($enabled)")
      footerPresenter.enableForwardNavigation(enabled)
    }

  override suspend fun setBackButtonEnabled(enabled: Boolean): Unit =
    withContext(Dispatchers.Main) {
      HyprMXLog.d("Updating back navigation to ($enabled)")
      footerPresenter.enableBackwardNavigation(enabled)
    }

  override fun didTapIcon(index: Int) {
    HyprMXLog.d("did tap icon at index $index")
    onNavBarButtonTapped(index)
  }

  override suspend fun loadThankYouPage(url: String): Unit = withContext(Dispatchers.Main) {
    webView.stopLoading()
    webTrafficHeaderPresenter.showCloseButtonOnly()
    insertNewWebView()
    webView.loadUrl(url)
  }

  override suspend fun setClosable(closable: Boolean): Unit = withContext(Dispatchers.Main) {
    super.setClosable(closable)
    if (closable) {
      webTrafficHeaderPresenter.showCloseButtonOnly()
    }
  }

  /**
   * Setup layout view programmatically
   */
  override fun setupLayout() {
    super.setupLayout()
    // setup LinearLayout for 2 RelativeLayout
    val inflater = activity.layoutInflater
    val webTrafficRootLayout =
      inflater.inflate(R.layout.hyprmx_web_traffic, rootLayout, true)
    webTrafficLayout = webTrafficRootLayout.findViewById(R.id.hyprmx_webtraffic)
    webTrafficContainer = webTrafficLayout.findViewById(R.id.webtraffic_container)

    val stub: View = webTrafficContainer.findViewById(R.id.webview_stub)
    val layoutParams = stub.layoutParams

    webTrafficContainer.removeView(stub)
    webTrafficContainer.addView(webView, layoutParams)

    offerContainer = webTrafficLayout.findViewById(R.id.offer_container)
    fullScreenVideoContainer = webTrafficLayout.findViewById(R.id.fullScreenVideoContainer)

    setupWebTrafficPresenters()
  }

  override suspend fun loadWebTrafficPage(
    url: String,
    index: Int,
    userAgent: String?,
  ): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("Open Web Page: $index")

    webTrafficHeaderPresenter.setPageVisited(index)

    insertNewWebView()

    userAgent?.let {
      webView.setUserAgent(it)
    }

    webView.loadUrl(url)

    if (webView.shouldTakeFocus()) {
      webView.requestFocus()
    }

    // Show the progress bar
    webTrafficHeaderPresenter.showProgressSpinner()

    if (ad.footer.navOnly) {
      footerPresenter.setVisible(false)
    }
  }

  /**
   * Activates countdown timer for webtraffic ad.
   */
  override suspend fun startCountDownTimer(time: Int): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("Starting Countdown Timer ($time)")

    // only launch timer if it hasn't been launched yet, or the previous timer is completed
    if (countDownRunnable == null || countDownRunnable?.isCompleted == true) {
      HyprMXLog.d("Starting count down timer")
      timeLeft = time
      countDownRunnable = launch {
        while (timeLeft > 0) {
          delay(1000)
          if (!isActive) {
            return@launch
          }
          timeLeft--
          if (timeLeft <= 0) {
            HyprMXLog.d("CountDownTimer fired!")
            webTrafficPageMinTimeComplete()
          } else {
            HyprMXLog.d("updateTimerView")
            updateTimerView()
          }
        }
      }
    }
  }

  private fun updateTimerView() {
    val b = StringBuilder()
    var time = timeLeft
    val seconds = time % 60
    time = (time - seconds) / 60
    val minutes = time % 60
    time = (time - minutes) / 60
    val hours = time

    when {
      hours > 0 -> b.append(
        String.format(Locale.US, "%d:%2d:%02d", hours, minutes, seconds),
      )
      minutes > 0 -> b.append(String.format(Locale.US, "%d:%02d", minutes, seconds))
      else -> b.append(String.format(Locale.US, ":%02d", seconds))
    }
    webTrafficHeaderPresenter.showCountDown(b.toString())
  }

  override suspend fun showNextButton(): Unit = withContext(Dispatchers.Main) {
    webTrafficHeaderPresenter.showNextButton()
  }

  override suspend fun showFinishButton(): Unit = withContext(Dispatchers.Main) {
    webTrafficHeaderPresenter.showFinishButton()
  }

  override fun didTapForward() {
    webView.navigateForward()
  }

  override fun didTapBack() {
    webView.navigateBack()
  }

  override fun didTapURL(url: String) {
    HyprMXLog.d("did tap url $url")
    windowOpenAttempt(url)
  }

  override fun didTapNext() {
    webTrafficNextTapped()
  }

  override fun didTapClose() {
    closeTapped()
  }

  override fun didTapFinish() {
    webTrafficFinishTapped()
  }

  /**
   * Displays network error dialog with OK button.
   * Calls page ready timeout to be removed.
   */
  override fun showNetworkErrorDialog() {
    HyprMXLog.d("Show network error dialog.")
    webView.loadUrl("about:blank") // load blank page to stop execution of javascript
    super.showNetworkErrorDialog()
  }

  override suspend fun startOMSession(sessionData: String): Unit = withContext(Dispatchers.Main) {
    super.startOMSession(sessionData)
    delayWTForOM = true
  }

  override suspend fun showWebTrafficHeader(
    pageCount: Int,
  ): Unit = withContext(Dispatchers.Main) {
    webTrafficHeaderPresenter.showWithTitleAndPageCount(pageCount)
  }

  private suspend fun insertNewWebView() {
    val layoutParams = webView.layoutParams
    webTrafficContainer.removeView(webView)
    if (delayWTForOM) {
      delayWTForOM = false
      val omWebView = webView
      launch {
        omWebView.apply {
          HyprMXLog.d("Delaying destroying webview for 1s")
          delay(1_000)
          stopLoading()
          cleanup()
        }
      }
    } else {
      webView.stopLoading()
      webView.cleanup()
    }

    webView = HyprMXWebView(context).apply {
      initialize(
        viewModelIdentifier,
        this@HyprMXWebTrafficViewController.baseAd.userAgent,
      )

      containingActivity = activity
    }
    webTrafficContainer.addView(webView, layoutParams)
  }

  override suspend fun pauseCountDownTimer(): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("Pausing Countdown Timer ($timeLeft)")
    countDownRunnable?.cancel()
  }

  override suspend fun resumeCountDownTimer(): Unit = withContext(Dispatchers.Main) {
    HyprMXLog.d("Resuming Countdown Timer ($timeLeft)")
    startCountDownTimer(timeLeft)
  }
}
