package com.hyprmx.android.sdk.analytics

import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.network.NetworkController
import com.hyprmx.android.sdk.utility.HyprMXErrorType
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.HyprMXProperties
import com.hyprmx.android.sdk.utility.isValidUrl
import org.json.JSONObject

/**
 * This class sets the endpoints for tracking, handles making the tracking requests
 * and returning requested params to shared js.
 */
internal class DefaultEventController(
  private val jsEngine: JSEngine,
  private val clientErrorController: ClientErrorControllerIf,
  private val networkController: NetworkController,
) : EventController {

  companion object {
    const val VIEWING = "viewings"
    const val SHARINGS_CREATE = "sharings"
    const val OFFER_COMPLETE = "offer_completion/complete"
  }

  private var sharingUrl: String

  init {
    sharingUrl = "${HyprMXProperties.baseUrl}/$SHARINGS_CREATE"
  }

  override fun setSharingEndpoint(sharingEndpoint: String) {
    if (sharingEndpoint.isValidUrl()) {
      sharingUrl = sharingEndpoint
    } else {
      handleInvalidEndPoint(sharingEndpoint)
    }
  }

  override fun handleInvalidEndPoint(completionEndpoint: String) {
    val errorLogMsg = "Invalid Endpoint: $completionEndpoint"
    clientErrorController.sendClientError(
      HyprMXErrorType.HYPRErrorInvalidEndpoint,
      errorLogMsg,
      ClientErrorControllerIf.SEVERITY_4,
    )
  }

  /**
   * Fires a tracking url for each webview displayed in HyprmxWebTrafficActivity
   *
   * @param url
   * @param viewingId
   */
  override fun sendTrackWebViewImpression(url: String, viewingId: String) {
    jsEngine.evaluate("HYPREventController.sendWebTrafficVisitEvent('$url', '$viewingId')")
  }

  override fun sendPixelTrackingEvents(imageTagsToFire: String) {
    jsEngine.evaluate("HYPREventController.sendPixelTrackingEvents('$imageTagsToFire')")
  }

  override suspend fun sendSharingTracking(viewingId: String?) {
    val json = JSONObject()
    try {
      if (viewingId != null) {
        json.put("viewing_id", viewingId)
      }
      json.put("bypass_redirect", "0")
      json.put("sharing_category", "cta")

      networkController.postRequest(sharingUrl, json.toString())
    } catch (e: Exception) {
      HyprMXLog.e("Error sending sharing tracking: ${e.message}")
    }
  }

  /**
   * Sends duration tracking.
   *
   * @param duration position of video playback
   */
  override fun sendDurationUpdateTracking(
    token: String,
    viewingId: String,
    duration: String,
  ) {
    jsEngine.evaluate("HYPREventController.sendDurationUpdateTracking('$token', '$duration', '$viewingId')")
  }
}
