package com.hyprmx.android.sdk.analytics

import android.Manifest
import android.content.Context
import android.content.pm.PackageManager
import android.graphics.Insets
import android.os.Build
import android.provider.Settings
import android.security.NetworkSecurityPolicy
import android.telephony.TelephonyManager
import android.text.TextUtils
import android.util.DisplayMetrics
import android.view.WindowInsets
import android.view.WindowManager
import android.view.WindowMetrics
import androidx.core.content.ContextCompat
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.powersavemode.PowerSaveModeListener
import com.hyprmx.android.sdk.utility.FetchGAIDImpl
import com.hyprmx.android.sdk.utility.FetchGaid
import com.hyprmx.android.sdk.utility.GaidResult
import com.hyprmx.android.sdk.utility.HyprMXConnection
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.convertPixelsToDp
import com.hyprmx.android.sdk.utility.getPermissionsListedInAndroidManifest
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.json.JSONArray
import org.json.JSONObject
import kotlin.math.floor

internal interface ParameterController : ParamsNativeInterface {
  suspend fun initialize()
}

internal class ParameterControllerImpl(
  private val context: Context,
  private val jsEngine: JSEngine,
  private val powerSaveModeListener: PowerSaveModeListener,
  private val connectionInfo: HyprMXConnection,
  private val gaidController: FetchGaid = FetchGAIDImpl,
  private val scope: CoroutineScope,
) : ParameterController {

  companion object {
    const val JSNAME = "HYPRRequestParamListener"
    const val PARAM_LOW_POWER_ENABLED = "low_power_enabled"
    private const val JSCONTROLLER = "HYPRRequestParameterManager"

    const val PARAM_UNITY_VERSION = "unity_version"
    const val PARAM_MEDIATOR = "mediator"
    const val PARAM_MEDIATOR_NAME = "name"
    const val PARAM_MEDIATOR_SDK_VERSION = "sdk_version"
    const val PARAM_MEDIATOR_ADAPTER_VERSION = "adapter_version"

    const val PARAM_CARRIER_DATA_NAME = "carrier_name"
    const val PARAM_CARRIER_DATA_MCC = "mobile_country_code"
    const val PARAM_CARRIER_DATA_MNC = "mobile_network_code"

    const val PLATFORM = "android"

    @Suppress("ClassName")
    enum class PERMISSION_STATE(val description: String) {
      GRANTED("authorized"),
      DENIED("denied"),
    }

    @Suppress("ClassName")
    enum class USER_PERMISSION(val permission: String, val permissionKey: String) {
      CAMERA(Manifest.permission.CAMERA, "camera_permission"),
      CALENDAR(Manifest.permission.WRITE_CALENDAR, "calendar_permission"),
      MICROPHONE(Manifest.permission.RECORD_AUDIO, "microphone_permission"),
    }
  }

  init {
    jsEngine.addJavascriptInterface(this, JSNAME)
  }

  override suspend fun initialize() {
    jsEngine.insertGlobal(JSCONTROLLER, "new RequestParameterManager({})")
    initGaid(context)
  }

  @Volatile
  private var googleAdId: String? = null

  @Volatile
  private var optedOutGaid: Boolean = false

  /**
   * Attempts to get Google's advertising ID in separate thread,
   *
   * @param context Context of application
   */
  private suspend fun initGaid(context: Context) = withContext(Dispatchers.Main) {
    when (val result = gaidController.retrieveGaid(context)) {
      is GaidResult.Success -> {
        googleAdId = result.googleAdId
        optedOutGaid = result.limitAdTracking
      }
      is GaidResult.Failure -> HyprMXLog.d("gaid fetched failed")
    }
  }

  override fun getUnityParams(): String {
    val jsonObject = JSONObject()
    hyprmxDelegate.hyprMXMediation.unityVersion?.let { jsonObject.put(PARAM_UNITY_VERSION, it) }
    return jsonObject.toString()
  }

  override fun getMediationParams(): String {
    val jsonObject = JSONObject()
    if (hyprmxDelegate.hyprMXMediation.mediator != null ||
      hyprmxDelegate.hyprMXMediation.mediatorSDKVersion != null ||
      hyprmxDelegate.hyprMXMediation.mediatorAdapterVersion != null
    ) {
      val mediator = JSONObject()
      hyprmxDelegate.hyprMXMediation.mediator?.let { mediator.put(PARAM_MEDIATOR_NAME, it) }
      hyprmxDelegate.hyprMXMediation.mediatorSDKVersion?.let { sdk -> mediator.put(PARAM_MEDIATOR_SDK_VERSION, sdk) }
      hyprmxDelegate.hyprMXMediation.mediatorAdapterVersion?.let { version ->
        mediator.put(
          PARAM_MEDIATOR_ADAPTER_VERSION,
          version,
        )
      }
      jsonObject.put(PARAM_MEDIATOR, mediator)
    }
    return jsonObject.toString()
  }

  override fun getMraidSupportsString(): String = """
    {"sms":true,"storePicture":true,"inlineVideo":true,"calendar":true,"tel":true}
  """.trimIndent()

  override fun getScreenSize(): String {
    return getMaxFrameSize()
  }

  override fun getMaxFrameSize(): String {
    val displayMetrics = context.resources.displayMetrics
    val screenWidth = floor(displayMetrics.widthPixels.convertPixelsToDp(context)).toInt()
    val screenHeight = floor(displayMetrics.heightPixels.convertPixelsToDp(context)).toInt()
    val obj = JSONObject()
    obj.put("width", screenWidth)
    obj.put("height", screenHeight)
    return obj.toString()
  }

  override fun getConnectionType(): String = connectionInfo.getConnectionTypeForRequest()

  override fun getBundleID(): String = context.packageName

  override fun getBundleVersion(): String {
    return try {
      if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) {
        context.packageManager.getPackageInfo(context.packageName, PackageManager.PackageInfoFlags.of(0)).versionName
      } else {
        context.packageManager.getPackageInfo(context.packageName, PackageManager.GET_ACTIVITIES).versionName
      }
    } catch (exception: java.lang.RuntimeException) {
      "Unknown"
    }
  }

  override fun getClearTextPermitted(): Boolean =
    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
      NetworkSecurityPolicy.getInstance().isCleartextTrafficPermitted
    } else {
      true
    }

  override fun getAdIdOptedOut(): Boolean {
    return optedOutGaid
  }

  override fun getPersistentID(): String? {
    return googleAdId
  }

  override fun getGAID(): String? {
    return googleAdId
  }

  override fun getIsAgeRestrictedUser(): Boolean {
    return hyprmxDelegate.isAgeRestrictedUser
  }

  override fun isTestModeEnabled(): Boolean {
    return hyprmxDelegate.testModeEnabled
  }

  override fun getUserExtra(key: String): String? {
    return hyprmxDelegate.userExtras[key]
  }

  override fun getUserExtras(): String {
    return JSONObject(hyprmxDelegate.userExtras.toMap()).toString()
  }

  override fun getUserPermissions(): String {
    val userPermissions = JSONObject()
    try {
      for (perm in USER_PERMISSION.values()) {
        val state = if (ContextCompat.checkSelfPermission(
            context,
            perm.permission,
          ) == PackageManager.PERMISSION_GRANTED
        ) {
          PERMISSION_STATE.GRANTED
        } else {
          PERMISSION_STATE.DENIED
        }
        userPermissions.put(perm.permissionKey, state.description)
      }
    } catch (e: PackageManager.NameNotFoundException) {
      HyprMXLog.d("Unable to get list of permissions from Android Manifest")
    }
    return userPermissions.toString()
  }

  override fun getPermissions(): String {
    return try {
      JSONArray(getPermissionsListedInAndroidManifest(context))
    } catch (e: PackageManager.NameNotFoundException) {
      HyprMXLog.d("Unable to get list of permissions from Android Manifest")
      JSONArray()
    }.toString()
  }

  override fun getCarriers(): String {
    val telephonyManager = context.getSystemService(Context.TELEPHONY_SERVICE) as TelephonyManager
    val carriers = JSONObject()
    if (telephonyManager.simState == TelephonyManager.SIM_STATE_READY) {
      // Add carrier information to body
      val carrierName = telephonyManager.networkOperatorName
      val networkOperator = telephonyManager.networkOperator

      var mcc: String? = null
      var mnc: String? = null
      if (!TextUtils.isEmpty(networkOperator)) {
        mcc = networkOperator.substring(0, 3)
        mnc = networkOperator.substring(3)
      }

      val carrierData = JSONObject()

      carrierData.put(PARAM_CARRIER_DATA_NAME, carrierName)
      carrierData.put(PARAM_CARRIER_DATA_MCC, mcc)
      carrierData.put(PARAM_CARRIER_DATA_MNC, mnc)
      carriers.put("0", carrierData)
    }
    return carriers.toString()
  }

  override fun getAndroidId(): String =
    Settings.Secure.getString(context.contentResolver, Settings.Secure.ANDROID_ID) ?: ""

  override fun getTargetSDKVersion(): Float = context.applicationContext.applicationInfo.targetSdkVersion.toFloat()

  override fun getDeviceWidth(): Int {
    val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
      val windowMetrics: WindowMetrics = windowManager.currentWindowMetrics
      val insets: Insets = windowMetrics.windowInsets
        .getInsetsIgnoringVisibility(WindowInsets.Type.systemBars())
      windowMetrics.bounds.width() - insets.left - insets.right
    } else {
      val displayMetrics = DisplayMetrics()
      windowManager.defaultDisplay.getMetrics(displayMetrics)
      displayMetrics.widthPixels
    }
  }

  override fun getDeviceHeight(): Int {
    val windowManager = context.getSystemService(Context.WINDOW_SERVICE) as WindowManager
    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.R) {
      val windowMetrics: WindowMetrics = windowManager.currentWindowMetrics
      val insets: Insets = windowMetrics.windowInsets
        .getInsetsIgnoringVisibility(WindowInsets.Type.systemBars())
      windowMetrics.bounds.width() - insets.left - insets.right
      windowMetrics.bounds.height() - insets.top - insets.bottom
    } else {
      val displayMetrics = DisplayMetrics()
      windowManager.defaultDisplay.getMetrics(displayMetrics)
      displayMetrics.heightPixels
    }
  }

  override fun getPxRatio(): Float = context.resources.displayMetrics.density

  override fun getIsLowPowerEnabled(): Boolean = powerSaveModeListener.isPowerSaveMode
}
