package com.hyprmx.android.sdk.api.data

import com.hyprmx.android.sdk.analytics.ClientErrorControllerIf
import com.hyprmx.android.sdk.footer.Footer
import com.hyprmx.android.sdk.header.WebTrafficHeader
import com.hyprmx.android.sdk.utility.HyprMXErrorType
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.getStringOrNull
import org.json.JSONException
import org.json.JSONObject
import java.io.Serializable
import com.hyprmx.android.sdk.utility.Result as HyprMXResult

/**
 * Represents an offer as returned from the HyprMX `offers_available`
 * endpoint.
 *
 * @version $Revision: 1.0 $
 */
internal interface Ad : Serializable {
  val type: String
  val allowedOrientation: Orientation
  val userAgent: String?

  companion object {

    const val PLAYER = "embedded_offers/player"

    fun fromJson(
      jsonString: String,
      clientErrorController: ClientErrorControllerIf,
    ): HyprMXResult<Ad> {
      try {
        val json = JSONObject(jsonString)
        val type = json.getString(FIELD_TYPE)

        val allowedOrientation =
          Orientation.getOrientation(json.optString(FIELD_ALLOWED_ORIENTATION, ""))

        val userAgent = json.getStringOrNull("user_agent")

        val baseAd = DefaultAd(
          type = type,
          allowedOrientation = allowedOrientation,
          userAgent = userAgent,
        )

        val ad: Ad = when (type) {
          WEBTRAFFIC_OFFER_TYPE -> {
            val webTrafficHeader =
              WebTrafficHeader.fromJson(json.optJSONObject(FIELD_HEADER))
            val footer = Footer.fromJson(json.optJSONObject(FIELD_FOOTER))
            if (footer.navOnly) {
              footer.bgColor = webTrafficHeader.bgColor
            }

            WebTrafficAd(
              ad = baseAd,
              webTrafficHeader = webTrafficHeader,
              footer = footer,
            )
          }
          MRAID_OFFER_TYPE -> {
            MraidAd(
              ad = baseAd,
            )
          }
          else -> baseAd
        }
        return HyprMXResult.Success(ad)
      } catch (e: RuntimeException) {
        HyprMXLog.e("Exception was thrown parsing Ad: ", e)
        clientErrorController.sendClientError(
          HyprMXErrorType.HYPRErrorTypeJSONParsingFailure,
          e.message!!,
          ClientErrorControllerIf.SEVERITY_2,
        )
        return HyprMXResult.Failure(
          "Runtime exception thrown while parsing ad",
          0,
          e,
        )
      } catch (e: JSONException) {
        HyprMXLog.e("Exception was thrown parsing Ad: ", e)
        clientErrorController.sendClientError(
          HyprMXErrorType.HYPRErrorTypeJSONParsingFailure,
          e.message!!,
          ClientErrorControllerIf.SEVERITY_2,
        )
        return HyprMXResult.Failure("JSON Exception thrown while parsing ad", 0, e)
      }
    }

    private const val FIELD_TYPE = "type"
    private const val FIELD_ALLOWED_ORIENTATION = "allowed_orientation"

    private const val FIELD_FOOTER = "footer"
    private const val FIELD_HEADER = "web_traffic_header"

    const val WEBTRAFFIC_OFFER_TYPE = "web_traffic"
    private const val MRAID_OFFER_TYPE = "mraid"
  }
}

enum class Orientation(val key: String) {

  PORTRAIT("portrait"), LANDSCAPE("landscape"), ANY("");

  companion object {
    fun getOrientation(key: String): Orientation {
      if (key == PORTRAIT.key) return PORTRAIT
      if (key == LANDSCAPE.key) return LANDSCAPE
      return ANY
    }
  }
}

internal class DefaultAd(
  override val type: String,
  override val allowedOrientation: Orientation,
  override val userAgent: String?,
) : Ad

internal class WebTrafficAd(
  ad: Ad,
  val webTrafficHeader: WebTrafficHeader,
  val footer: Footer,
) : Ad by ad

internal class MraidAd(
  ad: Ad,
) : Ad by ad
