package com.hyprmx.android.sdk.api.data

import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.Result
import com.hyprmx.android.sdk.utility.getStringOrNull
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject

internal class CalendarEvent(
  val id: String?,
  val description: String?,
  val location: String?,
  val summary: String?,
  val start: String?,
  val end: String?,
  val status: String?,
  val transparency: String?,
  val recurrence: CalendarRepeatRule?,
) {

  companion object {
    fun fromJson(jsonString: String): Result<CalendarEvent> {
      return try {
        val json = JSONObject(jsonString)
        val id = json.getStringOrNull("id")
        val description = json.getStringOrNull("description")
        val location = json.getStringOrNull("location")
        val summary = json.getStringOrNull("summary")
        val start = json.getStringOrNull("start")
        val end = json.getStringOrNull("end")
        val status = json.getStringOrNull("status")
        val transparency = json.getStringOrNull("transparency")
        val recurrence = if (json.has("recurrence")) {
          val repeatRuleResult = CalendarRepeatRule.fromJson(json.getStringOrNull("recurrence"))
          if (repeatRuleResult is Result.Failure) {
            return Result.Failure("Exception parsing repeat rule.", 0)
          } else {
            (repeatRuleResult as Result.Success).value
          }
        } else {
          null
        }
        Result.Success(
          CalendarEvent(
            id,
            description,
            location,
            summary,
            start,
            end,
            status,
            transparency,
            recurrence,
          ),
        )
      } catch (exception: JSONException) {
        Result.Failure("Exception parsing calendar event.", 0, exception)
      }
    }
  }
}

internal class CalendarRepeatRule internal constructor(
  val frequency: String?,
  val interval: Short,
  val expires: String?,
  val exceptionDates: String?,
  val daysInWeek: ShortArray,
  val daysInMonth: ShortArray,
  val daysInYear: ShortArray,
  val weeksInMonth: ShortArray,
  val monthsInYear: ShortArray,
) {
  companion object {
    fun fromJson(jsonString: String?): Result<CalendarRepeatRule> {
      if (jsonString == null) {
        return Result.Failure("Calendar repeat rule JSON is null", 0)
      }
      try {
        val json = JSONObject(jsonString)
        val frequency = json.getStringOrNull("frequency")
        val interval = json.optDouble("interval").toInt().toShort()
        val expires = json.getStringOrNull("expires")
        val exceptionDates = json.getStringOrNull("exceptionDates")
        val daysInWeek = toShortArray(json.optJSONArray("daysInWeek"))
        val daysInMonth = toShortArray(json.optJSONArray("daysInMonth"))
        val daysInYear = toShortArray(json.optJSONArray("daysInYear"))
        val weeksInMonth = toShortArray(json.optJSONArray("weeksInMonth"))
        val monthsInYear = toShortArray(json.optJSONArray("monthsInYear"))
        return Result.Success(
          CalendarRepeatRule(
            frequency, interval, expires, exceptionDates, daysInWeek,
            daysInMonth, daysInYear, weeksInMonth, monthsInYear,
          ),
        )
      } catch (exception: JSONException) {
        return Result.Failure("Exception parsing calendar repeat rule.", 0, exception)
      }
    }

    private fun toShortArray(array: JSONArray?): ShortArray {
      if (array == null) {
        return ShortArray(0)
      }

      return try {
        val length = array.length()
        val shortArray = ShortArray(length)
        for (i in 0 until length) {
          shortArray[i] = array.getDouble(i).toInt().toShort()
        }
        shortArray
      } catch (exception: JSONException) {
        HyprMXLog.e("Exception parsing JSON array")
        ShortArray(0)
      }
    }
  }
}
