package com.hyprmx.android.sdk.banner

import androidx.annotation.Keep
import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.core.LaunchBrowserNativeInterface
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import java.lang.ref.WeakReference

@Keep
internal interface BannerNativeInterface : LaunchBrowserNativeInterface {
  @RetainMethodSignature
  fun loadAdSuccess()

  @RetainMethodSignature
  fun loadAdFailed(error: String)

  @RetainMethodSignature
  fun onAdClicked()

  @RetainMethodSignature
  fun storePicture(url: String)

  @RetainMethodSignature
  fun createCalendarEvent(data: String)

  @RetainMethodSignature
  fun openOutsideApplication(data: String)

  @RetainMethodSignature
  fun startVisibilityTracking(rate: Long, opacityThresholdPercent: Int)

  @RetainMethodSignature
  fun stopVisibilityTracking()

  @RetainMethodSignature
  fun startOMSession()

  @RetainMethodSignature
  fun finishOMSession()

  @RetainMethodSignature
  fun adImpression()
}

internal interface BannerPresenterInterface {
  suspend fun loadAdSuccess()
  suspend fun loadAdFailed(error: String)
  suspend fun onAdClicked()

  suspend fun hyprMXBrowserClosed()
  suspend fun showHyprMXBrowser(viewModelIdentifier: String)
  suspend fun showPlatformBrowser(url: String)

  suspend fun storePicture(url: String)
  suspend fun createCalendarEvent(data: String)
  suspend fun openOutsideApplication(data: String)

  fun startVisibilityTracking(rate: Long, opacityThresholdPercent: Int)
  fun stopVisibilityTracking()
  fun startOMSession()
  fun finishOMSession()
  fun onAdImpression()
}

internal class BannerPresenterAdapter(presenter: BannerPresenterInterface, scope: CoroutineScope) : BannerNativeInterface, CoroutineScope by scope {
  private val presenter: WeakReference<BannerPresenterInterface> = WeakReference(presenter)
  override fun loadAdSuccess() {
    launch { presenter.get()?.loadAdSuccess() }
  }

  override fun loadAdFailed(error: String) {
    launch { presenter.get()?.loadAdFailed(error) }
  }

  override fun onAdClicked() {
    launch { presenter.get()?.onAdClicked() }
  }

  override fun storePicture(url: String) {
    launch { presenter.get()?.storePicture(url) }
  }

  override fun createCalendarEvent(data: String) {
    launch { presenter.get()?.createCalendarEvent(data) }
  }

  override fun openOutsideApplication(data: String) {
    launch { presenter.get()?.openOutsideApplication(data) }
  }

  override fun startVisibilityTracking(rate: Long, opacityThresholdPercent: Int) {
    launch { presenter.get()?.startVisibilityTracking(rate, opacityThresholdPercent) }
  }

  override fun stopVisibilityTracking() {
    launch { presenter.get()?.stopVisibilityTracking() }
  }

  override fun startOMSession() {
    launch { presenter.get()?.startOMSession() }
  }

  override fun finishOMSession() {
    launch { presenter.get()?.finishOMSession() }
  }

  override fun adImpression() {
    launch { presenter.get()?.onAdImpression() }
  }

  override fun hyprMXBrowserClosed() {
    launch { presenter.get()?.hyprMXBrowserClosed() }
  }

  override fun showHyprMXBrowser(viewModelIdentifier: String) {
    launch { presenter.get()?.showHyprMXBrowser(viewModelIdentifier) }
  }

  override fun showPlatformBrowser(url: String) {
    launch { presenter.get()?.showPlatformBrowser(url) }
  }
}
