package com.hyprmx.android.sdk.banner

import android.view.View
import com.hyprmx.android.sdk.core.SharedInterface
import com.hyprmx.android.sdk.presentation.ArgumentKey
import com.hyprmx.android.sdk.presentation.ArgumentKey.ACTUAL_HEIGHT
import com.hyprmx.android.sdk.presentation.ArgumentKey.ACTUAL_WIDTH
import com.hyprmx.android.sdk.presentation.ArgumentKey.ALPHA
import com.hyprmx.android.sdk.presentation.ArgumentKey.FULLY_OFFSCREEN
import com.hyprmx.android.sdk.presentation.ArgumentKey.FULLY_VISIBLE
import com.hyprmx.android.sdk.presentation.ArgumentKey.IS_SHOWN
import com.hyprmx.android.sdk.presentation.ArgumentKey.ON_SCREEN_X
import com.hyprmx.android.sdk.presentation.ArgumentKey.ON_SCREEN_Y
import com.hyprmx.android.sdk.presentation.ArgumentKey.PARENT_PASSES_ALPHA_THRESHOLD
import com.hyprmx.android.sdk.presentation.ArgumentKey.PARTIALLY_VISIBLE
import com.hyprmx.android.sdk.presentation.ArgumentKey.VISIBLE_HEIGHT
import com.hyprmx.android.sdk.presentation.ArgumentKey.VISIBLE_WIDTH
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.presentation.PublishingEvent

internal interface BannerSharedInterface : SharedInterface, VisibilityChangedListener {
  fun loadAd(definedSize: HyprMXBannerSize, actualWidth: Float, actualHeight: Float, bidResponse: String?)
  fun visibilityChanged(visibility: Int)
  fun onParentViewChanged(attached: Boolean)
  fun onContainerSizeChanged(width: Float, height: Float)
  fun onBannerClicked()
  fun onVisibleEvent(data: String)
}

internal class BannerSharedConnector(private val eventPublisher: PresentationEventPublisher) :
  BannerSharedInterface {
  override var viewModelIdentifier: String
    get() = eventPublisher.viewModelIdentifier
    set(value) {
      eventPublisher.viewModelIdentifier = value
    }

  override fun attach(nativeObject: Any) {
    eventPublisher.setPresenter(nativeObject)
  }

  override fun destroy() {
    eventPublisher.destroy()
  }

  override fun loadAd(
    definedSize: HyprMXBannerSize,
    actualWidth: Float,
    actualHeight: Float,
    bidResponse: String?,
  ) {
    eventPublisher.publishEvent(
      PublishingEvent.LOAD_AD,
      mapOf(
        ArgumentKey.DEFINED_SIZE to definedSize.toMap(),
        ArgumentKey.ACTUAL_SIZE to mapOf(ArgumentKey.CONTAINER_WIDTH to actualWidth, ArgumentKey.CONTAINER_HEIGHT to actualHeight),
        ArgumentKey.BID_RESPONSE to bidResponse,
      ),
    )
  }

  override fun visibilityChanged(visibility: Int) {
    eventPublisher.publishEvent(PublishingEvent.CONTAINER_VISIBLE_CHANGED, mapOf(ArgumentKey.CONTAINER_VISIBILITY to (visibility == View.VISIBLE)))
  }

  override fun onBannerClicked() {
    eventPublisher.publishEvent((PublishingEvent.ON_BANNER_CLICKED))
  }

  override fun onVisibleEvent(data: String) {
    eventPublisher.publishEvent((PublishingEvent.ON_VISIBLE_EVENT))
  }

  override fun onVisibleEvent(
    isShown: Boolean,
    visibleHeight: Int,
    visibleWidth: Int,
    actualHeight: Int,
    actualWidth: Int,
    fullyVisible: Boolean,
    partiallyVisible: Boolean,
    fullyOffscreen: Boolean,
    onScreenX: Int,
    onScreenY: Int,
    alpha: Float,
    parentAlphaPassesThreshold: Boolean,
  ) {
    eventPublisher.publishEvent(
      PublishingEvent.ON_VISIBLE_EVENT,
      mapOf(
        IS_SHOWN to isShown,
        VISIBLE_HEIGHT to visibleHeight,
        VISIBLE_WIDTH to visibleWidth,
        ACTUAL_HEIGHT to actualHeight,
        ACTUAL_WIDTH to actualWidth,
        FULLY_VISIBLE to fullyVisible,
        PARTIALLY_VISIBLE to partiallyVisible,
        FULLY_OFFSCREEN to fullyOffscreen,
        ON_SCREEN_X to onScreenX,
        ON_SCREEN_Y to onScreenY,
        ALPHA to alpha,
        PARENT_PASSES_ALPHA_THRESHOLD to parentAlphaPassesThreshold,
      ),
    )
  }

  override fun onParentViewChanged(attached: Boolean) {
    eventPublisher.publishEvent(PublishingEvent.PARENT_VISIBILITY_CHANGED, mapOf(ArgumentKey.PARENT_VIEW_ATTACHED to attached))
  }

  override fun onContainerSizeChanged(width: Float, height: Float) {
    eventPublisher.publishEvent(PublishingEvent.CONTAINER_SIZE_CHANGED, mapOf(ArgumentKey.CONTAINER_WIDTH to width, ArgumentKey.CONTAINER_HEIGHT to height))
  }
}
