package com.hyprmx.android.sdk.banner

import android.content.Context
import android.util.AttributeSet
import android.view.Gravity
import android.view.View
import android.widget.FrameLayout
import androidx.annotation.Keep
import androidx.annotation.VisibleForTesting
import androidx.core.view.contains
import com.hyprmx.android.sdk.core.HyprMX
import com.hyprmx.android.sdk.core.HyprMXControllerCleanupListener
import com.hyprmx.android.sdk.core.HyprMXErrors
import com.hyprmx.android.sdk.core.HyprMXIf
import com.hyprmx.android.sdk.core.HyprMXState
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.om.OpenMeasurementBannerSession
import com.hyprmx.android.sdk.overlay.HyprMXOverlay
import com.hyprmx.android.sdk.overlay.HyprMXOverlayAdapter
import com.hyprmx.android.sdk.placement.HyprMXLoadAdListener
import com.hyprmx.android.sdk.presentation.PresentationFactory
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.convertDpToPixel
import com.hyprmx.android.sdk.utility.convertPixelsToDp
import com.hyprmx.android.sdk.utility.getAdSizeFromAttributeSet
import com.hyprmx.android.sdk.utility.getPlacementNameAttributeSet
import com.hyprmx.android.sdk.webview.HyprMXWebView
import com.hyprmx.android.sdk.webview.createNewWebView
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine
import kotlin.math.roundToInt
import kotlin.properties.Delegates

/**
 * This view object represents a HyprMX Banner.
 *
 * To use this BannerView do the following:
 *
 * 1. Add the View to your layout using xml or programmatically
 * 2. Set the AdSize and Placement Name on the view
 * 3. Set your HyprMXBanner listener
 * 4. Ensure HyprMX is initialized
 * 5. Call loadAd on this view
 */
@Keep
class HyprMXBannerView @JvmOverloads constructor(
  context: Context,
  attrs: AttributeSet? = null,
  defStyleAttr: Int = 0,
) : FrameLayout(context, attrs, defStyleAttr),
  CoroutineScope by MainScope(),
  HyprMXBannerAd {

  private var result: ((Boolean) -> Unit)? = null

  @VisibleForTesting
  internal var presenter: HyprMXBannerContract.Presenter? = null

  @VisibleForTesting
  internal var presenterFactory: PresentationFactory? = null
    get() = field ?: hyprmxDelegate.hyprMXController?.presenterFactory

  private var attachedToWindow: Boolean by Delegates.observable(false) { _, _, new ->
    presenter?.onParentViewChanged(new)
  }

  @VisibleForTesting
  internal var hyprMX: HyprMXIf = HyprMX
  private lateinit var webView: HyprMXWebView
  private val visibilityTracker: ViewVisibilityTrackerInterface by lazy { ViewVisibilityTracker(this) }
  private var omSession: OpenMeasurementBannerSession? = null

  private val _bannerView by lazy {
    object :
      HyprMXBannerContract.View,
      HyprMXControllerCleanupListener,
      HyprMXOverlayAdapter.OverlayListener,
      HyprMXOverlay by HyprMXOverlayAdapter(context) {
      init {
        overlayListener = _overlayAdapterListener
      }

      override fun loadAdSuccess() {
        result?.let { it(true) }
        result = null
      }

      override fun loadAdFailure(error: HyprMXErrors) {
        HyprMXLog.e("Placement failed to load with error $error")
        result?.let { it(false) }
        result = null
      }

      override fun onAdClicked() {
        listener?.onAdClicked(this@HyprMXBannerView)
      }

      override fun reloadWebView() {
        removeAllViews()
      }

      override fun removePresenter() {
        presenter = null
      }

      override fun startVisibilityTracking(rate: Long, opacityThresholdPercent: Int) {
        visibilityTracker.startTracking(
          this@HyprMXBannerView,
          rate,
          opacityThresholdPercent,
          _visibilityChangedListener,
        )
      }

      override fun stopVisibilityTracking() {
        visibilityTracker.stopTracking()
      }

      override fun startOMSession() {
        HyprMXLog.d("startOMSession ")
        omSession =
          hyprmxDelegate.hyprMXController?.openMeasurementController?.generateBannerSession()
        omSession?.startOMSession(webView.webView)
      }

      override fun finishOMSession() {
        HyprMXLog.d("finishOMSession ")
        omSession?.finishOMSession()
        omSession = null
      }

      override fun onAdImpression() {
        HyprMXLog.d("onAdImpression")
        listener?.onAdImpression(this@HyprMXBannerView)
      }

      override fun storePicture(url: String) {
        launch {
          asyncSavePhoto(url)
        }
      }

      override fun cleanup() {
        finishOMSession()
        listener = null
        presenter?.cleanup()
        presenter?.view = null
        presenter = null

        visibilityTracker.stopTracking()

        launch {
          delay(500)
          webView.cleanup()
        }
      }
    }
  }

  private val _overlayAdapterListener by lazy {
    object : HyprMXOverlayAdapter.OverlayListener {
      @Suppress("DEPRECATION")
      override fun onOutsideAppPresented() {
        listener?.onAdLeftApplication(this@HyprMXBannerView)
      }

      override fun onHyprMXBrowserPresented() {
        listener?.onAdOpened(this@HyprMXBannerView)
      }

      override fun onHyprMXBrowserClosed() {
        listener?.onAdClosed(this@HyprMXBannerView)
      }
    }
  }

  private val _visibilityChangedListener by lazy {
    object : VisibilityChangedListener {
      override fun onVisibleEvent(
        isShown: Boolean,
        visibleHeight: Int,
        visibleWidth: Int,
        actualHeight: Int,
        actualWidth: Int,
        fullyVisible: Boolean,
        partiallyVisible: Boolean,
        fullyOffscreen: Boolean,
        onScreenX: Int,
        onScreenY: Int,
        alpha: Float,
        parentAlphaPassesThreshold: Boolean,
      ) {
        presenter?.onVisibleEvent(
          isShown = isShown,
          visibleWidth = visibleWidth,
          visibleHeight = visibleHeight,
          actualWidth = actualWidth,
          actualHeight = actualHeight,
          fullyVisible = fullyVisible,
          partiallyVisible = partiallyVisible,
          fullyOffscreen = fullyOffscreen,
          onScreenX = onScreenX,
          onScreenY = onScreenY,
          alpha = alpha,
          parentAlphaPassesThreshold = parentAlphaPassesThreshold,
        )
      }
    }
  }

  @Keep
  override var listener: HyprMXBannerListener? = null

  @Keep
  override var placementName: String = ""

  @Keep
  override var adSize: HyprMXBannerSize? = null

  // Set when the XML sets a custom size and we need to calculate the size at load time
  internal var useCustomSize = false
    private set

  @Keep
  constructor(
    context: Context,
    attrs: AttributeSet? = null,
    placementName: String,
    adSize: HyprMXBannerSize,
  ) : this(context, attrs) {
    this.placementName = placementName
    this.adSize = adSize
  }

  init {
    attrs?.let {
      getAdSizeFromAttributeSet(context, attrs)?.let {
        if (it is HyprMXBannerSize.HyprMXAdSizeCustom) {
          useCustomSize = true
        } else {
          adSize = it
        }
      }
      getPlacementNameAttributeSet(context, attrs)?.let { placementName = it }
    }
    prepareWebView(true)
  }

  private fun prepareWebView(init: Boolean = false) {
    if (init || !contains(webView)) {
      webView = HyprMXWebView(context, webView = createNewWebView(context))
      addView(webView)
    }
    updateWebViewSize()
  }

  private fun trackContainerVisibility() {
    tag = visibility
    viewTreeObserver.addOnGlobalLayoutListener {
      val newVis: Int = visibility
      if (tag as Int != newVis) {
        tag = visibility
        presenter?.visibilityChanged(visibility)
      }
    }
  }

  @Keep
  override fun loadAd(listener: HyprMXLoadAdListener) {
    internalLoadAd(onResult = listener::onAdLoaded)
  }

  @Keep
  override fun loadAd(bidResponse: String, listener: HyprMXLoadAdListener) {
    internalLoadAd(bidResponse = bidResponse, onResult = listener::onAdLoaded)
  }

  @Keep
  override suspend fun loadAd(bidResponse: String) = suspendCoroutine { continuation ->
    loadAd(bidResponse = bidResponse, onResult = continuation::resume)
  }

  /**
   * Loads a banner ad for this view
   */
  @Keep
  override suspend fun loadAd() = suspendCoroutine { continuation ->
    loadAd(onResult = continuation::resume)
  }

  private fun internalLoadAd(
    bidResponse: String? = null,
    onResult: (isAdAvailable: Boolean) -> Unit,
  ) {
    val actualWidth = this.width.convertPixelsToDp(context)
    val actualHeight = this.height.convertPixelsToDp(context)

    HyprMXLog.d(
      """
      HyprMXBannerView.loadAd 
          HyprMX = ${HyprMX.getInitializationState()}
          placementName = $placementName
          definedSize = $adSize
          actualWidth = $actualWidth
          actualHeight = $actualHeight
          useCustomSize = $useCustomSize
      """,
    )
    if (hyprMX.getInitializationState() != HyprMXState.INITIALIZATION_COMPLETE) {
      HyprMXLog.e(HyprMXErrors.SDK_NOT_INITIALIZED.toString())
      onResult(false)
      return
    }

    if (placementName.isBlank()) {
      HyprMXLog.e(HyprMXErrors.PLACEMENT_NAME_NOT_SET.toString())
      onResult(false)
      return
    }

    val definedSize = adSize ?: if (useCustomSize) {
      HyprMXBannerSize.HyprMXAdSizeCustom(
        width = actualWidth.roundToInt(), height = actualHeight.roundToInt(),
      )
    } else {
      HyprMXLog.e(HyprMXErrors.AD_SIZE_NOT_SET.toString())
      onResult(false)
      return
    }

    result = onResult

    if (presenter != null) {
      presenter?.loadAd(
        definedSize = definedSize,
        actualWidth = actualWidth,
        actualHeight = actualHeight,
        bidResponse = bidResponse,
      )
      return
    }

    prepareWebView()

    presenter = presenterFactory?.makeBannerPresenter(_bannerView, placementName)?.also {
      it.onContainerSizeChanged(
        width = width.convertPixelsToDp(context),
        height = height.convertPixelsToDp(context),
      )

      it.onParentViewChanged(attachedToWindow)
      it.visibilityChanged(visibility)
      this.webView.initialize(it.viewModelIdentifier)
      it.loadAd(
        definedSize = definedSize,
        actualWidth = actualWidth,
        actualHeight = actualHeight,
        bidResponse = bidResponse,
      )
    }
  }

  private fun updateWebViewSize() {
    adSize?.let {
      HyprMXLog.d("Updating webview banner with size: ${it.width}, ${it.height}")
      val webviewLayoutParams =
        LayoutParams(it.width.convertDpToPixel(context), it.height.convertDpToPixel(context))
      webviewLayoutParams.gravity = Gravity.CENTER
      webView.layoutParams = webviewLayoutParams
    }
  }

  @Keep
  override fun destroy() {
    cleanup()
  }

  @VisibleForTesting
  internal fun cleanup() {
    _bannerView.cleanup()
  }

  override fun onAttachedToWindow() {
    HyprMXLog.d("onAttachedToWindow ${presenter?.viewModelIdentifier}")
    attachedToWindow = true
    trackContainerVisibility()
    super.onAttachedToWindow()
  }

  override fun onDetachedFromWindow() {
    HyprMXLog.d("onDetachedFromWindow ${presenter?.viewModelIdentifier}")
    attachedToWindow = false
    visibilityTracker.stopTracking()
    super.onDetachedFromWindow()
  }

  override fun onLayout(changed: Boolean, left: Int, top: Int, right: Int, bottom: Int) {
    if (changed) {
      HyprMXLog.d("onLayout (${width.convertPixelsToDp(context)}, ${height.convertPixelsToDp(context)}) for ${presenter?.viewModelIdentifier}")
      presenter?.onContainerSizeChanged(
        width = width.convertPixelsToDp(context),
        height = height.convertPixelsToDp(context),
      )
    }
    super.onLayout(changed, left, top, right, bottom)
  }

  override fun onVisibilityChanged(changedView: View, visibility: Int) {
    super.onVisibilityChanged(changedView, visibility)
    presenter?.visibilityChanged(visibility)
  }
}
