package com.hyprmx.android.sdk.banner

import android.graphics.Rect
import android.view.View
import com.hyprmx.android.sdk.utility.HyprMXLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

internal class ViewVisibilityTracker(scope: CoroutineScope) : ViewVisibilityTrackerInterface, CoroutineScope by scope {
  private var trackingJob: Job? = null
  override fun startTracking(view: View, refreshTimeInMS: Long, opacityThresholdPercent: Int, vcl: VisibilityChangedListener) {
    if (trackingJob != null) {
      HyprMXLog.d("Tracking job already started on view ${view.hashCode()}")
      return
    }
    HyprMXLog.d("tracking starting on view ${view.hashCode()}")
    trackingJob = launch {
      while (true) {
        val globalVisibilityRectangle = Rect()
        view.getGlobalVisibleRect(globalVisibilityRectangle)

        val visibleWidth = globalVisibilityRectangle.right - globalVisibilityRectangle.left
        val visibleHeight = globalVisibilityRectangle.bottom - globalVisibilityRectangle.top

        val actualHeight: Int = view.measuredHeight
        val actualWidth: Int = view.measuredWidth

        val fullyVisible = visibleHeight == actualHeight
        val partiallyVisible = visibleHeight in 1 until actualHeight
        val fullyOffscreen = visibleHeight < actualHeight && visibleHeight <= 0

        val location = IntArray(2)
        view.getLocationOnScreen(location)
        val x = location[0]
        val y = location[1]

        vcl.onVisibleEvent(
          isShown = view.isShown,
          visibleWidth = visibleWidth,
          visibleHeight = visibleHeight,
          actualWidth = actualWidth,
          actualHeight = actualHeight,
          fullyVisible = fullyVisible,
          partiallyVisible = partiallyVisible,
          fullyOffscreen = fullyOffscreen,
          onScreenX = x,
          onScreenY = y,
          alpha = view.alpha,
          parentAlphaPassesThreshold = checkOpacityThreshold(view, opacityThresholdPercent),
        )

        delay(refreshTimeInMS)
      }
    }
  }

  override fun stopTracking() {
    trackingJob?.let {
      it.cancel()
      trackingJob = null
    }
  }

  private fun checkOpacityThreshold(view: View, opacityThresholdPercent: Int): Boolean {
    var parent = view.parent
    while (parent is View) {
      if (((parent as View).alpha * 100).toInt() < opacityThresholdPercent) {
        return false
      }
      parent = (parent as View).parent
    }
    return true
  }
}

internal interface ViewVisibilityTrackerInterface {
  fun startTracking(view: View, refreshTimeInMS: Long, opacityThresholdPercent: Int, vcl: VisibilityChangedListener)
  fun stopTracking()
}

internal interface VisibilityChangedListener {
  fun onVisibleEvent(
    isShown: Boolean,
    visibleHeight: Int,
    visibleWidth: Int,
    actualHeight: Int,
    actualWidth: Int,
    fullyVisible: Boolean,
    partiallyVisible: Boolean,
    fullyOffscreen: Boolean,
    onScreenX: Int,
    onScreenY: Int,
    alpha: Float,
    parentAlphaPassesThreshold: Boolean,
  )
}
