package com.hyprmx.android.sdk.calendar

import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.provider.CalendarContract
import com.hyprmx.android.sdk.api.data.CalendarEvent
import com.hyprmx.android.sdk.api.data.CalendarRepeatRule
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.Result
import java.text.ParseException
import java.text.SimpleDateFormat
import java.util.Date
import java.util.HashMap
import java.util.Locale
import java.util.TimeZone

internal interface CalendarEventControllerIf {
  fun createCalendarEvent(calEventString: String, context: Context): Boolean
}

internal class CalendarEventController : CalendarEventControllerIf {
  /**
   * Sets up intent to start Calendar Activity with the specified event detail from parameter.
   *
   * @param calEventString JSON in String value.
   * @return true if a calendar intent was presented
   */
  override fun createCalendarEvent(calEventString: String, context: Context): Boolean {
    return createAndLaunchCalendarEvent(calEventString, context)
  }

  fun createAndLaunchCalendarEvent(calEventString: String, context: Context): Boolean {
    return try {
      val calendarParams = convertToCalendarMapping(calEventString)
      val intent = Intent(Intent.ACTION_INSERT).setType(CALENDAR_MIME_TYPE)
      for (key in calendarParams.keys) {
        when (val value = calendarParams[key]) {
          is Long -> intent.putExtra(key, value.toLong())
          is Int -> intent.putExtra(key, value.toInt())
          else -> intent.putExtra(key, value as String)
        }
      }
      intent.flags = Intent.FLAG_ACTIVITY_NEW_TASK
      context.startActivity(intent)
      true
    } catch (e: ActivityNotFoundException) {
      HyprMXLog.d("Calendar app not installed.")
      false
    } catch (e: IllegalArgumentException) {
      HyprMXLog.d("Invalid params for calendar event.")
      false
    } catch (e: Exception) {
      HyprMXLog.d("Error creating calendar event.")
      false
    }
  }

  /**
   * Converts the JSON String to a Map and returns it.
   *
   * @param calEventString JSON in String value.
   * @return HashMap with key value pair for calendar event.
   */
  @Throws(IllegalArgumentException::class)
  fun convertToCalendarMapping(calEventString: String): Map<String, Any> {
    val calendarEventResult = CalendarEvent.fromJson(calEventString)
    val calendarMapping = HashMap<String, Any>()
    if (calendarEventResult is Result.Failure) {
      HyprMXLog.e("Unable to parse calendar event")
      throw IllegalArgumentException("Unable to parse calendar event")
    }
    val calendarEvent = (calendarEventResult as Result.Success).value

    if (calendarEvent.description == null || calendarEvent.start == null) {
      throw IllegalArgumentException("missing description or start fields")
    }

    calendarMapping[CalendarContract.Events.TITLE] = calendarEvent.description

    if (calendarEvent.start.isNotEmpty()) {
      val startDateTime = parseDate(calendarEvent.start)
      if (startDateTime != null) {
        calendarMapping[CalendarContract.EXTRA_EVENT_BEGIN_TIME] = startDateTime.time
      } else {
        throw IllegalArgumentException("calendar start time is malformed")
      }
    } else {
      throw IllegalArgumentException("calendar start is null")
    }

    if (!calendarEvent.end.isNullOrEmpty()) {
      val endDateTime = parseDate(calendarEvent.end)
      if (endDateTime != null) {
        calendarMapping[CalendarContract.EXTRA_EVENT_END_TIME] = endDateTime.time
      } else {
        throw IllegalArgumentException("calendar end time is malformed")
      }
    }

    if (calendarEvent.location != null) {
      calendarMapping[CalendarContract.Events.EVENT_LOCATION] = calendarEvent.location
    }

    if (calendarEvent.summary != null) {
      calendarMapping[CalendarContract.Events.DESCRIPTION] = calendarEvent.summary
    }

    if (calendarEvent.transparency != null) {
      calendarMapping[CalendarContract.Events.AVAILABILITY] =
        if (calendarEvent.transparency == "transparent") {
          CalendarContract.Events.AVAILABILITY_FREE
        } else {
          CalendarContract.Events.AVAILABILITY_BUSY
        }
    }

    if (calendarEvent.recurrence != null) {
      calendarMapping[CalendarContract.Events.RRULE] =
        parseCalendarRepeatRule(calendarEvent.recurrence)
    }

    return calendarMapping
  }

  /**
   * Android end date of a recurrence rule needs to be in the format as, [.DATE_EXPIRATION_FORMAT].
   *
   * @param date Date of expiration date of recurrence event.
   * @return String of expiration date in expected format.
   */
  private fun formatDate(date: Date): String? {
    var result: String? = null

    try {
      val simpleDateFormat = SimpleDateFormat(DATE_EXPIRATION_FORMAT, Locale.US)
      simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
      result = simpleDateFormat.format(date)
    } catch (e: Exception) {
      HyprMXLog.e(e.message)
    }

    return result
  }

  /**
   * Parses CalendarRepeatRule instance to a String value that
   * is supported by the Android Calendar Provider recurrence rule.
   *
   * @param calendarRepeatRule
   * @return
   * @throws IllegalArgumentException
   */
  @Throws(IllegalArgumentException::class)
  private fun parseCalendarRepeatRule(calendarRepeatRule: CalendarRepeatRule): String {
    val rule = StringBuilder()
    if (calendarRepeatRule.frequency != null) {
      val frequency = calendarRepeatRule.frequency
      var interval = -1
      if (calendarRepeatRule.interval > 0) {
        interval = calendarRepeatRule.interval.toInt()
      }
      if (frequency == "daily") {
        rule.append("FREQ=DAILY;")
        if (interval != -1) {
          rule.append("INTERVAL=$interval;")
        }
      } else if (frequency == "weekly") {
        rule.append("FREQ=WEEKLY;")
        if (interval != -1) {
          rule.append("INTERVAL=$interval;")
        }
        if (calendarRepeatRule.daysInWeek.isNotEmpty()) {
          val weekdays = translateWeekShortsToDays(calendarRepeatRule.daysInWeek)
          rule.append("BYDAY=$weekdays;")
        } else {
          throw IllegalArgumentException("invalid daysInWeek")
        }
      } else if (frequency == "monthly") {
        rule.append("FREQ=MONTHLY;")
        if (interval != -1) {
          rule.append("INTERVAL=$interval;")
        }
        if (calendarRepeatRule.daysInMonth.isNotEmpty()) {
          val monthDays = translateMonthShortsToDays(calendarRepeatRule.daysInMonth)
          rule.append("BYMONTHDAY=$monthDays;")
        } else {
          throw IllegalArgumentException("invalid daysInMonth")
        }
      } else if (frequency == "yearly") {
        rule.append("FREQ=YEARLY;")
        if (interval != -1) {
          rule.append("INTERVAL=$interval;")
        }
        if (calendarRepeatRule.daysInYear.isNotEmpty()) {
          val yearDays = translateYearShortsToDays(calendarRepeatRule.daysInYear)
          rule.append("BYYEARDAY=$yearDays;")
        }

        if (calendarRepeatRule.monthsInYear.isNotEmpty()) {
          val yearMonths = translateYearShortsToMonths(calendarRepeatRule.monthsInYear)
          rule.append("BYMONTH=$yearMonths;")
        }
      } else {
        throw IllegalArgumentException("unsupported frequency: $frequency")
      }

      if (calendarRepeatRule.expires != null) {
        val lastDateTime = formatDate(parseDate(calendarRepeatRule.expires)!!)
        if (lastDateTime != null) {
          rule.append("UNTIL=$lastDateTime;")
        } else {
          throw IllegalArgumentException("calendar recurrence expiration time is malformed")
        }
      }
    }
    return rule.toString()
  }

  /**
   * Converts array of days in week to String value.
   * ex. [0, 2] gets converted to "SU,TU".
   */
  @Throws(IllegalArgumentException::class)
  private fun translateWeekShortsToDays(daysInWeek: ShortArray): String {
    if (daysInWeek.isEmpty()) {
      throw IllegalArgumentException("must have at least 1 day of the week")
    }
    return daysInWeek.joinToString(separator = ",") { dayNumberToDayOfWeekString(it.toInt()) }
  }

  /**
   * Converts array of days in month to String value.
   * Possible days are ranging from -30 to 31.
   * ex. [4, 10] gets converted to "4, 10".
   */
  @Throws(IllegalArgumentException::class)
  private fun translateMonthShortsToDays(daysInMonth: ShortArray): String {
    if (daysInMonth.isEmpty()) {
      throw IllegalArgumentException("must have at least 1 day of the month")
    }
    return daysInMonth.joinToString(separator = ",") { convertToDayOfMonthString(it.toInt()) }
  }

  /**
   * Converts array of days in year to String value.
   * Possible days are ranging from -364 to 365.
   * ex. [156, 157] gets converted to "156, 157".
   */
  @Throws(IllegalArgumentException::class)
  private fun translateYearShortsToDays(daysInYear: ShortArray): String {
    if (daysInYear.isEmpty()) {
      throw IllegalArgumentException("must have at least 1 day of the year")
    }
    return daysInYear.joinToString(separator = ",") { convertToDayOfYearString(it.toInt()) }
  }

  /**
   * Converts array of days in year to String value.
   * Possible days are ranging from 1 to 12.
   * ex. [2, 12] gets converted to "2, 12".
   */
  @Throws(IllegalArgumentException::class)
  private fun translateYearShortsToMonths(monthsInYear: ShortArray): String {
    if (monthsInYear.isEmpty()) {
      throw IllegalArgumentException("must have at least 1 month of the year")
    }
    return monthsInYear.joinToString(separator = ",") { convertToMonthOfYearString(it.toInt()) }
  }

  @Throws(IllegalArgumentException::class)
  private fun dayNumberToDayOfWeekString(number: Int): String {
    return when (number) {
      0 -> "SU"
      1 -> "MO"
      2 -> "TU"
      3 -> "WE"
      4 -> "TH"
      5 -> "FR"
      6 -> "SA"
      else -> throw IllegalArgumentException("invalid day of the week: $number")
    }
  }

  /**
   * Validating and converting number from int to String.
   * Sending invalid numbers crash the Google Calendar app.
   *
   * @param number day of month in int value
   * @return day of month in String value
   * @throws IllegalArgumentException
   */
  @Throws(IllegalArgumentException::class)
  private fun convertToDayOfMonthString(number: Int): String {
    if (number != 0 && number > -MAX_NUMBER_DAYS_IN_MONTH && number <= MAX_NUMBER_DAYS_IN_MONTH) {
      return number.toString()
    } else {
      throw IllegalArgumentException("invalid day of the month: $number")
    }
  }

  /**
   * Validating and converting number from int to String.
   * Sending invalid numbers crash the Google Calendar app.
   *
   * @param number day of year in int value
   * @return day of year in String value
   * @throws IllegalArgumentException
   */
  @Throws(IllegalArgumentException::class)
  private fun convertToDayOfYearString(number: Int): String {
    if (number != 0 && number > -MAX_NUMBER_DAYS_IN_YEAR && number <= MAX_NUMBER_DAYS_IN_YEAR) {
      return number.toString()
    } else {
      throw IllegalArgumentException("invalid day of the year: $number")
    }
  }

  /**
   * Validating and converting number from int to String.
   * Sending invalid numbers crash the Google Calendar app.
   *
   * @param number month of year in int value
   * @return month of year in String value
   * @throws IllegalArgumentException
   */
  @Throws(IllegalArgumentException::class)
  private fun convertToMonthOfYearString(number: Int): String {
    if (number != 0 && number <= MAX_NUMBER_MONTHS_IN_YEAR) {
      return number.toString()
    } else {
      throw IllegalArgumentException("invalid month of the year: $number")
    }
  }

  companion object {

    const val CALENDAR_MIME_TYPE = "vnd.android.cursor.item/event"

    private const val MAX_NUMBER_DAYS_IN_MONTH = 31
    private const val MAX_NUMBER_DAYS_IN_YEAR = 366 // accounting for leap years
    private const val MAX_NUMBER_MONTHS_IN_YEAR = 12
    private val DATE_FORMATS = listOf(
      "yyyy-MM-dd'T'HH:mm:ss.SSSX",
      "yyyy-MM-dd'T'HH:mm:ss.SSSZ",
      "yyyy-MM-dd'T'HH:mm:ssX",
      "yyyy-MM-dd'T'HH:mm:ssZ",
      "yyyy-MM-dd'T'HH:mmX",
      "yyyy-MM-dd'T'HH:mmZ",
      "yyyy-MM-dd'T'HH:mm:ss.SSS",
      "yyyy-MM-dd'T'HH:mm:ss",
      "yyyy-MM-dd'T'HH:mm",
      "yyyyMMdd'T'HH:mm:ss.SSSX",
      "yyyyMMdd'T'HH:mm:ss.SSSZ",
      "yyyyMMdd'T'HH:mm:ssX",
      "yyyyMMdd'T'HH:mm:ssZ",
      "yyyyMMdd'T'HH:mmX",
      "yyyyMMdd'T'HH:mmZ",
      "yyyyMMdd'T'HH:mm:ss.SSS",
      "yyyyMMdd'T'HH:mm:ss",
      "yyyyMMdd'T'HH:mm",
      "yyyy-MM-dd'T'HHmmss.SSSX",
      "yyyy-MM-dd'T'HHmmss.SSSZ",
      "yyyy-MM-dd'T'HHmmssX",
      "yyyy-MM-dd'T'HHmmssZ",
      "yyyy-MM-dd'T'HHmmX",
      "yyyy-MM-dd'T'HHmmZ",
      "yyyy-MM-dd'T'HHmmss.SSS",
      "yyyy-MM-dd'T'HHmmss",
      "yyyy-MM-dd'T'HHmm",
      "yyyyMMdd'T'HHmmss.SSSX",
      "yyyyMMdd'T'HHmmss.SSSZ",
      "yyyyMMdd'T'HHmmssX",
      "yyyyMMdd'T'HHmmssZ",
      "yyyyMMdd'T'HHmmX",
      "yyyyMMdd'T'HHmmZ",
      "yyyyMMdd'T'HHmmss.SSS",
      "yyyyMMdd'T'HHmmss",
      "yyyyMMdd'T'HHmm",
    )
    private const val DATE_EXPIRATION_FORMAT = "yyyyMMdd'T'HHmmss'Z'"

    private fun parseDate(dateTime: String): Date? {
      var result: Date? = null
      var parser: String? = null
      for (DATE_FORMAT in DATE_FORMATS) {
        try {
          val simpleDateFormat = SimpleDateFormat(DATE_FORMAT, Locale.US)
          simpleDateFormat.timeZone = TimeZone.getTimeZone("UTC")
          result = simpleDateFormat.parse(dateTime)
          if (result != null) {
            parser = DATE_FORMAT
            break
          }
        } catch (e: IllegalArgumentException) {
          // an exception is okay, just try the next format and find the first one that works
        } catch (e: ParseException) {
          HyprMXLog.e("Error parsing date $dateTime")
        }
      }

      if (result == null) {
        HyprMXLog.d("No parser for $dateTime")
      } else {
        HyprMXLog.d("$dateTime parsed with $parser")
      }
      return result
    }
  }
}
