package com.hyprmx.android.sdk.core

import android.annotation.SuppressLint
import android.content.Context
import androidx.annotation.Keep
import androidx.annotation.VisibleForTesting
import com.hyprmx.android.sdk.consent.ConsentStatus
import com.hyprmx.android.sdk.exceptions.InitializationFailureException
import com.hyprmx.android.sdk.placement.Placement
import com.hyprmx.android.sdk.placement.invalidPlacement
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.MINIMUM_SUPPORTED_VERSION
import com.hyprmx.android.sdk.utility.isBuildVersionUnsupported
import com.hyprmx.android.sdk.webview.SystemWebViewAvailability
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.cancelChildren
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

/**
 * Provides public facing API to initialize and to interact with the HyprMX SDK.
 */

internal val hyprmxDelegate: HyprMXDelegate = HyprMXDelegate()

@Keep
object HyprMX : HyprMXIf by hyprmxDelegate

@SuppressLint("StaticFieldLeak")
internal class HyprMXDelegate(
  private val factory: HyprMXControllerFactory = DefaultHyprMXControllerFactory(),
  private val job: Job = SupervisorJob(),
  private val sytemWebViewAvailability: SystemWebViewAvailability = SystemWebViewAvailability(),
  internal val hyprMXMediation: HyprMXMediationAdapter = HyprMXMediationAdapter(),
) : HyprMXIf, HyprMXMediation by hyprMXMediation {

  private val scope = CoroutineScope(Dispatchers.Main + job)

  internal var isAgeRestrictedUser: Boolean = false
    private set

  internal var testModeEnabled: Boolean = false
    private set

  var hyprMXController: HyprMXController? = null
  private var audioAdListener: HyprMXIf.HyprMXAudioAdListener? = null
  internal val userExtras: MutableMap<String, String?> = HashMap()
  internal var failureCount = 0

  private fun validateInitialize(context: Context): InitResult? =
    when {
      isBuildVersionUnsupported() -> {
        "HyprMX requires Android OS version API $MINIMUM_SUPPORTED_VERSION or newer. SDK disabled.".let {
          HyprMXLog.e(it)
          InitResult(success = false, message = it)
        }
      }

      !sytemWebViewAvailability.hasSystemWebViewAvailable(context) -> {
        "HyprMX requires a system webview be enabled. Please enable your system webview.".let {
          HyprMXLog.e(it)
          InitResult(success = false, message = it)
        }
      }

      else -> null
    }

  private fun definitiveFailureResponse(): InitResult {
    return "HyprMX Initialization Failed recently, please wait before trying again.".let {
      HyprMXLog.i(it)
      InitResult(success = false, message = it)
    }
  }

  /**
   * [context] The context
   * [distributorId] The distribution id provided to you by HyprMX.
   * [listener] The HyprMX Initialization Listener. HyprMX will callback to this when initialization is complete.
   */
  override fun initialize(
    context: Context,
    distributorId: String,
    listener: HyprMXIf.HyprMXInitializationListener,
  ) {
    // If there was an existing child job, cancel it since this is a new request
    job.cancelChildren()

    scope.launch {
      try {
        init(context, distributorId)
      } catch (ex: Exception) {
        definitiveFailureResponse()
      }.also {
        failureCount = if (it.success) { 0 } else { failureCount + 1 }
        listener.onInitialized(it)
      }
    }
  }

  override suspend fun initialize(context: Context, distributorId: String) =
    suspendCoroutine { continuation ->
      val listener = object : HyprMXIf.HyprMXInitializationListener {
        override fun onInitialized(result: InitResult) {
          continuation.resume(result)
        }
      }

      initialize(context, distributorId, listener)
    }

  @VisibleForTesting
  internal suspend fun init(context: Context, distributorId: String): InitResult {
    validateInitialize(context)?.let {
      return it
    }

    HyprMXLog.setup(context)
    hyprMXController?.cleanup()

    return initializeHyprController(
      context = context,
      distributorId = distributorId,
      consentStatus = getConsentStatus(),
    ) ?: InitResult(success = false)
  }

  override fun setAgeRestrictedUser(enabled: Boolean) {
    this.isAgeRestrictedUser = enabled
  }

  private suspend fun initializeHyprController(
    context: Context,
    distributorId: String,
    consentStatus: ConsentStatus,
  ) = withContext(Dispatchers.Main) {
    hyprMXController = factory.createHyprMXController(
      context.applicationContext,
      distributorId,
      consentStatus = consentStatus,
    )

    hyprMXController?.let {
      when (val initResult = it.initialize()) {
        is HyprMXController.InitResult.SuccessWithUpdate -> {
          hyprMXController = initResult.hyprMXController
          InitResult(success = true)
        }

        is HyprMXController.InitResult.Success -> {
          hyprMXController = it
          InitResult(success = true)
        }

        is HyprMXController.InitResult.Failure -> {
          hyprMXController = null
          throw InitializationFailureException()
        }
      }
    }
  }

  /**
   * Gets the placement for the given name.
   */
  override fun getPlacement(placementName: String): Placement {
    return hyprMXController?.getPlacement(placementName) ?: invalidPlacement(placementName)
  }

  override fun getPlacements(): Set<Placement> {
    return hyprMXController?.getPlacements() ?: emptySet()
  }

  override fun sessionToken(): String? =
    if (hyprMXController?.isInitialized() == false) {
      HyprMXLog.e("HyprMX needs to be initialized before retrieving session token")
      null
    } else {
      try {
        hyprMXController?.getSessionToken()
      } catch (exception: Exception) {
        HyprMXLog.e("There was an error generating the session token")
        null
      }
    }

  /**
   * Gets the HyprMX initialization state.
   */
  override fun getInitializationState(): HyprMXState {
    return hyprMXController?.getInitializationState() ?: HyprMXState.NOT_INITIALIZED
  }

  override fun setConsentStatus(consentStatus: ConsentStatus) {
    hyprMXController?.setConsentStatus(consentStatus)
  }

  override fun enableTestMode() {
    if (testModeEnabled) {
      HyprMXLog.i("Test mode is already enabled")
      return
    }

    testModeEnabled = true
    val state = hyprMXController?.getInitializationState()
    if (state == HyprMXState.INITIALIZING || state == HyprMXState.INITIALIZATION_COMPLETE) {
      HyprMXLog.e("Test mode must be set before HyprMX is initialized")
    }
  }

  fun getSharedJSVersion(): Int? {
    return hyprMXController?.getSharedJSVersion()
  }

  fun getConsentStatus(): ConsentStatus {
    return hyprMXController?.getConsentStatus() ?: ConsentStatus.CONSENT_STATUS_UNKNOWN
  }

  override fun setAudioAdListener(listener: HyprMXIf.HyprMXAudioAdListener?) {
    audioAdListener = listener
  }

  override fun setUserExtras(key: String, value: String?) {
    value?.let {
      userExtras[key] = it
    } ?: userExtras.remove(key)
  }

  internal fun onAudioStarted() {
    audioAdListener?.onAdAudioStart()
  }

  internal fun onAudioEnded() {
    audioAdListener?.onAdAudioEnd()
  }
}
