package com.hyprmx.android.sdk.core

import android.annotation.TargetApi
import android.content.Context
import android.os.Build
import androidx.core.content.edit
import com.hyprmx.android.sdk.bidding.BiddingController
import com.hyprmx.android.sdk.consent.ConsentStatus
import com.hyprmx.android.sdk.initialization.InitializationDelegator
import com.hyprmx.android.sdk.initialization.InitializationResult
import com.hyprmx.android.sdk.initialization.UpdateResult
import com.hyprmx.android.sdk.om.OpenMeasurementController
import com.hyprmx.android.sdk.placement.Placement
import com.hyprmx.android.sdk.placement.PlacementImpl
import com.hyprmx.android.sdk.presentation.PresentationController
import com.hyprmx.android.sdk.presentation.PresentationDelegator
import com.hyprmx.android.sdk.utility.HyprMXLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.cancel
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import org.json.JSONException
import java.io.FileNotFoundException
import java.lang.ref.WeakReference

/**
 * The main component of the HyprMX SDK.
 *
 * This component is responsible for managing the sub-components and use-cases.
 *
 * To work with the webview
 * 1. Prepare webview with JS enabled. (call setupWebView)
 * 2. Attach the JSInterface bindings. (call controller constructor and bind in the init function)
 * 3. Load the page by calling loadWebView (this will let the binding work).  Wait for it to finish
 * 4. Evaluate JS (eg. Initialize the controllers make calls)
 *
 * [applicationContext] The application context
 * [distributorId] The distributor ID
 * [consentStatus] The current consent status
 * [applicationModule] The application module that holds the sdk dependencies
 */

internal class HyprMXController(
  context: Context,
  distributorId: String,
  consentStatus: ConsentStatus,
  private val applicationModule: ApplicationModule = DefaultApplicationModule(
    applicationContext = context.applicationContext,
    distributorId = distributorId,
    consentStatus = consentStatus,
  ),
) :
  ApplicationModule by applicationModule,
  InitializationDelegator,
  PlacementImpl.PlacementDelegator,
  PresentationDelegator,
  BiddingController by applicationModule.biddingController,
  CoroutineScope by applicationModule.scope {

  companion object {
    const val HYPRMX_PREFS_INTERNAL = "hyprmx_prefs_internal"
    const val PREF_DISTRIBUTOR_ID = "distributor_id"

    fun copyPlacements(
      newHyprMXController: HyprMXController,
      oldHyprMXController: HyprMXController,
    ) {
      oldHyprMXController.placementController.placements.forEach {
        it.placementDelegate = newHyprMXController
        (newHyprMXController.placementController.placements as MutableSet).add(it)
      }
    }
  }

  lateinit var presentationController: PresentationController
  private lateinit var coreJS: String
  internal var mobileJSVersion: Int? = null
  private var hyprMXState = HyprMXState.NOT_INITIALIZED

  sealed class InitResult {
    class SuccessWithUpdate(val hyprMXController: HyprMXController) : InitResult()
    object Success : InitResult()
    object Failure : InitResult()
  }

  @TargetApi(Build.VERSION_CODES.KITKAT)
  suspend fun initialize(newCoreJS: String? = null): InitResult {
    setInitializationState(HyprMXState.INITIALIZING)
    this.presentationController = PresentationController(
      applicationModule = applicationModule,
      presentationDelegator = this,
    )

    storageHelper.deleteSharedJSIfSdkVersionUpdated()

    try {
      this@HyprMXController.coreJS = newCoreJS ?: storageHelper.getCoreJSFromFile()
    } catch (exception: FileNotFoundException) {
      HyprMXLog.e("Failed to find file exception: ${exception.localizedMessage}")
      setInitializationState(HyprMXState.INITIALIZATION_FAILED)
      return InitResult.Failure
    }

    val success = jsEngine.loadSharedJS(coreJS)

    if (!success) {
      // We had an error loading JS in the web view
      HyprMXLog.e("There was a failure loading the shared JS")
      setInitializationState(HyprMXState.INITIALIZATION_FAILED)
      return InitResult.Failure
    } else {
      HyprMXLog.d("Shared JS successfully loaded")
    }

    initCache()
    parameterController.initialize()

    val initState = if (newCoreJS == null) {
      initializeBase()
    } else {
      initializeFromUpgrade()
    }

    when (initState) {
      is InitResult.Failure -> setInitializationState(HyprMXState.INITIALIZATION_FAILED)
      else -> {
        setInitializationState(HyprMXState.INITIALIZATION_COMPLETE)
        // Try to save the new javascript to disk
        newCoreJS?.let {
          if (!writeCoreJSToFile(newCoreJS)) {
            HyprMXLog.e("There was a failure storing the core to disk")
            return InitResult.Failure
          } else {
            HyprMXLog.d("Successfully save update.")
          }
        }
      }
    }

    if (jsEngine.hasFailures) {
      HyprMXLog.e("Uncaught errors detected while initializing.  Returning failure.")
      setInitializationState(HyprMXState.INITIALIZATION_FAILED)
      return InitResult.Failure
    }

    return initState
  }

  /**
   * Initialize the HyprController
   */
  private suspend fun initializeBase(): InitResult {
    return try {
      when (
        val initResult = initializationController.initialize(this@HyprMXController)
      ) {
        is InitializationResult.UpdateJavascript -> updateJavascript(
          initResult.coreJSURL,
          initResult.timeout.toLong(),
        )

        is InitializationResult.RollbackJavascript -> rollbackCoreJS()
        is InitializationResult.Success -> {
          initializePlacements(initResult.placementJSON)
          InitResult.Success
        }

        is InitializationResult.Failed -> InitResult.Failure
      }
    } catch (exception: Exception) {
      HyprMXLog.e("Exception initializing HyprMX: ${exception.localizedMessage}")
      InitResult.Failure
    }
  }

  /**
   * Initialize the HyprController from an upgrade call.
   *
   * Should we receive and upgrade/rollback during the upgrade we will just fail the current upgrade and stay at the current version.
   *
   */
  suspend fun initializeFromUpgrade(): InitResult {
    return when (
      val initResult = initializationController.initialize(
        this@HyprMXController,
      )
    ) {
      is InitializationResult.UpdateJavascript -> {
        HyprMXLog.e("Upgrade called during upgrade.")
        InitResult.Failure
      }

      is InitializationResult.RollbackJavascript -> {
        HyprMXLog.e("Rollback called during upgrade.")
        InitResult.Failure
      }

      is InitializationResult.Success -> {
        initializePlacements(initResult.placementJSON)
        InitResult.Success
      }

      is InitializationResult.Failed -> {
        HyprMXLog.e("Initialization failed during upgrade.")
        InitResult.Failure
      }
    }
  }

  /**
   * Rolls back the JS to the one the SDK was shipped with.
   */
  private suspend fun rollbackCoreJS(): InitResult {
    val rollbackResult = updateController.sharedJSRollback(this@HyprMXController, consentStatus)
    return if (rollbackResult is UpdateResult.Success) {
      InitResult.SuccessWithUpdate(rollbackResult.hyprMXController)
    } else {
      InitResult.Failure
    }
  }

  /**
   * Request to update the javascript to a different version.
   *
   * Should the process take more than the max time time a Failed result will be returned
   *
   * @param coreJSURL The URL pointing to the new javascript
   * @param updateTimeout The max time in seconds that the update should take.
   * @return The result of the update.
   */
  private suspend fun updateJavascript(coreJSURL: String, updateTimeout: Long): InitResult {
    val updateResult = updateController.updateJavascript(
      coreJSURL,
      this@HyprMXController,
      consentStatus,
      updateTimeout,
    )
    if (updateResult is UpdateResult.Success) {
      return InitResult.SuccessWithUpdate(updateResult.hyprMXController)
    } else {
      val backupInitResult =
        initializationController.javascriptUpgradeFailed("Could not go to the new version")
      if (backupInitResult is InitializationResult.Success) {
        initializePlacements(backupInitResult.placementJSON)
        return InitResult.Success
      }
      return InitResult.Failure
    }
  }

  private suspend fun initializePlacements(placementJSON: String) {
    try {
      placementController.initializePlacements(placementJSON, this@HyprMXController)
    } catch (exception: JSONException) {
      HyprMXLog.e("Exception parsing placements")
    }
  }

  override fun getPlacement(placementName: String): Placement {
    isInitialized()
    return placementController.getPlacement(placementName)
  }

  private suspend fun writeCoreJSToFile(coreJS: String): Boolean {
    HyprMXLog.d("writeCoreJSToFile")
    return storageHelper.writeToCoreJSFile(coreJS)
  }

  override fun showAd(placementName: String) {
    val placement = getPlacement(placementName) as PlacementImpl
    launch {
      presentationController.showAd(placement)
    }
  }

  override suspend fun loadAd(placementName: String) = placementController.loadAd(placementName)

  override fun isAdAvailable(placementName: String): Boolean =
    placementController.isAdAvailable(placementName)

  override fun onSharingEndpointReceived(sharingEndpoint: String) {
    launch {
      eventController.setSharingEndpoint(sharingEndpoint)
    }
  }

  fun cleanup() {
    notifyPresentersOfReinitialization()

    preferenceController.stopMonitoring()
    powerSaveModeListener.disable()
    cancelJavascriptExecution()
    scope.cancel()
  }

  private val sdkReInitListeners = mutableListOf<WeakReference<HyprMXReInit>>()
  internal fun registerSDKReInitListener(listener: HyprMXReInit) {
    sdkReInitListeners.add(WeakReference(listener))
  }

  private fun notifyPresentersOfReinitialization() {
    sdkReInitListeners.forEach {
      it.get()?.onSDKReInit()
    }
    sdkReInitListeners.clear()
  }

  fun cancelJavascriptExecution() {
    jsEngine.close()
  }

  override fun onInitializeOMSDK(
    omSdkUrl: String,
    omPartnerName: String,
    omApiVersion: String,
  ) {
    openMeasurementController = OpenMeasurementController(
      applicationContext,
      omPartnerName,
      omApiVersion,
      networkController,
      coroutineScope = scope,
    )
  }

  /**
   * Checks to see if the SDK has been initialized.  This will return true when the
   * initialization controller returns with success.
   */
  fun isInitialized(): Boolean {
    if (hyprMXState != HyprMXState.INITIALIZATION_COMPLETE) {
      HyprMXLog.w(
        "HyprMX is not initialized.  Please call HyprMX.initialize and " +
          "wait for HyprMXInitializationListener.setInitializationComplete to proceed",
      )
      return false
    }
    return true
  }

  private suspend fun initCache() = withContext(Dispatchers.Main) {
    saveInitializationData(distributorId)
  }

  /**
   * Save the data required to create and initialize HyprMXHelper to SharedPreferences.
   *
   * @param distributorId distributor ID provided by publisher
   */
  private suspend fun saveInitializationData(distributorId: String) =
    withContext(Dispatchers.IO) {
      val sharedPreferences =
        applicationContext.getSharedPreferences(HYPRMX_PREFS_INTERNAL, Context.MODE_PRIVATE)

      sharedPreferences.edit {
        putString(PREF_DISTRIBUTOR_ID, distributorId)
      }
    }

  fun getInitializationState(): HyprMXState {
    return hyprMXState
  }

  private fun setInitializationState(status: HyprMXState) {
    HyprMXLog.d("Initialization Status transitioning from $hyprMXState to $status")
    hyprMXState = status
  }

  internal fun setConsentStatus(consentStatus: ConsentStatus) {
    consentController.setConsent(consentStatus)
  }

  internal fun getConsentStatus(): ConsentStatus {
    return consentController.givenConsent
  }

  internal fun getSharedJSVersion(): Int? {
    return mobileJSVersion
  }

  internal fun getPlacements(): Set<Placement> {
    return placementController.placements
  }
}

internal class DefaultHyprMXControllerFactory : HyprMXControllerFactory {
  override fun createHyprMXController(
    context: Context,
    distributorId: String,
    consentStatus: ConsentStatus,
  ): HyprMXController {
    return HyprMXController(
      context.applicationContext,
      distributorId,
      consentStatus = consentStatus,
    )
  }
}

internal interface HyprMXControllerFactory {
  fun createHyprMXController(
    context: Context,
    distributorId: String,
    consentStatus: ConsentStatus,
  ): HyprMXController
}

internal interface HyprMXControllerCleanupListener {
  fun cleanup()
}

internal interface HyprMXReInit {
  fun onSDKReInit()
}
