package com.hyprmx.android.sdk.extensions

import java.io.InputStream
import java.io.OutputStream
import java.nio.charset.Charset

fun InputStream.readTextAndClose(charset: Charset = Charsets.UTF_8): String =
  bufferedReader(charset).use { it.readText() }

/**
 * Copies this stream to the given output stream, returning the number of bytes copied.
 *
 * **Note** It is the caller's responsibility to close both of these resources.
 *
 * @param out The output stream to copy to
 * @param bufferSize The buffer size to use.  Defaults to kotlin.io.DEFAULT_BUFFER_SIZE
 * @param shouldCancel Check to indicate if the stream should be cancelled
 *
 * @return The number of bytes copied
 */
fun InputStream.copyTo(
  out: OutputStream,
  bufferSize: Int = DEFAULT_BUFFER_SIZE,
  shouldCancel: () -> Boolean,
): Long {
  var bytesCopied: Long = 0
  val buffer = ByteArray(bufferSize)
  var bytes = read(buffer)
  while (bytes >= 0 && !shouldCancel()) {
    out.write(buffer, 0, bytes)
    bytesCopied += bytes
    bytes = read(buffer)
  }
  // If we reached -1 then the end of the stream has been reached.
  return bytesCopied
}
