package com.hyprmx.android.sdk.footer

import android.graphics.Bitmap
import android.os.Bundle
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.ImageButton
import android.widget.RelativeLayout
import android.widget.TextView
import androidx.annotation.Keep
import androidx.fragment.app.Fragment
import androidx.fragment.app.FragmentActivity
import com.hyprmx.android.R
import com.hyprmx.android.sdk.utility.CustomLinkMovement
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.URLHandler
import com.hyprmx.android.sdk.utility.Utils
import com.hyprmx.android.sdk.utility.convertDpToPixel

/**
 * The implementation of the footer view.
 *
 * This view is designed to show the following:
 * Nav(back/forward) footerText icon1 icon2
 *
 * All components are optional and if none are available defaults to a 0 height.
 *
 * Adding components converts their visibility to VISIBLE instead of GONE.
 *
 */
@Keep
internal class FooterFragment :
  Fragment(),
  FooterContract.View,
  URLHandler,
  View.OnLayoutChangeListener {

  /**
   * Flag to track current footer height to avoid relayout loops
   */
  private var footerHeight: Int = 0

  /**
   * Flag to indicate we should recalculate the margins to make sure the images
   * are at the bottom of the view
   */
  private var adjustMargins = false

  override lateinit var presenter: FooterContract.Presenter

  private var footer: View? = null

  private var navigationView: View? = null
  private var backButton: ImageButton? = null
  private var forwardButton: ImageButton? = null

  private var textView: TextView? = null

  private var imageButton2: ImageButton? = null
  private var imageButton1: ImageButton? = null

  override fun onCreateView(
    inflater: LayoutInflater,
    container: ViewGroup?,
    savedInstanceState: Bundle?,
  ): View? {
    return inflater.inflate(R.layout.hyprmx_footer, container, false)
  }

  override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
    super.onViewCreated(view, savedInstanceState)
    footer = view.apply {
      navigationView = findViewById(R.id.hyprmx_navigation_view)

      backButton = findViewById(R.id.hyprmx_navigate_back)
      backButton?.isEnabled = false
      backButton?.setOnClickListener { presenter.didTapBack() }

      forwardButton = findViewById(R.id.hyprmx_navigate_forward)
      forwardButton?.isEnabled = false
      forwardButton?.setOnClickListener { presenter.didTapForward() }

      textView = findViewById(R.id.hyprmx_footer_text)
      textView?.movementMethod = CustomLinkMovement(this@FooterFragment)

      imageButton1 = findViewById(R.id.hyprmx_image_icon_1)
      imageButton2 = findViewById(R.id.hyprmx_image_icon_2)

      addOnLayoutChangeListener(this@FooterFragment)
    }
  }

  override fun onDestroyView() {
    super.onDestroyView()
    navigationView = null
    backButton = null
    forwardButton = null
    textView = null
    imageButton2 = null
    imageButton1 = null
    footer = null
  }

  override fun setMinimumHeight(minHeight: Int) {
    footer?.minimumHeight = minHeight.convertDpToPixel(context)
  }

  override fun enableBackNavigation(enable: Boolean) {
    backButton?.isEnabled = enable
  }

  override fun enableForwardNavigation(enable: Boolean) {
    forwardButton?.isEnabled = enable
  }

  override fun enableNavigation(enable: Boolean) {
    navigationView?.visibility = if (enable) View.VISIBLE else View.GONE
    adjustMargins = true
  }

  override fun setIcon1(image: Bitmap, width: Int, height: Int) {
    imageButton1?.let { button ->
      setIconButton(button, image, width, height, 0)
    }
  }

  override fun setIcon2(image: Bitmap, width: Int, height: Int) {
    imageButton2?.let { button ->
      setIconButton(button, image, width, height, 1)
    }
  }

  private fun setIconButton(button: ImageButton, image: Bitmap, width: Int, height: Int, index: Int) {
    button.apply {
      setImageBitmap(scaledImage(image, width, height))
      setOnClickListener { presenter.didTapIcon(index) }
      visibility = View.VISIBLE
    }

    adjustMargins = true
  }

  override fun setText(text: String) {
    textView?.text = Utils.getSpannedText(text)
    textView?.visibility = View.VISIBLE
  }

  override fun onLinkedClicked(url: String) {
    presenter.didTapURL(url)
  }

  override fun setVisible(visible: Boolean) {
    footer?.visibility = if (visible) View.VISIBLE else View.GONE
  }

  override fun setBackgroundColor(color: Int) {
    footer?.setBackgroundColor(color)
  }

  /**
   * Scales a bitmap to
   */
  private fun scaledImage(image: Bitmap, width: Int, height: Int): Bitmap? {
    val pixelWidth = width.convertDpToPixel(context)
    val pixelHeight = height.convertDpToPixel(context)
    return try {
      Bitmap.createScaledBitmap(image, pixelWidth, pixelHeight, false)
    } catch (e: IllegalArgumentException) {
      HyprMXLog.e("IllegalArgumentException thrown for scaling bitmap.", e)
      null
    }
  }

  /**
   * Called when the layout bounds of a view changes due to layout processing.
   *
   * @param v The view whose bounds have changed.
   * @param left The new value of the view's left property.
   * @param top The new value of the view's top property.
   * @param right The new value of the view's right property.
   * @param bottom The new value of the view's bottom property.
   * @param oldLeft The previous value of the view's left property.
   * @param oldTop The previous value of the view's top property.
   * @param oldRight The previous value of the view's right property.
   * @param oldBottom The previous value of the view's bottom property.
   */
  override fun onLayoutChange(
    v: View?,
    left: Int,
    top: Int,
    right: Int,
    bottom: Int,
    oldLeft: Int,
    oldTop: Int,
    oldRight: Int,
    oldBottom: Int,
  ) {
    // We are only monitoring footer height changes
    if (v != footer) return

    val footerHeight = footer?.measuredHeight

    if (this.footerHeight == footerHeight && !adjustMargins) {
      return
    }

    this.footerHeight = footerHeight ?: 0

    adjustMargins = false

    // Adjust the margins of the buttons to "push" them to the bottom of the relative layout
    imageButton1?.let { adjustImageButtonMargins(it) }
    imageButton2?.let { adjustImageButtonMargins(it) }
    adjustNavigationMargins()
  }

  /**
   * Adjust the top margin on the image view to "push" the button down.
   * This is because we use a relative layout and aligning to the bottom stretches the layout
   * to full screen when wrapping content
   */
  private fun adjustImageButtonMargins(imageButton: ImageButton) {
    if (imageButton.visibility != View.GONE) {
      val imageButtonHeight = imageButton.measuredHeight
      val layoutParams = imageButton.layoutParams as RelativeLayout.LayoutParams
      layoutParams.topMargin = footerHeight - imageButtonHeight
      imageButton.layoutParams = layoutParams
    }
  }

  /**
   * Adjust the top margin on the navigation view to "push" the button down.
   * This is because we use a relative layout and aligning to the bottom stretches the layout
   * to full screen when wrapping content
   */
  private fun adjustNavigationMargins() {
    if (navigationView?.visibility != View.GONE) {
      val navigationViewHeight = navigationView?.measuredHeight ?: 0
      val layoutParams = navigationView?.layoutParams as RelativeLayout.LayoutParams
      layoutParams.topMargin = footerHeight - navigationViewHeight
      navigationView?.layoutParams = layoutParams
    }
  }

  /**
   * Returns false if Activity is null or finished, true otherwise.
   */
  override fun isContextInvalid(): Boolean {
    return activity == null || (activity as FragmentActivity).isFinishing
  }
}
