package com.hyprmx.android.sdk.fullscreen

import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.core.ActivityEventInterface
import com.hyprmx.android.sdk.core.CaptureImageNativeInterface
import com.hyprmx.android.sdk.core.DialogNativeInterface
import com.hyprmx.android.sdk.core.LaunchBrowserNativeInterface
import com.hyprmx.android.sdk.core.MraidNativeInterface
import com.hyprmx.android.sdk.core.PermissionRequestNativeInterface
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import java.lang.ref.WeakReference

internal interface FullScreenNativeInterface :
  WebTrafficNativeInterface,
  ActivityEventInterface,
  CaptureImageNativeInterface,
  LaunchBrowserNativeInterface,
  DialogNativeInterface,
  MraidNativeInterface,
  PermissionRequestNativeInterface {

  @RetainMethodSignature
  fun closeAdExperience()

  /**
   * Shows a dialog and closes out the ad when the OK button is pressed.
   */
  @RetainMethodSignature
  fun showErrorDialog(message: String)

  /**
   * Sets the ad closeable.
   */
  @RetainMethodSignature
  fun setClosable(closable: Boolean)

  /**
   * Tells the native presenter to report the power state to the
   * webview
   */
  @RetainMethodSignature
  fun reportPowerState()

  @RetainMethodSignature
  fun startOMSession(args: String)

  @RetainMethodSignature
  fun endOMSession()

  @RetainMethodSignature
  fun dismissOfferCancellationDialog()

  @RetainMethodSignature
  fun showCancelDialog(message: String, exitButton: String, continueButton: String)
}

internal interface WebTrafficNativeInterface {
  @RetainMethodSignature
  fun setForwardButtonEnabled(enabled: Boolean)

  @RetainMethodSignature
  fun setBackButtonEnabled(enabled: Boolean)

  @RetainMethodSignature
  fun showWebTrafficHeader(pageCount: Int) {}

  @RetainMethodSignature
  fun loadWebTrafficPage(url: String, index: Int, userAgent: String?) { }

  @RetainMethodSignature
  fun loadThankYouPage(url: String) { }

  @RetainMethodSignature
  fun skipThankYouPage(completed: Boolean) {}

  @RetainMethodSignature
  fun showNextButton() {}

  @RetainMethodSignature
  fun showFinishButton() {}

  @RetainMethodSignature
  fun startCountDownTimer(time: Int) {}

  @RetainMethodSignature
  fun pauseCountDownTimer() {}

  @RetainMethodSignature
  fun resumeCountDownTimer() {}
}

internal interface FullScreenPresenterInterface {
  suspend fun captureImage()
  suspend fun hyprMXBrowserClosed()
  suspend fun showHyprMXBrowser(viewModelIdentifier: String)
  suspend fun showPlatformBrowser(url: String)
  suspend fun openShareSheet(data: String)
  suspend fun createCalendarEvent(data: String)
  suspend fun storePicture(url: String)
  suspend fun showToast(resourceId: Int)
  suspend fun openOutsideApplication(url: String)
  suspend fun showDialog(jsonConfiguration: String)
  suspend fun useCustomClose(enable: Boolean)
  suspend fun setOrientationProperties(allowOrientationChange: Boolean, forceOrientation: String)
  suspend fun setClosable(closable: Boolean)
  suspend fun permissionRequest(permissions: String, permissionId: Int)
  suspend fun reportPowerState()
  suspend fun startOMSession(sessionData: String)
  suspend fun endOMSession()
  suspend fun showCancelDialog(message: String, exitButton: String, continueButton: String)
  suspend fun dismissOfferCancellationDialog()

  suspend fun closeAdExperience()
  suspend fun showErrorDialog(message: String)

  // WT
  suspend fun showWebTrafficHeader(pageCount: Int) {}
  suspend fun loadWebTrafficPage(url: String, index: Int, userAgent: String?) { }
  suspend fun loadThankYouPage(url: String) { }
  suspend fun skipThankYouPage(completed: Boolean) {}
  suspend fun showNextButton() {}
  suspend fun showFinishButton() {}
  suspend fun startCountDownTimer(time: Int) {}
  suspend fun pauseCountDownTimer() {}
  suspend fun resumeCountDownTimer() {}
  suspend fun setForwardButtonEnabled(enabled: Boolean) {}
  suspend fun setBackButtonEnabled(enabled: Boolean) {}
}

internal class FullScreenPresenterAdapter(presenter: FullScreenPresenterInterface, scope: CoroutineScope) : FullScreenNativeInterface, CoroutineScope by scope {
  private val fullscreenPresenter: WeakReference<FullScreenPresenterInterface> = WeakReference(presenter)

  override fun closeAdExperience() {
    launch { fullscreenPresenter.get()?.closeAdExperience() }
  }

  override fun showErrorDialog(message: String) {
    launch { fullscreenPresenter.get()?.showErrorDialog(message) }
  }

  override fun setClosable(closable: Boolean) {
    launch { fullscreenPresenter.get()?.setClosable(closable) }
  }

  override fun permissionRequest(permissions: String, permissionId: Int) {
    launch { fullscreenPresenter.get()?.permissionRequest(permissions, permissionId) }
  }

  override fun reportPowerState() {
    launch { fullscreenPresenter.get()?.reportPowerState() }
  }

  override fun startOMSession(args: String) {
    launch { fullscreenPresenter.get()?.startOMSession(args) }
  }

  override fun endOMSession() {
    launch { fullscreenPresenter.get()?.endOMSession() }
  }

  override fun setForwardButtonEnabled(enabled: Boolean) {
    launch { fullscreenPresenter.get()?.setForwardButtonEnabled(enabled) }
  }

  override fun setBackButtonEnabled(enabled: Boolean) {
    launch { fullscreenPresenter.get()?.setBackButtonEnabled(enabled) }
  }

  override fun dismissOfferCancellationDialog() {
    launch { fullscreenPresenter.get()?.dismissOfferCancellationDialog() }
  }

  override fun setOrientationProperties(
    allowOrientationChange: Boolean,
    forceOrientationChange: String,
  ) {
    launch { fullscreenPresenter.get()?.setOrientationProperties(allowOrientationChange, forceOrientationChange) }
  }

  override fun showWebTrafficHeader(pageCount: Int) {
    launch { fullscreenPresenter.get()?.showWebTrafficHeader(pageCount) }
  }

  override fun loadWebTrafficPage(url: String, index: Int, userAgent: String?) {
    launch { fullscreenPresenter.get()?.loadWebTrafficPage(url, index, userAgent) }
  }

  override fun loadThankYouPage(url: String) {
    launch { fullscreenPresenter.get()?.loadThankYouPage(url) }
  }

  override fun skipThankYouPage(completed: Boolean) {
    launch { fullscreenPresenter.get()?.skipThankYouPage(completed) }
  }

  override fun showNextButton() {
    launch { fullscreenPresenter.get()?.showNextButton() }
  }

  override fun showFinishButton() {
    launch { fullscreenPresenter.get()?.showFinishButton() }
  }

  override fun startCountDownTimer(time: Int) {
    launch { fullscreenPresenter.get()?.startCountDownTimer(time) }
  }

  override fun pauseCountDownTimer() {
    launch { fullscreenPresenter.get()?.pauseCountDownTimer() }
  }

  override fun resumeCountDownTimer() {
    launch { fullscreenPresenter.get()?.resumeCountDownTimer() }
  }

  override fun showCancelDialog(message: String, exitButton: String, continueButton: String) {
    launch { fullscreenPresenter.get()?.showCancelDialog(message, exitButton, continueButton) }
  }

  override fun openShareSheet(data: String) {
    launch { fullscreenPresenter.get()?.openShareSheet(data) }
  }

  override fun createCalendarEvent(data: String) {
    launch { fullscreenPresenter.get()?.createCalendarEvent(data) }
  }

  override fun storePicture(url: String) {
    launch { fullscreenPresenter.get()?.storePicture(url) }
  }

  override fun showToast(resourceId: Int) {
    launch { fullscreenPresenter.get()?.showToast(resourceId) }
  }

  override fun openOutsideApplication(url: String) {
    launch { fullscreenPresenter.get()?.openOutsideApplication(url) }
  }

  override fun showDialog(jsonConfiguration: String) {
    launch { fullscreenPresenter.get()?.showDialog(jsonConfiguration) }
  }

  override fun useCustomClose(customClose: Boolean) {
    launch { fullscreenPresenter.get()?.useCustomClose(customClose) }
  }

  override fun captureImage() {
    launch { fullscreenPresenter.get()?.captureImage() }
  }

  override fun hyprMXBrowserClosed() {
    launch { fullscreenPresenter.get()?.hyprMXBrowserClosed() }
  }

  override fun showHyprMXBrowser(viewModelIdentifier: String) {
    launch { fullscreenPresenter.get()?.showHyprMXBrowser(viewModelIdentifier) }
  }

  override fun showPlatformBrowser(url: String) {
    launch { fullscreenPresenter.get()?.showPlatformBrowser(url) }
  }
}
