package com.hyprmx.android.sdk.fullscreen

import com.hyprmx.android.sdk.core.ImageCaptureInterface
import com.hyprmx.android.sdk.core.SharedInterface
import com.hyprmx.android.sdk.presentation.ArgumentKey
import com.hyprmx.android.sdk.presentation.ArgumentKey.CONTAINER_HEIGHT
import com.hyprmx.android.sdk.presentation.ArgumentKey.CONTAINER_VISIBILITY
import com.hyprmx.android.sdk.presentation.ArgumentKey.CONTAINER_WIDTH
import com.hyprmx.android.sdk.presentation.ArgumentKey.INDEX
import com.hyprmx.android.sdk.presentation.ArgumentKey.URL
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.presentation.PublishingEvent
import com.hyprmx.android.sdk.presentation.PublishingEvent.AD_COMPLETED
import com.hyprmx.android.sdk.presentation.PublishingEvent.CANCEL_DIALOG_EXIT_PRESSED
import com.hyprmx.android.sdk.presentation.PublishingEvent.CLEAR_BROWSER_REQUEST
import com.hyprmx.android.sdk.presentation.PublishingEvent.CLOSE_TAPPED
import com.hyprmx.android.sdk.presentation.PublishingEvent.CONTAINER_VISIBLE_CHANGED
import com.hyprmx.android.sdk.presentation.PublishingEvent.NATIVE_CLOSE_PRESSED
import com.hyprmx.android.sdk.presentation.PublishingEvent.OM_CUSTOM_DATA
import com.hyprmx.android.sdk.presentation.PublishingEvent.ON_CLOSE
import com.hyprmx.android.sdk.presentation.PublishingEvent.PAYOUT_COMPLETE
import com.hyprmx.android.sdk.presentation.PublishingEvent.RESTORE_STATE
import com.hyprmx.android.sdk.presentation.PublishingEvent.THANK_YOU_URL
import com.hyprmx.android.sdk.presentation.PublishingEvent.UNKNOWN_ERROR_OCCURRED
import com.hyprmx.android.sdk.presentation.PublishingEvent.VIEWING_ID
import com.hyprmx.android.sdk.presentation.PublishingEvent.WEB_TRAFFIC_FINISH_TAPPED
import com.hyprmx.android.sdk.presentation.PublishingEvent.WEB_TRAFFIC_NEXT_TAPPED
import com.hyprmx.android.sdk.presentation.PublishingEvent.WINDOW_OPEN_ATTEMPT
import com.hyprmx.android.sdk.utility.PermissionResult

internal interface FullScreenSharedInterface : WebTrafficSharedInterface, ImageCaptureInterface, OpenMeasurementSharedInterface, SharedInterface {
  var thankYouUrl: String?
  var viewingId: String?
  var payoutComplete: Boolean
  var adCompleted: Boolean
  var closable: Boolean

  fun restoreState()
  fun windowOpenAttempt(url: String)
  fun containerVisibilityChanged(visible: Boolean)
  fun containerSizedChanged(width: Int, height: Int)
  fun onClose()
  fun permissionResult(permissionResults: List<PermissionResult>, requestCode: Int)
  fun clearBrowserRequest()
  fun cancelDialogExitPressed()
  fun nativeClosePressed()
  fun onErrorDialogOKPressed()
  fun onBackButtonPressed()
  fun unknownErrorOccurred(message: String)
  fun internetLossDetected()
}

internal interface WebTrafficSharedInterface {
  fun webTrafficPageLoadTimeout(url: String) {}
  fun webTrafficPageMinTimeComplete() {}
  fun webTrafficNextTapped() {}
  fun webTrafficFinishTapped() {}
  fun closeTapped() {}
  fun onNavBarButtonTapped(index: Int) {}
}

internal interface OpenMeasurementSharedInterface {
  fun openMeasurementCustomData(): String
}

internal class FullScreenSharedConnector(private val eventPublisher: PresentationEventPublisher) : FullScreenSharedInterface {
  override fun webTrafficPageLoadTimeout(url: String) {
    eventPublisher.publishEvent(this::webTrafficPageLoadTimeout.name, mapOf(URL to url))
  }

  override fun webTrafficPageMinTimeComplete() {
    eventPublisher.publishEvent(this::webTrafficPageMinTimeComplete.name)
  }

  override fun webTrafficNextTapped() {
    eventPublisher.publishEvent(WEB_TRAFFIC_NEXT_TAPPED)
  }

  override fun webTrafficFinishTapped() {
    eventPublisher.publishEvent(WEB_TRAFFIC_FINISH_TAPPED)
  }
  override fun closeTapped() {
    eventPublisher.publishEvent(CLOSE_TAPPED)
  }

  override fun imageCaptured(url: String?) {
    eventPublisher.publishEvent(PublishingEvent.IMAGE_CAPTURED, mapOf(URL to (url ?: "")))
  }

  override fun openMeasurementCustomData(): String {
    return eventPublisher.publishEvent(OM_CUSTOM_DATA) as String
  }

  override var viewModelIdentifier: String
    get() = eventPublisher.viewModelIdentifier
    set(value) {
      eventPublisher.viewModelIdentifier = value
    }

  override fun attach(nativeObject: Any) {
    eventPublisher.setPresenter(nativeObject)
  }

  override fun destroy() {
    eventPublisher.destroy()
  }

  override var thankYouUrl: String?
    get() = eventPublisher.getProperty(THANK_YOU_URL) as String?
    set(_) {
      // cannot set
    }

  override var viewingId: String?
    get() = eventPublisher.getProperty(VIEWING_ID) as String?
    set(_) {
      // cannot set
    }

  override var payoutComplete: Boolean
    get() = eventPublisher.getProperty(PAYOUT_COMPLETE) as? Boolean == true

    set(value) {
      eventPublisher.setProperty(PAYOUT_COMPLETE, value)
    }

  override var adCompleted: Boolean
    get() = eventPublisher.getProperty(AD_COMPLETED) as Boolean
    set(_) {
      // only set in shared code
    }

  override var closable: Boolean
    get() = eventPublisher.getProperty(this::closable.name) as Boolean
    set(_) {
      // only set in shared code
    }

  override fun restoreState() {
    eventPublisher.publishEvent(RESTORE_STATE)
  }

  override fun cancelDialogExitPressed() {
    eventPublisher.publishEvent(CANCEL_DIALOG_EXIT_PRESSED)
  }

  override fun nativeClosePressed() {
    eventPublisher.publishEvent(NATIVE_CLOSE_PRESSED)
  }

  override fun onErrorDialogOKPressed() {
    eventPublisher.publishEvent(this::onErrorDialogOKPressed.name)
  }

  override fun onBackButtonPressed() {
    eventPublisher.publishEvent(this::onBackButtonPressed.name)
  }

  override fun unknownErrorOccurred(message: String) {
    eventPublisher.callMethodWithArg(UNKNOWN_ERROR_OCCURRED, message)
  }

  override fun internetLossDetected() {
    eventPublisher.publishEvent(this::internetLossDetected.name)
  }

  override fun windowOpenAttempt(url: String) {
    eventPublisher.publishEvent(WINDOW_OPEN_ATTEMPT, mapOf(URL to url))
  }

  override fun onNavBarButtonTapped(index: Int) {
    eventPublisher.publishEvent(::onNavBarButtonTapped.name, mapOf(INDEX to index))
  }

  override fun containerVisibilityChanged(visible: Boolean) {
    eventPublisher.publishEvent(CONTAINER_VISIBLE_CHANGED, mapOf(CONTAINER_VISIBILITY to true))
  }

  override fun containerSizedChanged(width: Int, height: Int) {
    eventPublisher.publishEvent(
      PublishingEvent.CONTAINER_SIZE_CHANGED,
      mapOf(
        CONTAINER_WIDTH to width,
        CONTAINER_HEIGHT to height,
      ),
    )
  }

  override fun onClose() {
    eventPublisher.publishEvent(ON_CLOSE)
  }

  override fun permissionResult(permissionResults: List<PermissionResult>, requestCode: Int) {
    eventPublisher.publishEvent(PublishingEvent.PERMISSION_RESULT, mapOf(ArgumentKey.PERMISSION_LIST to permissionResults.map { it.toJSON() }, ArgumentKey.PERMISSION_ID to requestCode))
  }

  override fun clearBrowserRequest() {
    eventPublisher.publishEvent(CLEAR_BROWSER_REQUEST)
  }
}
