package com.hyprmx.android.sdk.model

import android.text.TextUtils
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.io.UnsupportedEncodingException
import java.net.URLEncoder
import java.util.ArrayList

/**
 * Returns a url that is used for calling Player(Catalog Frame)
 *
 * @param catalogFrameParams Catalog frame parameters
 * @return Url to call player for the offer to show.
 */
fun getQueryParametersString(catalogFrameParams: String): String {
  return try {
    val queryParamsList = getQueryStringPairsFromJSONObject(null, JSONObject(catalogFrameParams))
    TextUtils.join("&", queryParamsList)
  } catch (e: JSONException) {
    ""
  }
}

@Throws(JSONException::class)
fun getQueryStringPairsFromJSONObject(key: String?, value: Any): List<String> {
  val queryString = ArrayList<String>()

  when (value) {
    is JSONObject -> {
      val iterator = value.keys()
      while (iterator.hasNext()) {
        val nestedKey = iterator.next()
        val nestedObject = value.get(nestedKey)
        queryString.addAll(
          getQueryStringPairsFromJSONObject(
            if (key != null) {
              String.format(
                "%s[%s]",
                key,
                nestedKey,
              )
            } else {
              nestedKey
            },
            nestedObject,
          ),
        )
      }
    }
    is JSONArray -> for (i in 0 until value.length()) {
      queryString.addAll(
        getQueryStringPairsFromJSONObject(
          String.format("%s[]", key),
          value.get(i),
        ),
      )
    }
    else -> key?.let {
      getUrlEncodedValueWithEncoding(key, value.toString())?.let {
        queryString.add(
          it,
        )
      }
    }
  }

  return queryString
}

fun getUrlEncodedValueWithEncoding(field: String, value: String?): String? {
  return try {
    if (value == null || value == "null") {
      URLEncoder.encode(field, "UTF-8")
    } else {
      String.format("%s=%s", URLEncoder.encode(field, "UTF-8"), URLEncoder.encode(value, "UTF-8"))
    }
  } catch (e: UnsupportedEncodingException) {
    null
  }
}
