@file:Suppress("FunctionName")

package com.hyprmx.android.sdk.network

import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.extensions.readTextAndClose
import com.hyprmx.android.sdk.utility.HyprMXLog
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.Job
import kotlinx.coroutines.launch
import org.json.JSONObject

internal class JSNetworkController(
  private val networkController: NetworkController,
  private val jsEngine: JSEngine,
  private val coroutineScope: CoroutineScope,
  private val ioDispatcher: CoroutineDispatcher = Dispatchers.IO,
) : NetworkControllerJSInterface {

  private val jobMap = mutableMapOf<String, Job?>()

  init {
    jsEngine.addJavascriptInterface(this, JS_INTERFACE_NAME)
  }

  @RetainMethodSignature
  override fun request(id: String, url: String, body: String?, method: String, connectionConfiguration: String, callback: String) {
    jobMap[id] = coroutineScope.launch(ioDispatcher) {
      try {
        HyprMXLog.d("Network request $id to $url with method $method")
        when (
          val response = networkController.request(url, body, method, connectionConfiguration.toConnectionConfiguration()) { inputStream ->
            inputStream.use { it.readTextAndClose() }
          }
        ) {
          is NetworkResponse.Success -> {
            HyprMXLog.d("Network response returned with ${response.value}")

            val responseObject = JSONObject()
            responseObject.putMap("headers", response.headers)
            responseObject.put("code", response.code)
            responseObject.put("body", response.value)

            jsEngine.evaluate("$callback('$id', $responseObject);")
          }
          is NetworkResponse.Failure -> {
            val responseObject = JSONObject()
            responseObject.put("code", response.code)
            responseObject.put("error", response.errorMessage)

            jsEngine.evaluate("$callback('$id', $responseObject);")
          }
        }
      } catch (e: IllegalArgumentException) {
        HyprMXLog.e("Error making request to url: " + e.message)
        val responseObject = JSONObject()
        responseObject.put("code", -1)
        responseObject.put("error", "Exception making network request")
        jsEngine.evaluate("$callback('$id', $responseObject);")
      }
      jobMap[id] = null
    }
  }

  @RetainMethodSignature
  override fun abortRequest(id: String) {
    jobMap[id]?.cancel()
    jobMap[id] = null
  }

  companion object {
    const val JS_INTERFACE_NAME = "HYPRNativeNetworkController"
  }
}

internal interface NetworkControllerJSInterface {
  fun request(id: String, url: String, body: String?, method: String, connectionConfiguration: String, callback: String)
  fun abortRequest(id: String)
}

internal fun String.toConnectionConfiguration(): ConnectionConfiguration {
  val jsonObject = JSONObject(this)
  val followRedirects = jsonObject.getBoolean("followRedirect")
  val readTimeout = jsonObject.getInt("readTimeout")
  val connectionTimeout = jsonObject.getInt("connectionTimeout")
  val headers = jsonObject.getJSONObject("headers")
  val headerMap = HashMap<String, String>()
  headers.keys().forEach {
    headerMap[it] = headers.getString(it)
  }
  return ConnectionConfiguration(followRedirects, readTimeout, connectionTimeout, headerMap)
}

/**
 * Converts the List<String> to a string with comma separation and adds it to the JSON Object
 */
internal fun JSONObject.putMap(key: String, map: Map<String, List<String>>) {
  val jsonMap = JSONObject()
  map.forEach { entry ->
    @Suppress("USELESS_ELVIS") // removing this elvis operator will crash the app
    jsonMap.put(entry.key ?: "", entry.value.joinToString())
  }
  this.put(key, jsonMap)
}
