@file:Suppress("FunctionName")

package com.hyprmx.android.sdk.network

import android.annotation.SuppressLint
import android.content.Context
import android.net.ConnectivityManager
import android.net.Network
import android.net.NetworkRequest
import android.os.Build
import androidx.annotation.RequiresApi
import com.hyprmx.android.sdk.utility.HyprMXLog

internal interface NetworkConnectionMonitor {
  val networkConnectionListeners: MutableSet<NetworkConnectionListener>
  fun startMonitoring(listener: NetworkConnectionListener)
  fun stopMonitoring(listener: NetworkConnectionListener)
}

internal interface NetworkConnectionListener {
  fun handleInternetConnectivityChange(active: Boolean)
}

@RequiresApi(Build.VERSION_CODES.N)
internal class DefaultNetworkConnectionMonitorAPI24AndAbove(val context: Context) :
  NetworkConnectionMonitor, ConnectivityManager.NetworkCallback() {
  override val networkConnectionListeners: MutableSet<NetworkConnectionListener> = HashSet()

  override fun startMonitoring(listener: NetworkConnectionListener) {
    networkConnectionListeners.add(listener)
    if (networkConnectionListeners.size == 1) {
      registerForNetworkMonitoring()
    }
  }

  override fun stopMonitoring(listener: NetworkConnectionListener) {
    if (networkConnectionListeners.remove(listener) && networkConnectionListeners.isEmpty()) {
      deregisterForNetworkMonitoring()
    }
  }

  private fun deregisterForNetworkMonitoring() {
    val connectivityManager =
      context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
    try {
      connectivityManager.unregisterNetworkCallback(this)
    } catch (exception: IllegalStateException) {
      HyprMXLog.d("Exception caught de-registering the network monitor")
    }
  }

  private fun registerForNetworkMonitoring() {
    val connectivityManager =
      context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
    connectivityManager.registerDefaultNetworkCallback(this)
  }

  override fun onAvailable(network: Network) {
    for (listener in networkConnectionListeners) {
      listener.handleInternetConnectivityChange(true)
    }
  }

  override fun onLost(network: Network) {
    for (listener in networkConnectionListeners) {
      listener.handleInternetConnectivityChange(false)
    }
  }
}

@RequiresApi(Build.VERSION_CODES.LOLLIPOP)
internal class DefaultNetworkConnectionMonitorAPI21to23(val context: Context) :
  NetworkConnectionMonitor, ConnectivityManager.NetworkCallback() {
  override val networkConnectionListeners: MutableSet<NetworkConnectionListener> = HashSet()

  override fun startMonitoring(listener: NetworkConnectionListener) {
    networkConnectionListeners.add(listener)
    if (networkConnectionListeners.size == 1) {
      registerForNetworkMonitoring()
    }
  }

  override fun stopMonitoring(listener: NetworkConnectionListener) {
    if (networkConnectionListeners.remove(listener) && networkConnectionListeners.isEmpty()) {
      deregisterForNetworkMonitoring()
    }
  }

  private fun deregisterForNetworkMonitoring() {
    val connectivityManager =
      context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
    try {
      connectivityManager.unregisterNetworkCallback(this)
    } catch (exception: IllegalStateException) {
      HyprMXLog.d("Exception caught de-registering the network monitor")
    }
  }

  private fun registerForNetworkMonitoring() {
    val connectivityManager =
      context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
    val builder = NetworkRequest.Builder()
    connectivityManager.registerNetworkCallback(builder.build(), this)
  }

  override fun onAvailable(network: Network) {
    for (listener in networkConnectionListeners) {
      listener.handleInternetConnectivityChange(true)
    }
  }

  override fun onLost(network: Network) {
    for (listener in networkConnectionListeners) {
      listener.handleInternetConnectivityChange(false)
    }
  }
}

@SuppressLint("NewApi")
internal fun NetworkConnectionMonitor(context: Context): NetworkConnectionMonitor =
  if (Build.VERSION.SDK_INT < Build.VERSION_CODES.N) {
    DefaultNetworkConnectionMonitorAPI21to23(context)
  } else {
    DefaultNetworkConnectionMonitorAPI24AndAbove(context)
  }
