package com.hyprmx.android.sdk.om

import android.view.View
import com.hyprmx.android.sdk.tracking.VideoTracking
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.iab.omid.library.jungroup.adsession.AdSession
import com.iab.omid.library.jungroup.adsession.FriendlyObstructionPurpose

/**
 * This class wraps the Open Measurement implementation and allows for concrete implementations
 * that may differ across ad types
 */
internal interface OpenMeasurementAdSession {

  var adSession: AdSession?

  fun initializeAdSession(adView: View)

  fun startSession() {
    adSession?.start()
  }

  fun endSession() {
    adSession?.finish()
    adSession = null
  }

  fun registerAdView(adView: View) {
    adSession?.registerAdView(adView)
  }

  /**
   * Add friendly obstruction which should then be excluded from all
   * ad session viewability calculations.
   * @param friendlyObstruction View to be excluded from all ad session viewability calculations.
   * @param purpose Why this obstruction was necessary
   * @param detailedReason An explanation for why this obstruction is part of the ad experience
   * if not already obvious from the FriendlyObstructionPurpose selected.
   * Must be 50 characters or less and only contain characters `A-z`,`0-9` or space.
   */
  fun addFriendlyObstruction(
    friendlyObstruction: View,
    purpose: FriendlyObstructionPurpose,
    detailedReason: String?,
  ) {
    try {
      adSession?.addFriendlyObstruction(friendlyObstruction, purpose, null)
    } catch (e: IllegalArgumentException) {
      HyprMXLog.e("Error registering obstruction with error msg - ${e.localizedMessage}")
    }
  }

  /**
   * Remove registered friendly obstruction.
   *
   * @param friendlyObstruction View to be removed
   * from the list of registered friendly obstructions.
   */
  fun removeFriendlyObstruction(friendlyObstruction: View) {
    try {
      adSession?.removeFriendlyObstruction(friendlyObstruction)
    } catch (e: IllegalArgumentException) {
      HyprMXLog.e("Error removing registered obstruction with error msg - ${e.localizedMessage}")
    }
  }

  /**
   * Utility method to remove all registered friendly obstructions.
   * This method will have no effect if called after the ad session has finished.
   */
  fun removeAllFriendlyObstruction() {
    adSession?.removeAllFriendlyObstructions()
  }

  fun getOMTrackingHandler(
    duration: Float,
  ): VideoTracking {
    // Empty tracking object
    return object : VideoTracking {}
  }
}
