package com.hyprmx.android.sdk.overlay

import android.content.Intent
import android.content.pm.PackageManager
import android.os.Build
import android.os.Bundle
import android.os.PersistableBundle
import android.view.View
import androidx.activity.addCallback
import androidx.annotation.RequiresApi
import androidx.appcompat.app.AppCompatActivity
import androidx.constraintlayout.widget.ConstraintSet
import androidx.constraintlayout.widget.ConstraintSet.BOTTOM
import androidx.constraintlayout.widget.ConstraintSet.END
import androidx.constraintlayout.widget.ConstraintSet.START
import androidx.constraintlayout.widget.ConstraintSet.TOP
import androidx.core.app.ActivityCompat
import androidx.lifecycle.lifecycleScope
import com.hyprmx.android.R
import com.hyprmx.android.databinding.HyprmxBrowserBinding
import com.hyprmx.android.sdk.core.ImageCaptureInterface
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.mvp.LifecycleEvent.ON_CREATE
import com.hyprmx.android.sdk.mvp.LifecycleEvent.ON_DESTROY
import com.hyprmx.android.sdk.mvp.LifecycleEvent.ON_PAUSE
import com.hyprmx.android.sdk.mvp.LifecycleEvent.ON_RESUME
import com.hyprmx.android.sdk.mvp.LifecycleEvent.ON_START
import com.hyprmx.android.sdk.mvp.LifecycleEvent.ON_STOP
import com.hyprmx.android.sdk.overlay.HyprMXBrowserContract.Presenter
import com.hyprmx.android.sdk.presentation.PresentationFactory
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.PermissionResult
import com.hyprmx.android.sdk.webview.HyprMXWebView
import kotlinx.coroutines.launch

@Suppress("UNUSED_PARAMETER")
internal class HyprMXBrowserActivity : AppCompatActivity(), HyprMXBrowserContract.View, HyprMXOverlay by HyprMXOverlayAdapter(), ImageCapture by ImageCapturer() {
  private var _binding: HyprmxBrowserBinding? = null
  private val binding: HyprmxBrowserBinding
    get() = _binding!!

  private var hyprmxBrowserPresenter: Presenter? = null
  lateinit var placementName: String
  lateinit var viewModelIdentifier: String

  private var hyprmxWebView: HyprMXWebView? = null

  private var presenterFactory: PresentationFactory? = null
    get() = field ?: hyprmxDelegate.hyprMXController?.presenterFactory

  override fun onCreate(savedInstanceState: Bundle?) {
    super.onCreate(savedInstanceState)
    _binding = HyprmxBrowserBinding.inflate(layoutInflater)
    setContentView(binding.root)

    this.overlayContext = this

    intent?.let {
      viewModelIdentifier = it.getStringExtra(VIEW_MODEL_ID_KEY)!!
    }

    hyprmxBrowserPresenter = presenterFactory?.makeHyprMXBrowserPresenter(view = this, viewModelIdentifier = viewModelIdentifier)?.also {
      attachWebView(viewModelIdentifier, it)
    }

    onBackPressedDispatcher.addCallback {
      hyprmxBrowserPresenter?.onBackPressed()
    }
  }

  private fun attachWebView(
    identifier: String,
    presenter: Presenter,
  ) {
    hyprmxWebView = presenterFactory?.makeBrowserWebView(
      this@HyprMXBrowserActivity.applicationContext,
      identifier,
    )

    hyprmxWebView?.id = R.id.hyprmx_webview
    binding.root.addView(hyprmxWebView)

    val set = ConstraintSet()
    set.clone(binding.root)
    set.constrainHeight(R.id.hyprmx_webview, 0)
    set.constrainWidth(R.id.hyprmx_webview, ConstraintSet.MATCH_CONSTRAINT_SPREAD)
    set.connect(R.id.hyprmx_webview, START, R.id.hyprmx_browser_layout, START)
    set.connect(R.id.hyprmx_webview, END, R.id.hyprmx_browser_layout, END)
    set.connect(R.id.hyprmx_webview, BOTTOM, R.id.hyprmx_browser_footer, TOP)
    set.connect(R.id.hyprmx_webview, TOP, R.id.hyprmx_browser_header, BOTTOM)
    set.applyTo(binding.root)

    presenter.view = this@HyprMXBrowserActivity
    presenter.initializeViewState()
    hyprmxWebView?.containingActivity = this

    if (hyprmxWebView?.shouldTakeFocus() == true) {
      hyprmxWebView?.requestFocus()
    }
  }

  fun onNavigateForwardClicked(view: View) {
    hyprmxBrowserPresenter?.onNavigateForwardPressed()
  }

  fun onNavigateBackClicked(view: View) {
    hyprmxBrowserPresenter?.onNavigateBackPressed()
  }

  fun onCloseClicked(view: View) {
    if (isFinishing) {
      HyprMXLog.d("Activity already finishing.  Ignoring close click.")
      return
    }
    hyprmxBrowserPresenter?.onCloseClicked()
  }

  fun onShareSheetClicked(view: View) {
    hyprmxBrowserPresenter?.onSharePressed()
  }

  companion object {
    const val PLACEMENT_NAME_KEY = "placementName"
    const val VIEW_MODEL_ID_KEY = "baseAdId"
  }

  override fun navigateBackEnabled(enabled: Boolean) {
    binding.hyprmxBrowserFooter.hyprmxNavigateBack.isEnabled = enabled
  }

  override fun navigateForwardEnabled(enabled: Boolean) {
    binding.hyprmxBrowserFooter.hyprmxNavigateForward.isEnabled = enabled
  }

  override fun closeButtonEnabled(enabled: Boolean) {
    binding.hyprmxBrowserHeader.hyprmxCloseButton.isEnabled = enabled
  }

  override fun setTitleText(title: String) {
    binding.hyprmxBrowserHeader.hyprmxBrowserTitle.text = title
  }

  override fun finishActivity() {
    finish()
  }

  override fun captureImage() {
    dispatchImageCaptureIntent(this)
  }

  override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
    super.onActivityResult(requestCode, resultCode, data)
    lifecycleScope.launch {
      processActivityResult(this@HyprMXBrowserActivity, requestCode, resultCode, data, hyprmxBrowserPresenter as ImageCaptureInterface)
    }
  }

  override fun requestPermission(permission: Array<String>, requestCode: Int) {
    ActivityCompat.requestPermissions(this, permission, requestCode)
  }

  @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
  override fun onRequestPermissionsResult(
    requestCode: Int,
    permissions: Array<String>,
    grantResults: IntArray,
  ) {
    super.onRequestPermissionsResult(requestCode, permissions, grantResults)
    val resultList = ArrayList<PermissionResult>()
    for (i in permissions.indices) {
      resultList.add(PermissionResult(permissions[i], grantResults[i] == PackageManager.PERMISSION_GRANTED))
    }
    hyprmxBrowserPresenter?.onPermissionResponse(resultList, requestCode)
  }

  override fun onCreate(savedInstanceState: Bundle?, persistentState: PersistableBundle?) {
    hyprmxBrowserPresenter?.onLifecycleEvent(ON_CREATE)
    super.onCreate(savedInstanceState, persistentState)
  }

  override fun onStart() {
    hyprmxBrowserPresenter?.onLifecycleEvent(ON_START)
    super.onStart()
  }

  override fun onResume() {
    hyprmxBrowserPresenter?.onLifecycleEvent(ON_RESUME)
    super.onResume()
  }

  override fun onPause() {
    hyprmxBrowserPresenter?.onLifecycleEvent(ON_PAUSE)
    super.onPause()
  }

  override fun onStop() {
    hyprmxBrowserPresenter?.onLifecycleEvent(ON_STOP)
    super.onStop()
  }

  override fun cleanupPresenters() {
    hyprmxBrowserPresenter = null
    hyprmxWebView = null
  }

  override fun onDestroy() {
    _binding = null
    hyprmxWebView?.cleanup()
    hyprmxWebView = null
    hyprmxBrowserPresenter?.onLifecycleEvent(ON_DESTROY)
    hyprmxBrowserPresenter?.cleanup()
    hyprmxBrowserPresenter = null
    super.onDestroy()
  }
}
