package com.hyprmx.android.sdk.overlay

import android.app.Application
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.Intent
import android.net.Uri
import android.widget.Toast
import androidx.annotation.VisibleForTesting
import androidx.browser.customtabs.CustomTabsIntent
import com.hyprmx.android.sdk.calendar.CalendarEventController
import com.hyprmx.android.sdk.core.ActivityEventInterface
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.presentation.PresentationFactory
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.StorePictureManager
import com.hyprmx.android.sdk.utility.startResolvedActivity

internal interface HyprMXOverlay : ActivityEventInterface {
  var overlayListener: HyprMXOverlayAdapter.OverlayListener?
  var overlayContext: Context?
  var isOverlayPresented: Boolean
  fun hyprMXBrowserClosed()
  fun showHyprMXBrowser(viewModelIdentifier: String)
  fun showPlatformBrowser(url: String)
  suspend fun asyncSavePhoto(url: String)
}

/**
 * Delegates the overlay responsibility.
 * @param overlayContext - If this is not set, then it needs to be set later for the actions to work
 */
internal class HyprMXOverlayAdapter(
  override var overlayContext: Context? = null,
  private val calendarEventController: CalendarEventController = CalendarEventController(),
  private val monitorPlatformOverlay: Boolean = false,
) : HyprMXOverlay {

  @VisibleForTesting
  var presenterFactory: PresentationFactory? = null
    get() = field ?: hyprmxDelegate.hyprMXController?.presenterFactory

  @VisibleForTesting
  var storePictureManager: StorePictureManager? = null
    get() = field ?: hyprmxDelegate.hyprMXController?.storePictureManager

  override var isOverlayPresented = false

  override var overlayListener: OverlayListener? = null

  override fun showHyprMXBrowser(viewModelIdentifier: String) {
    val context = overlayContext ?: return
    val factory = presenterFactory ?: return

    if (monitorPlatformOverlay && isOverlayPresented) {
      HyprMXLog.d("HyprMX Overlay already presented.")
      return
    }

    val browserPresenter =
      factory.makeHyprMXBrowserPresenter(null, viewModelIdentifier)

    val vmi = browserPresenter.viewModelIdentifier

    browserPresenter.launchContext = context
    browserPresenter.stateListener = object : PresentationStateListener {
      override fun onBrowserPresented() {
        overlayListener?.onHyprMXBrowserPresented()
        isOverlayPresented = true
      }

      override fun onBrowserClosed() {
        overlayListener?.onHyprMXBrowserClosed()
        isOverlayPresented = false
      }

      override fun onOutsideAppPresented() {
        overlayListener?.onOutsideAppPresented()
      }
    }
    browserPresenter.onBrowserReady()

    val intent = Intent(context, HyprMXBrowserActivity::class.java)
    intent.putExtra(HyprMXBrowserActivity.VIEW_MODEL_ID_KEY, vmi)
    intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
    try {
      context.startActivity(intent)
      browserPresenter.stateListener?.onBrowserPresented()
    } catch (ex: Exception) {
      HyprMXLog.d("Could not launch activity")
    }
  }

  override fun showPlatformBrowser(url: String) {
    val context = overlayContext ?: return
    if (monitorPlatformOverlay && isOverlayPresented) {
      HyprMXLog.d("HyprMX Overlay already presented.")
      return
    }

    val builder = CustomTabsIntent.Builder()
    val customTabsIntent = builder.build()
    try {
      if (context is Application) {
        customTabsIntent.intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
      }
      customTabsIntent.launchUrl(context, Uri.parse(url))
      overlayListener?.onOutsideAppPresented()
      isOverlayPresented = true
    } catch (ex: ActivityNotFoundException) {
      HyprMXLog.d("Could not find custom tab activity: ${ex.localizedMessage}")
    } catch (ex: Exception) {
      HyprMXLog.d("Could not launch custom tab: ${ex.localizedMessage}")
    }
  }

  override suspend fun asyncSavePhoto(url: String) {
    val context = overlayContext ?: return
    storePictureManager?.storePicture(url, context)
  }

  override fun openOutsideApplication(url: String) {
    val context = overlayContext ?: return
    if (startResolvedActivity(context, url)) {
      overlayListener?.onOutsideAppPresented()
    }
  }

  override fun openShareSheet(data: String) {
    val context = overlayContext ?: return

    val sendIntent: Intent = Intent().apply {
      action = Intent.ACTION_SEND
      putExtra(Intent.EXTRA_TEXT, data)
      type = "text/plain"
    }

    val shareIntent = Intent.createChooser(sendIntent, null)
    context.startActivity(shareIntent)
    overlayListener?.onOutsideAppPresented()
  }

  override fun createCalendarEvent(data: String) {
    val context = overlayContext ?: return
    if (calendarEventController.createCalendarEvent(data, context)) {
      overlayListener?.onOutsideAppPresented()
    }
  }

  override fun storePicture(url: String) {
    // Use asyncSavePhoto
  }

  override fun showToast(resourceId: Int) {
    val context = overlayContext ?: return

    Toast.makeText(
      context,
      context.getString(resourceId),
      Toast.LENGTH_SHORT,
    ).show()
  }

  override fun hyprMXBrowserClosed() {
    overlayListener?.onHyprMXBrowserClosed()
  }

  interface OverlayListener {
    fun onOutsideAppPresented() {}
    fun onHyprMXBrowserPresented() {}
    fun onHyprMXBrowserClosed() {}
  }
}
