package com.hyprmx.android.sdk.placement

import androidx.annotation.Keep

@Keep
interface Placement {
  val name: String?
  var type: PlacementType

  /**
   * Sets the placement expiry listener to check when the ad is no longer available for the placement.
   *
   * @param listener The listener to set. May be null.
   * @return The placement the listener was set on
   */
  fun setPlacementExpiryListener(listener: HyprMXPlacementExpiryListener?)

  /**
   * Checks to see if there is an ad available.
   *
   * @param onResult - The specified function block to where loadAd will callback with the
   * loaded ad's availability.
   */
  fun loadAd(onResult: (isAdAvailable: Boolean) -> Unit) {
    loadAd(object : HyprMXLoadAdListener {
      override fun onAdLoaded(isAdAvailable: Boolean) {
        onResult(isAdAvailable)
      }
    })
  }

  /**
   * Checks to see if there is an ad available.
   *
   * @param listener - loadAd will callback with the loaded ad's availability.
   */
  fun loadAd(listener: HyprMXLoadAdListener)

  /**
   * Checks to see if there is an ad available.
   *
   * @return The loaded ad's availability. True if the ad was successfully loaded, false otherwise.
   */
  suspend fun loadAd(): Boolean

  /**
   * Load the placement with the bid response.
   *
   * @param bidResponse The bid response from the bidding mediator.
   * @param onResult - The specified function block to where loadAd will callback with the
   * loaded ad's availability.
   *
   * @return true if the ad was successfully loaded, false otherwise.
   */
  fun loadAd(bidResponse: String, onResult: (isAdAvailable: Boolean) -> Unit) {
    loadAd(
      bidResponse = bidResponse,
      listener = object : HyprMXLoadAdListener {
        override fun onAdLoaded(isAdAvailable: Boolean) {
          onResult(isAdAvailable)
        }
      },
    )
  }

  /**
   * Load the placement with the bid response.
   *
   * @param bidResponse The bid response from the bidding mediator.
   * @param listener - loadAd will callback with the loaded ad's availability.
   *
   * @return true if the ad was successfully loaded, false otherwise.
   */
  fun loadAd(bidResponse: String, listener: HyprMXLoadAdListener)

  /**
   * Load the placement with the bid response.
   *
   * @param bidResponse The bid response from the bidding mediator.
   *
   * @return true if the ad was successfully loaded, false otherwise.
   */
  suspend fun loadAd(bidResponse: String): Boolean

  /**
   * Checks to see if there is an ad available
   *
   * @return True if an ad can be shown, false otherwise
   */
  fun isAdAvailable(): Boolean

  /**
   * Shows the ad associated with this placement.
   *
   * @param listener The listener to where ad states will be called back.
   */
  fun showAd(listener: HyprMXShowListener)
}
