package com.hyprmx.android.sdk.placement

import com.hyprmx.android.sdk.bidding.BiddingLoadAd
import com.hyprmx.android.sdk.core.HyprMXErrors
import com.hyprmx.android.sdk.utility.HyprMXLog
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.MainScope
import kotlinx.coroutines.launch
import org.json.JSONObject
import kotlin.coroutines.resume
import kotlin.coroutines.suspendCoroutine

/**
 * Data class for PlacementImpl which has an id and type for each instance.
 */
internal class PlacementImpl(
  var placementDelegate: PlacementDelegator,
  val id: Long,
  override var type: PlacementType,
  override val name: String,
  val scope: CoroutineScope = MainScope(),
) : Placement, CoroutineScope by scope {

  internal var placementExpiryListener: HyprMXPlacementExpiryListener? = null
  internal var showListener: HyprMXShowListener? = null
  override fun setPlacementExpiryListener(listener: HyprMXPlacementExpiryListener?) {
    placementExpiryListener = listener
  }

  override fun loadAd(listener: HyprMXLoadAdListener) {
    internalLoadAd(listener::onAdLoaded)
  }

  override fun loadAd(bidResponse: String, listener: HyprMXLoadAdListener) {
    internalLoadAd(bidResponse, listener::onAdLoaded)
  }

  private fun internalLoadAd(onResult: (isAdAvailable: Boolean) -> Unit) {
    val isInvalid = type == PlacementType.INVALID

    if (isInvalid) {
      HyprMXLog.e("loadAd was called on an invalid placement!")
      onResult(false)
    } else {
      scope.launch {
        placementDelegate.loadAd(name).also(onResult)
      }
    }
  }

  private fun internalLoadAd(bidResponse: String, onResult: (isAdAvailable: Boolean) -> Unit) {
    scope.launch {
      placementDelegate.loadBid(name, bidResponse).also(onResult)
    }
  }

  override suspend fun loadAd() =
    suspendCoroutine { continuation ->
      loadAd(continuation::resume)
    }

  override suspend fun loadAd(bidResponse: String) =
    suspendCoroutine { continuation ->
      loadAd(bidResponse, continuation::resume)
    }

  override fun isAdAvailable(): Boolean {
    return placementDelegate.isAdAvailable(name)
  }

  override fun showAd(listener: HyprMXShowListener) {
    if (type == PlacementType.INVALID) {
      HyprMXLog.e("showAd called on an invalid placement!")
      listener.onAdStarted(this)
      listener.onAdDisplayError(this, HyprMXErrors.PLACEMENT_DOES_NOT_EXIST)
      listener.onAdClosed(this, false)
    } else {
      showListener = listener
      placementDelegate.showAd(name)
    }
  }

  companion object {

    fun fromJson(placementDelegate: PlacementDelegator, jsonString: String): PlacementImpl {
      val json = JSONObject(jsonString)

      val id = json.optLong("id")
      val type = json.optString("type")
      val name = json.optString("name")
      val placementType = PlacementType.fromString(type)

      return PlacementImpl(placementDelegate, id, placementType, name)
    }
  }

  interface PlacementDelegator : BiddingLoadAd {
    fun showAd(placementName: String)
    suspend fun loadAd(placementName: String): Boolean
    fun isAdAvailable(placementName: String): Boolean
  }
}

internal fun invalidPlacement(placementName: String): Placement =
  PlacementImpl(
    object : PlacementImpl.PlacementDelegator {
      override fun showAd(placementName: String) {}
      override suspend fun loadAd(placementName: String) = false
      override suspend fun loadBid(placementName: String, bidResponseData: String): Boolean = false
      override fun isAdAvailable(placementName: String) = false
    },
    0,
    PlacementType.INVALID,
    placementName,
  )
