@file:Suppress("FunctionName")

package com.hyprmx.android.sdk.powersavemode

import android.annotation.TargetApi
import android.content.BroadcastReceiver
import android.content.Context
import android.content.Intent
import android.content.IntentFilter
import android.os.Build
import android.os.PowerManager
import android.os.PowerManager.ACTION_POWER_SAVE_MODE_CHANGED
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleObserver
import androidx.lifecycle.OnLifecycleEvent
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.sendCustomEvent
import com.hyprmx.android.sdk.webview.HyprMXWebViewContract
import kotlinx.coroutines.CoroutineName
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import kotlinx.coroutines.plus

internal interface PowerSaveModeListener : LifecycleObserver {
  var isPowerSaveMode: Boolean
  fun enable()
  fun disable()
  fun sendPowerStateEvent(webview: HyprMXWebViewContract.InternalView)
}

internal fun PowerSaveModeListener(
  context: Context,
  powerManager: PowerManager,
  scope: CoroutineScope,
): PowerSaveModeListener = DefaultPowerSaveModeListener(context, powerManager, scope)

@TargetApi(Build.VERSION_CODES.LOLLIPOP)
internal class DefaultPowerSaveModeListener(
  private val context: Context,
  private val powerManager: PowerManager,
  private val scope: CoroutineScope,
) :
  BroadcastReceiver(),
  PowerSaveModeListener,
  CoroutineScope by scope + CoroutineName("DefaultPowerSaveModeListener") {

  val filter = IntentFilter().apply {
    addAction(ACTION_POWER_SAVE_MODE_CHANGED)
  }

  private var enabled = false
  private var webview: HyprMXWebViewContract.InternalView? = null

  override var isPowerSaveMode = false
    set(value) {
      HyprMXLog.d("isPowerSaveMode set to $value")
      field = value
    }

  init {
    launch { isPowerSaveMode = powerManager.isPowerSaveMode }
    enable()
  }

  override fun onReceive(context: Context?, intent: Intent?) {
    HyprMXLog.d("$ACTION_POWER_SAVE_MODE_CHANGED event received")
    launch {
      isPowerSaveMode = powerManager.isPowerSaveMode
      webview?.let { sendPowerStateEvent(it) }
    }
  }

  override fun sendPowerStateEvent(webview: HyprMXWebViewContract.InternalView) {
    launch {
      if (enabled) {
        HyprMXLog.d("sending hyprDevicePowerState event...")
        this@DefaultPowerSaveModeListener.webview = webview
        val lowPowerState = if (isPowerSaveMode) "low_power_mode_on" else "low_power_mode_off"
        sendCustomEvent(webview, "hyprDevicePowerState", lowPowerState)
      }
    }
  }

  override fun enable() {
    HyprMXLog.d("Enabling PowerSaveModeListener $this")
    enabled = true
    try {
      context.registerReceiver(this, filter)
    } catch (e: IllegalArgumentException) {
      HyprMXLog.e("Receiver $this is already registered!")
    }
  }

  @OnLifecycleEvent(Lifecycle.Event.ON_DESTROY)
  fun removeWebview() {
    webview = null
  }

  override fun disable() {
    HyprMXLog.d("Disabling PowerSaveModeListener $this")
    enabled = false
    try {
      context.unregisterReceiver(this)
    } catch (e: IllegalArgumentException) {
      HyprMXLog.e("Receiver $this cannot be unregistered!")
    }
  }
}
