package com.hyprmx.android.sdk.preload

import android.content.Context
import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.ImageCacheManagerIf
import com.hyprmx.android.sdk.webview.HyprMXWebView
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch

/**
 * This class is the javascript interface class that starts all preloading for SDK.
 */
internal class PreloadController(
  jsEngine: JSEngine,
  private val imageCacheManager: ImageCacheManagerIf,
  private val applicationContext: Context,
  private val preloadedWebViewMap: MutableMap<String, HyprMXWebView> = mutableMapOf(),
  private val coroutineScope: CoroutineScope,
  preloadControllerSharedInterface: PreloadControllerSharedInterface = PreloadControllerSharedConnector(jsEngine),
) : PreloadControllerIf, PreloadControllerSharedInterface by preloadControllerSharedInterface, CoroutineScope by coroutineScope {

  init {
    addJavascriptInterface(this)
  }

  @RetainMethodSignature
  override fun preloadUIImage(
    url: String,
    height: Int,
    width: Int,
    scale: Float?,
    tiled: Boolean?,
    x: Int?,
    y: Int?,
  ) {
    launch { imageCacheManager.downloadAndCacheImage(url) }
  }

  @RetainMethodSignature
  override fun preloadPortraitImage(
    portraitUrl: String,
    height: Int,
    width: Int,
    fillScreenWidth: Boolean?,
  ) {
    launch { imageCacheManager.downloadAndCacheImage(portraitUrl) }
  }

  override fun storeInstance(viewModelIdentifier: String) {
    HyprMXLog.d("storeInstance for $viewModelIdentifier")
    launch {
      val preloadedWebView = HyprMXWebView(context = applicationContext)
      preloadedWebView.initialize(viewModelIdentifier)
      preloadedWebViewMap[viewModelIdentifier] = preloadedWebView
    }
  }

  override fun removeInstance(viewModelIdentifier: String) {
    HyprMXLog.d("removeInstance for $viewModelIdentifier")
    preloadedWebViewMap.remove(viewModelIdentifier)
    nativePreloadDidRemove(viewModelIdentifier)
  }

  override fun getPreloadedWebView(viewModelIdentifier: String): HyprMXWebView? {
    HyprMXLog.d("getPreloadedWebView for $viewModelIdentifier")
    if (preloadedWebViewMap.containsKey(viewModelIdentifier)) {
      HyprMXLog.d("Preloaded WebView found for $viewModelIdentifier")
      val preloaded = preloadedWebViewMap[viewModelIdentifier]
      removeInstance(viewModelIdentifier)
      return preloaded
    }
    return null
  }
}

internal interface PreloadControllerIf : PreloadNativeInterface, PreloadedWebView
