@file:Suppress("FunctionName")

package com.hyprmx.android.sdk.presentation

import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.presentation.PresentationController.Companion.JS_BIND_BANNER_VIEW_MODEL
import com.hyprmx.android.sdk.presentation.PresentationController.Companion.JS_DESTROY_VIEW_MODEL
import com.hyprmx.android.sdk.presentation.PresentationController.Companion.JS_SET_PRESENTER
import com.hyprmx.android.sdk.presentation.PresentationController.Companion.JS_SET_WEBVIEW_PRESENTER
import org.json.JSONArray
import org.json.JSONObject

internal interface PresentationEventPublisher : ViewModelIdentifier {
  fun publishEvent(eventName: String, argMap: Map<String, Any?>? = null): Any?
  fun callMethodWithArg(method: String, arg: String? = null): Any?
  fun <T> getProperty(property: String): T
  fun <T> setProperty(property: String, value: T)
  fun setPresenter(nativeObject: Any)
  fun setWebViewPresenter(nativeObject: Any)
  fun destroy()
}

internal interface ViewModelIdentifier {
  var viewModelIdentifier: String
}

internal class HyprMXPresentationEventPublisher(
  private val jsEngine: JSEngine,
  override var viewModelIdentifier: String,
  private val destroyScript: String? = null,
) : PresentationEventPublisher {

  private var nativeId = ""
  override fun publishEvent(eventName: String, argMap: Map<String, Any?>?): Any? {
    val filteredMap = argMap?.filter { it.value != null }
    val jsonMap = if (filteredMap != null) {
      JSONArray().put(JSONObject(filteredMap)).toString()
    } else {
      "[]"
    }

    val script = "${PresentationController.JS_PUBLISH_EVENT}('$viewModelIdentifier', '$eventName', $jsonMap);"
    return jsEngine.evaluateScriptForResponse(script)
  }

  override fun callMethodWithArg(method: String, arg: String?): Any? {
    val script = if (arg !== null) {
      "${PresentationController.JS_GET_VIEW_MODEL}('$viewModelIdentifier').$method('$arg');"
    } else {
      "${PresentationController.JS_GET_VIEW_MODEL}('$viewModelIdentifier').$method();"
    }
    return jsEngine.evaluateScriptForResponse(script)
  }

  override fun <T> getProperty(property: String): T {
    val script = "${PresentationController.JS_GET_VIEW_MODEL}('$viewModelIdentifier').$property;"
    return jsEngine.evaluateScriptForResponse(script) as T
  }

  override fun <T> setProperty(property: String, value: T) {
    val script = "${PresentationController.JS_GET_VIEW_MODEL}('$viewModelIdentifier').$property = $value;"
    jsEngine.evaluateScriptForResponse(script)
  }

  override fun setPresenter(nativeObject: Any) {
    nativeId = "HyprMXNative${nativeObject.hashCode()}"
    jsEngine.addJavascriptInterface(nativeObject, nativeId)
    val script = "${PresentationController.JS_GET_VIEW_MODEL}('$viewModelIdentifier').$JS_SET_PRESENTER($nativeId);"
    jsEngine.evaluateScriptForResponse(script)
  }

  override fun setWebViewPresenter(nativeObject: Any) {
    nativeId = "HyprMXNative${nativeObject.hashCode()}"
    jsEngine.addJavascriptInterface(nativeObject, nativeId)
    val script = "${PresentationController.JS_GET_VIEW_MODEL}('$viewModelIdentifier').$JS_SET_WEBVIEW_PRESENTER($nativeId);"
    jsEngine.evaluateScriptForResponse(script)
  }

  override fun destroy() {
    jsEngine.removeJavascriptInterface(nativeId)
    destroyScript?.let {
      jsEngine.evaluateScriptForResponse("$destroyScript('$viewModelIdentifier');")
    }
  }
}

internal fun WebViewPresentationEventPublisher(
  jsEngine: JSEngine,
  viewModelIdentifier: String,
): PresentationEventPublisher = HyprMXPresentationEventPublisher(
  jsEngine = jsEngine,
  destroyScript = JS_DESTROY_VIEW_MODEL,
  viewModelIdentifier = viewModelIdentifier,
)

internal fun FullScreenPresentationEventPublisher(
  jsEngine: JSEngine,
  viewModelIdentifier: String,
): PresentationEventPublisher = HyprMXPresentationEventPublisher(
  jsEngine = jsEngine,
  viewModelIdentifier = viewModelIdentifier,
)

internal fun BannerPresentationEventPublisher(
  jsEngine: JSEngine,
  placementName: String,
): PresentationEventPublisher {
  val viewModelIdentifier = jsEngine.evaluateScriptForResponse("$JS_BIND_BANNER_VIEW_MODEL('$placementName');") as String?
  return HyprMXPresentationEventPublisher(
    jsEngine = jsEngine,
    destroyScript = JS_DESTROY_VIEW_MODEL,
    viewModelIdentifier = viewModelIdentifier ?: "",
  )
}

internal fun BrowserEventPublisher(
  jsEngine: JSEngine,
  viewModelIdentifier: String,
): PresentationEventPublisher = HyprMXPresentationEventPublisher(
  jsEngine = jsEngine,
  destroyScript = JS_DESTROY_VIEW_MODEL,
  viewModelIdentifier = viewModelIdentifier,
)

/**
 * The list of keys we can use in the JSON object map for arguments.
 * These should match iOS.
 */
internal object ArgumentKey {
  const val URL = "url"
  const val INDEX = "index"
  const val MIME_TYPE = "mimeType"
  const val SCHEME = "scheme"
  const val IS_MAIN_FRAME = "isMainFrame"
  const val MESSAGE = "message"
  const val SHOW_CANCEL = "showCancel"
  const val METHOD_NAME = "name"
  const val BODY = "body"
  const val PERMISSIONS = "permissions"
  const val ERROR_MESSAGE = "errorMessage"
  const val ERROR_CODE = "errorCode"
  const val EVENT = "event"
  const val ALLOW_ORIENTATION_CHANGE = "allowOrientationChange"
  const val FORCE_ORIENTATION_CHANGE = "forceOrientationChange"

  // WebView history
  const val CURRENT_INDEX = "currentIndex"
  const val CURRENT_URL = "currentUrl"
  const val CURRENT_HOST = "currentHost"
  const val HISTORY = "history"
  const val CAN_NAVIGATE_FORWARD = "canNavigateForward"
  const val CAN_NAVIGATE_BACK = "canNavigateBack"
  const val CURRENT_TITLE = "currentTitle"

  const val DEFINED_SIZE = "definedSize"
  const val ACTUAL_SIZE = "actualSize"
  const val BID_RESPONSE = "bidResponse"

  const val CONTAINER_WIDTH = "width"
  const val CONTAINER_HEIGHT = "height"

  const val PARENT_VIEW_ATTACHED = "parentView"

  const val CONTAINER_VISIBILITY = "visible"

  const val PERMISSION_ID = "permissionId"
  const val PERMISSION_LIST = "permissions"

  const val ACCEPT_TYPES = "acceptTypes"

  const val PROGRESS_STATE = "state"
  const val CLOSED_ACTION = "action"

  // Visibility Events
  const val IS_SHOWN = "isShown"
  const val VISIBLE_HEIGHT = "visibleHeight"
  const val VISIBLE_WIDTH = "visibleWidth"
  const val ACTUAL_HEIGHT = "actualHeight"
  const val ACTUAL_WIDTH = "actualWidth"
  const val FULLY_VISIBLE = "fullyVisible"
  const val PARTIALLY_VISIBLE = "partiallyVisible"
  const val FULLY_OFFSCREEN = "fullyOffscreen"
  const val ON_SCREEN_X = "onScreenX"
  const val ON_SCREEN_Y = "onScreenY"
  const val ALPHA = "alpha"
  const val PARENT_PASSES_ALPHA_THRESHOLD = "parentAlphaPassesThreshold"
}

/**
 * The list of events we can publish to the presenter.
 * This should match the list we send on iOS, or be commented on the difference
 */
internal object PublishingEvent {
  const val SHOULD_TAKE_FOCUS = "shouldTakeFocus"
  const val SHOULD_LOAD_ABOUT_BLANK = "shouldLoadAboutBlank"

  const val URL_NAVIGATION_ATTEMPT = "urlNavigationAttempt"
  const val WINDOW_OPEN_ATTEMPT = "windowOpenAttempt"

  const val SHOULD_REDIRECT_URL = "shouldRedirectURL"
  const val ON_PAGE_STARTED = "onPageStarted"
  const val ON_PAGE_FINISHED = "onPageFinished"
  const val ON_RECEIVED_ERROR = "onReceivedError"
  const val ON_HISTORY_CHANGED = "onHistoryChanged"
  const val WEB_VIEW_CONFIGURATION = "getWebViewConfigurationString"

  const val WEB_VIEW_SIZE_CHANGED = "webViewSizeChange"

  const val SHOULD_INTERCEPT_REQUEST = "shouldInterceptRequest"
  const val PERMISSION_REQUEST = "permissionRequest"
  const val PERMISSION_RESULT = "permissionResponse"
  const val IMAGE_CAPTURED = "imageCaptured"

  const val LOAD_AD = "loadAd"

  const val CLEAR_BROWSER_REQUEST = "clearBrowserRequest"

  const val ON_WEBVIEW_CRASH = "onWebViewCrash"
  const val ON_JS_MESSAGE = "onJSMessage"
  const val ON_LOAD_DATA = "onLoadData"

  const val ON_LIFECYCLE_EVENT = "onLifecycleEvent"

  const val JAVASCRIPT_ALERT_ATTEMPTED = "javaScriptAlertAttempt"

  const val ON_SHOW_FILE_CHOOSER = "openFileChooser"

  // Browser Navigation
  const val NAVIGATE_BACK_PRESSED = "onNavigateBackPressed"
  const val NAVIGATE_FORWARD_PRESSED = "onNavigateForwardPressed"
  const val ON_CLOSE = "onClose"
  const val SHARE_SHEET_PRESSED = "onSharePressed"

  const val ON_BROWSER_READY = "onBrowserReady"

  // Container Events
  const val CONTAINER_SIZE_CHANGED = "containerSizeChange"
  const val CONTAINER_VISIBLE_CHANGED = "containerVisibleChange"

  const val ON_VISIBLE_EVENT = "onVisibleEvent"
  const val ON_BANNER_CLICKED = "onBannerClicked"
  const val PARENT_VISIBILITY_CHANGED = "onParentViewChangeEvent"

  // Open Measurement
  const val OM_CUSTOM_DATA = "openMeasurementCustomData"

  // Fullscreen Properties
  const val RECOVERY_PARAMS = "recoveryParams"
  const val PAGE_READY = "isPageReady"
  const val CLOSE_TAPPED = "didTapClose"
  const val THANK_YOU_URL = "thankYouUrl"
  const val VIEWING_ID = "viewingId"
  const val RESTORE_STATE = "restoreState"
  const val PAYOUT_COMPLETE = "payoutComplete"
  const val AD_COMPLETED = "adCompleted"
  const val CLOSABLE = "closable"

  const val ON_ERROR_DIALOG_OK_PRESSED = "onErrorDialogOKPressed"
  const val NATIVE_CLOSE_PRESSED = "nativeClosePressed"
  const val CANCEL_DIALOG_EXIT_PRESSED = "cancelDialogExitPressed"
  const val UNKNOWN_ERROR_OCCURRED = "unknownErrorOccurred"
  const val INTERNET_LOSS_DETECTED = "internetLossDetected"

  // WT Events
  const val WEB_TRAFFIC_NEXT_TAPPED = "didTapNext"
  const val WEB_TRAFFIC_FINISH_TAPPED = "didTapFinish"

  // WebView Properties
  const val AD_BASE_URL = "adBaseUrl"
  const val AD_BASE_ENCODING = "adBaseEncoding"
  const val AD_BASE_MIME = "adBaseMime"
  const val USER_AGENT = "userAgent"
  const val QUERY_PARAMS = "queryParams"
}
