@file:Suppress("FunctionName")

package com.hyprmx.android.sdk.utility

import android.content.Context
import android.net.ConnectivityManager
import android.net.NetworkCapabilities
import android.net.NetworkInfo
import android.os.Build
import android.telephony.TelephonyManager
import androidx.annotation.RequiresApi
import androidx.core.content.getSystemService

interface HyprMXConnection {
  /**
   * Returns network connection type of device.
   * Called on both main thread and background thread.
   *
   * @return Connection type in string
   */
  fun getConnectionTypeForRequest(): String

  @RequiresApi(Build.VERSION_CODES.M)
  fun getConnectionTypeForRequestApi23AndAbove(): String
  fun getConnectionTypeForRequestPre23(): String
}

/**
 * Determines the transport type in use
 */
internal class DefaultHyprMXConnection(private val context: Context) : HyprMXConnection {

  /**
   * Returns network connection type of device.
   * Called on both main thread and background thread.
   *
   * @return Connection type in string
   */
  override fun getConnectionTypeForRequest(): String {
    val connectionType = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
      getConnectionTypeForRequestApi23AndAbove()
    } else {
      getConnectionTypeForRequestPre23()
    }

    HyprMXLog.d("Connection Type:$connectionType")
    return connectionType
  }

  @RequiresApi(Build.VERSION_CODES.M)
  override fun getConnectionTypeForRequestApi23AndAbove(): String {
    var networkTransport: String? = null
    // Use ConnectivityManager to check whether you are actually connected to the Internet,
    // and if so, what type of connection is in place.
    // val cm = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
    val cm: ConnectivityManager? = context.getSystemService()
    val networkCapabilities = cm?.getNetworkCapabilities(cm.activeNetwork)
    // NET_CAPABILITY_VALIDATED -
    //    Indicates that connectivity on this network was successfully validated.
    // NET_CAPABILITY_INTERNET -
    //    Indicates that this network should be able to reach the internet.
    if (networkCapabilities?.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET) == true &&
      networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)
    ) {
      when {
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) ->
          networkTransport =
            "WIFI"
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) -> {
          val telephony: TelephonyManager? = context.getSystemService()
          networkTransport = if (telephony != null && telephony.isNetworkRoaming) {
            "roaming"
          } else {
            cm.activeNetworkInfo?.subtypeName
          }
        }
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_BLUETOOTH) ->
          networkTransport =
            "Bluetooth"
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET) ->
          networkTransport =
            "Ethernet"
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_VPN) ->
          networkTransport =
            "VPN"
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI_AWARE) ->
          networkTransport =
            "Wifi Aware"
        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_LOWPAN) ->
          networkTransport =
            "Lowpan"
      }
    }
    return networkTransport ?: "Undetermined"
  }

  override fun getConnectionTypeForRequestPre23(): String {
    var networkTransport: String? = null
    val cm: ConnectivityManager? = context.getSystemService()
    val activeNetwork: NetworkInfo? = cm?.activeNetworkInfo
    val isConnected: Boolean = activeNetwork?.isConnected == true
    val isWiFi: Boolean = activeNetwork?.type == ConnectivityManager.TYPE_WIFI
    val isMob: Boolean = activeNetwork?.type == ConnectivityManager.TYPE_MOBILE
    if (isConnected) {
      if (isWiFi) {
        networkTransport = "WIFI"
      } else if (isMob) {
        val telephony: TelephonyManager? = context.getSystemService()
        networkTransport = if (telephony != null && telephony.isNetworkRoaming) {
          "roaming"
        } else {
          activeNetwork?.subtypeName
        }
      }
    }
    return networkTransport ?: "Undetermined"
  }
}

fun HyprMXConnection(context: Context): HyprMXConnection = DefaultHyprMXConnection(context)
