package com.hyprmx.android.sdk.utility

import android.content.Context
import android.util.Log
import androidx.annotation.Keep
import com.hyprmx.android.sdk.core.HyprMXController
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

/**
 * Provide different type of logging methods.
 * This class captures all the log messages before logging messages.
 */
@Keep
object HyprMXLog {

  private const val VERBOSE_LOG = 0
  private const val DEBUG_LOG = 1
  private const val INFO_LOG = 2
  private const val WARNING_LOG = 3
  private const val ERROR_LOG = 4

  private const val SEPARATOR_STR = " ---- @ "

  private const val MAX_LOG_SIZE = 800
  private val logMessage = StringBuffer()

  internal const val PREF_ALL_LOGGING_ENABLED = "ALL_LOGGING_ENABLED_PREF"
  internal var ioDispatcher = Dispatchers.IO

  /**
   * Indicates if we want to print a log message to logcat.
   */
  private var isLoggingToLogcatEnabled = false

  /**
   * Flag to indicate we should enable logging regardless of what is set in
   * isLoggingToLogcatEnabled.  This flag will be set by the server response
   * in initialization
   */
  private var isLoggingOverrideFromServerEnabled = false

  /**
   * Check to indicate whether we should log to logcat
   */
  internal fun isLoggingEnabled(): Boolean =
    isLoggingOverrideFromServerEnabled || isLoggingToLogcatEnabled

  /**
   * Log to system out.  This is useful for logging in unit tests.
   */
  private var logToSystemOut = false

  internal fun logToSystemOut(enable: Boolean) {
    logToSystemOut = enable
  }

  /**
   * Sends a [android.util.Log.VERBOSE] log message if HyprMXProperties.debug is true.
   *
   * @param message The message you would like logged.
   */
  @JvmStatic
  fun v(message: String?) {
    message?.let {
      captureLog(it)
      if (isLoggingEnabled()) {
        out(it, VERBOSE_LOG)
      }
    }
  }

  /**
   * Sends a [android.util.Log.DEBUG] log message if HyprMXProperties.debug is true.
   *
   * @param message The message you would like logged.
   */
  @JvmStatic
  fun d(message: String?) {
    message?.let {
      captureLog(it)
      if (isLoggingEnabled()) {
        out(it, DEBUG_LOG)
      }
    }
  }

  /**
   * Sends a [android.util.Log.DEBUG] log message if HyprMXProperties.debug is true.
   *
   * @param message The message you would like logged.
   */
  @JvmStatic
  fun d(tag: String, message: String) {
    captureLog(message)
    if (isLoggingEnabled()) {
      out(tag, message, DEBUG_LOG)
    }
  }

  /**
   * Sends a [android.util.Log.INFO] log message.
   *
   * @param message The message you would like logged.
   */
  @JvmStatic
  fun i(tag: String, message: String) {
    captureLog(message)
    out(tag, message, INFO_LOG)
  }

  /**
   * Sends a [android.util.Log.INFO] log message.
   *
   * @param message The message you would like logged.
   */
  @JvmStatic
  fun i(message: String?) {
    message?.let {
      captureLog(it)
      out(it, INFO_LOG)
    }
  }

  /**
   * Sends a [android.util.Log.WARN] log message.
   *
   * @param message The message you would like logged.
   */
  @JvmStatic
  fun w(message: String?) {
    message?.let {
      captureLog(it)
      out(it, WARNING_LOG)
    }
  }

  /**
   * Sends a [android.util.Log.WARN] log message.
   *
   * @param tag Tag to include with the log
   * @param message The message you would like logged.
   */
  @JvmStatic
  fun w(tag: String, message: String) {
    captureLog(message)
    out(tag, message, WARNING_LOG)
  }

  /**
   * Sends a [android.util.Log.ERROR] log message.
   *
   * @param message The message you would like logged.
   */
  @JvmStatic
  fun e(message: String?) {
    message?.let {
      captureLog(it)
      out(it, ERROR_LOG)
    }
  }

  /**
   * Sends a [android.util.Log.ERROR] log message.
   *
   * @param message The message you would like logged.
   */
  @JvmStatic
  fun e(tag: String, message: String) {
    captureLog(message)
    out(tag, message, ERROR_LOG)
  }

  /**
   * Sends a [android.util.Log.ERROR] log message.
   *
   * @param throwable An exception to log.
   */
  @JvmStatic
  fun e(throwable: Throwable) {
    val errorMessage = Log.getStackTraceString(throwable)
    captureLog(errorMessage)
    out(errorMessage, ERROR_LOG)
  }

  /**
   * Sends a [android.util.Log.INFO] log message.
   *
   * @param message The message you would like logged.
   * @param throwable An exception to log
   */
  fun i(message: String, throwable: Throwable) {
    captureLog(message, throwable)
    out(message + "\n" + Log.getStackTraceString(throwable), INFO_LOG)
  }

  /**
   * Sends a [android.util.Log.WARN] log message.
   *
   * @param message The message you would like logged.
   * @param throwable An exception to log
   */
  fun w(message: String, throwable: Throwable) {
    captureLog(message, throwable)
    out(message + "\n" + Log.getStackTraceString(throwable), WARNING_LOG)
  }

  /**
   * Sends a [android.util.Log.ERROR] log message.
   *
   * @param message The message you would like logged.
   * @param throwable An exception to log
   */
  @JvmStatic
  fun e(message: String, throwable: Throwable) {
    captureLog(message, throwable)
    out(message + "\n" + Log.getStackTraceString(throwable), ERROR_LOG)
  }

  private fun out(msg: String, logType: Int) {
    val thread = Thread.currentThread()
    val stack = thread.stackTrace
    val element = stack[4]

    val start = element.className.lastIndexOf(".") + 1
    var end = element.className.lastIndexOf("$")
    if (end == -1) end = element.className.length
    val tag = element.className.substring(start, end)

    out(tag, msg + SEPARATOR_STR + element.toString(), logType)
  }

  private fun out(tag: String, msg: String, logType: Int) {
    if (logToSystemOut) {
      println("$tag - $msg\n")
    }

    when (logType) {
      VERBOSE_LOG -> Log.v(tag, msg)
      DEBUG_LOG -> Log.d(tag, msg)
      INFO_LOG -> Log.i(tag, msg)
      WARNING_LOG -> Log.w(tag, msg)
      ERROR_LOG -> Log.e(tag, msg)
    }
  }

  /**
   * Appends the log message to [.logMessage].
   *
   * @param message The message to be logged.
   */
  @Synchronized
  private fun captureLog(message: String) {
    checkBuffer()
    logMessage.append(message + "\n")
  }

  /**
   * Appends the log message to [.logMessage].
   *
   * @param message The message to be logged.
   * @param throwable An exception to log.
   */
  @Synchronized
  private fun captureLog(message: String, throwable: Throwable) {
    checkBuffer()
    logMessage.append("$message $throwable\n")
  }

  /**
   * Checks the buffer size and if it exceed the [.MAX_LOG_SIZE],
   * it will trim the size from the start of the log messages.
   */
  private fun checkBuffer() {
    if (logMessage.length > MAX_LOG_SIZE) {
      logMessage.delete(0, logMessage.length - MAX_LOG_SIZE)
    }
  }

  /**
   * @return stored logged message.
   */
  internal val loggedMessages: String
    get() = logMessage.toString()

  /**
   * Clear stored logged messages.
   */
  @Synchronized
  fun resetLoggedMessages() {
    logMessage.delete(0, logMessage.length)
  }

  /**
   * Helper method to log long debug messages.
   *
   * @param message The message you would like logged.
   */
  fun longDebugLog(tag: String, message: String) {
    if (message.length > 4000) {
      d(tag, message.substring(0, 4000))
      longDebugLog(tag, message.substring(4000))
    } else {
      d(tag, message)
    }
  }

  /**
   * Enables logging of debug messages. This is disabled in
   * release builds by default.
   * @param enabled
   */
  @JvmStatic
  fun enableDebugLogs(enabled: Boolean) {
    isLoggingToLogcatEnabled = enabled
  }

  internal suspend fun setup(context: Context) = withContext(ioDispatcher) {
    val sharedPref =
      context.getSharedPreferences(HyprMXController.HYPRMX_PREFS_INTERNAL, Context.MODE_PRIVATE)
    isLoggingOverrideFromServerEnabled = sharedPref
      .getBoolean(PREF_ALL_LOGGING_ENABLED, false)
  }

  internal suspend fun setAllLoggingEnabled(context: Context, enabled: Boolean) =
    withContext(ioDispatcher) {
      isLoggingOverrideFromServerEnabled = enabled
      val sharedPref =
        context.getSharedPreferences(HyprMXController.HYPRMX_PREFS_INTERNAL, Context.MODE_PRIVATE)
      with(sharedPref.edit()) {
        putBoolean(PREF_ALL_LOGGING_ENABLED, enabled)
        commit()
      }
    }
}
