package com.hyprmx.android.sdk.utility

import com.hyprmx.android.sdk.annotation.RetainMethodSignature
import com.hyprmx.android.sdk.core.js.JSEngine
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Job
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch

internal class HyprMXNativeTimer(
  private val jsEngine: JSEngine,
  private val coroutineScope: CoroutineScope,
) : HyprMXNativeTimerJSInterface {
  init {
    jsEngine.addJavascriptInterface(this, JS_INTERFACE_NAME)
  }

  private val timerMap: MutableMap<String, Job?> = HashMap()
  private val callbackMap: MutableMap<String, String> = HashMap()

  override fun startNativeTimer(id: String, time: Long, callback: String) {
    HyprMXLog.d("startNativeTimer($id, $time)")
    callbackMap[id] = callback
    timerMap[id] = createTimer(id, time)
  }

  override fun stopTimer(id: String) {
    HyprMXLog.d("stopTimer($id)")
    timerMap[id]?.cancel()
    timerMap[id] = null
  }

  override fun updateTimer(id: String, time: Long) {
    HyprMXLog.d("updateTimer($id, $time)")
    timerMap[id]?.cancel()
    timerMap[id] = createTimer(id, time)
  }

  private fun createTimer(id: String, time: Long): Job =
    coroutineScope.launch {
      delay(time)
      callbackMap[id]?.let {
        jsEngine.evaluate(it)
      }
      timerMap.remove(id)
      callbackMap.remove(id)
    }

  companion object {
    const val JS_INTERFACE_NAME = "HYPRNativeTimer"
  }
}

internal interface HyprMXNativeTimerJSInterface {
  @RetainMethodSignature
  fun startNativeTimer(id: String, time: Long, callback: String)

  @RetainMethodSignature
  fun stopTimer(id: String)

  @RetainMethodSignature
  fun updateTimer(id: String, time: Long)
}
