package com.hyprmx.android.sdk.utility

import android.content.Context
import android.graphics.Color
import android.widget.Toast

internal fun Context.pxToFloatDp(pixels: Float): Float {
  return if (pixels <= 0) {
    0f
  } else {
    pixels / getDensity()
  }
}

internal fun Context.dpToFloatPixels(dp: Float): Float {
  return if (dp <= 0) {
    0f
  } else {
    return dp * getDensity()
  }
}

internal fun Context.getDensity(): Float {
  return this.resources.displayMetrics.density
}

internal fun Int.convertDpToPixel(context: Context?): Int {
  return if (context == null) {
    0
  } else {
    (context.dpToFloatPixels(this.toFloat()) + 0.5f).toInt()
  }
}

/**
 * This method converts device specific pixels to density independent pixels.
 *
 * @param context Context to get resources and device specific display metrics
 * @return A int value to represent dp equivalent to px value
 */
internal fun Int.convertPixelsToDp(context: Context): Float {
  return context.pxToFloatDp(this.toFloat())
}

internal fun String.showAsToast(context: Context?) = context?.let {
  Toast.makeText(context.applicationContext, this, Toast.LENGTH_LONG).show()
}

internal fun getColor(alpha: Float, color: String): Int {
  var colorInt = 0
  try {
    colorInt = Color.parseColor("#$color")
  } catch (e: IllegalArgumentException) {
    try {
      colorInt = Color.parseColor("#" + Integer.toHexString((255 * alpha).toInt()) + color)
    } catch (exception: IllegalArgumentException) {
      HyprMXLog.d(exception.message)
    }
  }

  return colorInt
}

/**
 * The color string we are getting from the server is in the format RRGGBBAA or RRGGBB
 * but Android uses AARRGGBB or RRGGBB.
 *
 *
 * This method converts and returns the appropriate color
 */
internal fun String.parseColor(): Int {
  var newColor = this
  if (newColor.length == 8) {
    newColor = this.substring(6) + this.substring(0, 6)
  }
  return Color.parseColor("#$newColor")
}
