package com.hyprmx.android.sdk.utility

import android.app.Activity
import android.content.ActivityNotFoundException
import android.content.ContentValues
import android.content.Context
import android.content.Intent
import android.content.Intent.URI_ANDROID_APP_SCHEME
import android.content.Intent.URI_INTENT_SCHEME
import android.content.pm.ActivityInfo
import android.content.pm.PackageManager
import android.graphics.Bitmap
import android.graphics.BitmapFactory
import android.media.MediaPlayer
import android.net.ConnectivityManager
import android.net.Uri
import android.os.Build
import android.os.Environment
import android.os.StatFs
import android.provider.MediaStore
import android.text.Html
import android.text.Spanned
import android.util.AttributeSet
import android.util.DisplayMetrics
import android.view.Surface
import android.webkit.MimeTypeMap
import android.webkit.URLUtil
import androidx.annotation.RequiresApi
import androidx.core.content.ContextCompat
import androidx.core.net.toUri
import com.hyprmx.android.R
import com.hyprmx.android.sdk.analytics.ClientErrorControllerIf
import com.hyprmx.android.sdk.banner.HyprMXBannerSize
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import org.json.JSONArray
import org.json.JSONException
import org.json.JSONObject
import java.io.File
import java.io.FileOutputStream
import java.io.IOException
import java.io.OutputStream
import java.net.URISyntaxException
import java.net.URL
import java.security.MessageDigest
import java.text.SimpleDateFormat
import java.util.ArrayList
import java.util.Date
import java.util.Locale
import java.util.TimeZone
import kotlin.math.ceil

/**
 * HyprMX sdk's utility class.
 *
 * This class will have no thread asserts
 * since it will have no usage of local and global variables.
 */
internal object Utils {

  /**
   * Convert string to md5 hash.
   *
   * @param data String to be converted
   * @return md5 hash of `data`
   */
  fun convertToMD5(data: String): String {
    val bytes = MessageDigest.getInstance("MD5").digest(data.toByteArray())
    return bytes.toHex()
  }

  private fun ByteArray.toHex(): String {
    return joinToString("") { "%02x".format(it) }
  }

  /**
   * Calculate the available free space on internal memory.
   * @return the number of bytes that are free on the device.
   */
  @RequiresApi(Build.VERSION_CODES.JELLY_BEAN_MR2)
  fun getAvailableInternalMemorySpace(): Long {
    val path = Environment.getDataDirectory()
    val stat = StatFs(path.path)
    val blockSize = stat.blockSizeLong
    val availableBlocks = stat.availableBlocksLong
    return availableBlocks * blockSize
  }

  /**
   * Get current time in UTC.
   *
   * @return Current time in "yyyy-MM-ddTHH:mm:sssZ" format.
   * for example, 2015-03-24 16:54:59 +0000
   */
  fun getCurrentDateAsString(): String {
    val dateFormatGmt = SimpleDateFormat("yyyy-MM-dd'T'HH:mm:sss'Z'", Locale.US)
    dateFormatGmt.timeZone = TimeZone.getTimeZone("UTC")
    return dateFormatGmt.format(Date())
  }

  private var trackProbability = -1f
  fun getTrackingProbability(): Float {
    return trackProbability
  }

  // TODO: Must remove setTrackingProbability in production. Just used for testing purpose.
  fun setTrackingProbability(probabilityPercentage: Int) {
    trackProbability = probabilityPercentage / 100f
  }

  fun isMediaSupported(mediaFilePath: String): Boolean {
    var mp: MediaPlayer? = null
    return try {
      mp = MediaPlayer()
      mp.setDataSource(mediaFilePath)
      mp.prepare()
      true
    } catch (e: Exception) {
      false
    } finally {
      mp?.release()
    }
  }

  /**
   * Returns common network error msg from offer Activity classes.
   *
   * @param context activity or application context
   * @return error msg
   */
  fun getActivityNetworkErrorMsg(context: Context): String {
    return if (!isNetworkAvailable(context)) {
      context.getString(R.string.hyprmx_no_internet_error_message)
    } else {
      context.getString(R.string.hyprmx_ad_display_error)
    }
  }

  fun getCurrentScreenOrientation(activity: Activity): Int {
    val rotation = activity.windowManager.defaultDisplay.rotation
    val dm = DisplayMetrics()
    activity.windowManager.defaultDisplay.getMetrics(dm)
    val width = dm.widthPixels
    val height = dm.heightPixels
    val orientation: Int

    // if the device's natural orientation is portrait:
    if ((rotation == Surface.ROTATION_0 || rotation == Surface.ROTATION_180) && height > width ||
      (rotation == Surface.ROTATION_90 || rotation == Surface.ROTATION_270) && width > height
    ) {
      orientation = when (rotation) {
        Surface.ROTATION_0 -> ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
        Surface.ROTATION_90 -> ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
        Surface.ROTATION_180 -> ActivityInfo.SCREEN_ORIENTATION_REVERSE_PORTRAIT
        Surface.ROTATION_270 -> ActivityInfo.SCREEN_ORIENTATION_REVERSE_LANDSCAPE
        else -> {
          HyprMXLog.e("Unknown screen orientation. Defaulting to portrait.")
          ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
        }
      }
    } else {
      orientation = when (rotation) {
        Surface.ROTATION_0 -> ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
        Surface.ROTATION_90 -> ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
        Surface.ROTATION_180 -> ActivityInfo.SCREEN_ORIENTATION_REVERSE_LANDSCAPE
        Surface.ROTATION_270 -> ActivityInfo.SCREEN_ORIENTATION_REVERSE_PORTRAIT
        else -> {
          HyprMXLog.e("Unknown screen orientation. Defaulting to landscape.")
          ActivityInfo.SCREEN_ORIENTATION_LANDSCAPE
        }
      }
    } // if the device's natural orientation is landscape or if the device is square:

    return orientation
  }

  /** Returns true if a photo can be added to the device's gallery, false otherwise.  */

  /**
   * Validates the url value mapped from key in json. If valid then returns the value,
   * otherwise it returns the default url and fires cec.
   * If there is no mapping it will return the default url without firing cec.
   *
   * @param json JSONObject being parsed
   * @param key key mapped to get value from json
   * @param defaultUrl Default url path to fallback
   * @param validateUrl true then send cec if url is invalid
   * @return the url if valid, otherwise default url
   */
  fun getValidatedUrlString(
    json: JSONObject,
    key: String,
    defaultUrl: String,
    validateUrl: Boolean,
    clientErrorController: ClientErrorControllerIf,
  ): String {
    try {
      json.get(key)
    } catch (e: JSONException) {
      return defaultUrl
    }

    var validUrl = json.getStringOrNull(key)
    if (validUrl.isNullOrEmpty()) {
      validUrl = defaultUrl
      HyprMXLog.d("URL for $key is null or empty")
    } else {
      var url: URL? = null

      try {
        url = URL(validUrl)
      } catch (e: IOException) {
        validUrl = defaultUrl
        if (validateUrl) {
          val errorMsg = "URL for $key is invalid"
          HyprMXLog.e(errorMsg)
          clientErrorController.sendClientError(
            HyprMXErrorType.HYPRErrorTypeJSONParsingFailure,
            errorMsg,
            ClientErrorControllerIf.SEVERITY_3,
          )
        }
      }

      if (url != null) {
        try {
          url.toURI()
        } catch (e: URISyntaxException) {
          validUrl = defaultUrl
          if (validateUrl) {
            val errorMsg = "URL for $key contains invalid characters"
            HyprMXLog.e(errorMsg)
            clientErrorController.sendClientError(
              HyprMXErrorType.HYPRErrorTypeJSONParsingFailure,
              errorMsg,
              ClientErrorControllerIf.SEVERITY_3,
            )
          }
        }
      }
    }
    return validUrl
  }

  @Suppress("DEPRECATION")
  fun getSpannedText(spanText: String): Spanned {
    return if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
      Html.fromHtml(spanText, Html.FROM_HTML_OPTION_USE_CSS_COLORS)
    } else {
      Html.fromHtml(spanText)
    }
  }
}

internal fun <T : Any> T?.whenNull(f: () -> Unit) {
  if (this == null) f()
}

/**
 * Returns false if url is null, empty, invalid or contains invalid characters.
 * Returns true otherwise.
 *
 * @return false if url is null, empty, invalid or contains invalid characters, true otherwise
 */
internal fun String.isValidUrl(): Boolean = URLUtil.isValidUrl(this)

internal fun String.isInvalidUrl(): Boolean = !this.isValidUrl()

internal fun String.toThirdPartyAppIntent(): Intent {
  val uriFlags = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
    URI_INTENT_SCHEME or URI_ANDROID_APP_SCHEME
  } else {
    URI_INTENT_SCHEME
  }
  return Intent.parseUri(this, uriFlags)
    .addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
}

/**
 * Returns true if a new activity was started. Will not start "Browser" activities.
 *
 * @param context the context to start a new activity from
 * @param intent the intent used to launch the activity
 * @return true if an activity was started successfully
 */
internal fun startResolvedActivity(
  context: Context,
  intent: Intent,
): Boolean = try {
  HyprMXLog.d("Starting Activity for intent $intent")
  context.startActivity(intent)
  true
} catch (exception: ActivityNotFoundException) {
  HyprMXLog.d("Unable to start activity for intent $intent")
  false
}

internal fun startResolvedActivity(context: Context, url: String): Boolean =
  startResolvedActivity(context, url.toThirdPartyAppIntent())

/** Returns a List of permissions that have been explicitly listed in the Android Manifest.  */
@Throws(PackageManager.NameNotFoundException::class)
internal fun getPermissionsListedInAndroidManifest(context: Context): List<String> {
  val permissions = ArrayList<String>()
  val info =
    context.packageManager.getPackageInfo(context.packageName, PackageManager.GET_PERMISSIONS)
  if (info.requestedPermissions != null) {
    for (permission in info.requestedPermissions) {
      permissions.add(permission)
    }
  }
  return permissions
}

internal suspend fun galleryAddPic(context: Context, photoPath: String) = withContext(Dispatchers.IO) {
  val file = File(photoPath)
  if (!file.exists() || file.length() == 0L) {
    HyprMXLog.d("No file found to save.")
    return@withContext
  }

  if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
    addToGalleryQandAbove(photoPath, context)
  } else {
    addToGalleryPreQ(photoPath, context)
  }
}

private suspend fun addToGalleryPreQ(photoPath: String, context: Context) = withContext(Dispatchers.IO) {
  val path = File(Environment.getExternalStorageDirectory(), "Pictures")

  if (!path.exists()) {
    path.mkdirs()
  }

  val incomingFileName = File(photoPath).name

  val file: File? = try {
    val file = File(path, incomingFileName)

    // Get the file output stream
    val stream: OutputStream = FileOutputStream(file)

    // Compress the bitmap
    BitmapFactory.decodeFile(photoPath).compress(Bitmap.CompressFormat.JPEG, 100, stream)

    // Flush the output stream
    stream.flush()

    // Close the output stream
    stream.close()
    file
  } catch (e: Exception) { // Catch the exception
    HyprMXLog.e("Exception when trying to store a picture", e)
    null
  }

  file?.let {
    val mediaScanIntent = Intent(Intent.ACTION_MEDIA_SCANNER_SCAN_FILE)
    val f = File(photoPath)
    val contentUri = Uri.fromFile(f)
    mediaScanIntent.data = contentUri
    context.sendBroadcast(mediaScanIntent)
    photoPath
  }
}

@RequiresApi(Build.VERSION_CODES.Q)
private suspend fun addToGalleryQandAbove(
  photoPath: String,
  context: Context,
) = withContext(Dispatchers.IO) {
  try {
    val file = File(photoPath)

    val extension = file.extension

    val mimeType: String
    val compressionFormat: Bitmap.CompressFormat

    if (extension.equals("png", ignoreCase = true)) {
      compressionFormat = Bitmap.CompressFormat.PNG
      mimeType = "image/png"
    } else {
      compressionFormat = Bitmap.CompressFormat.JPEG
      mimeType = "image/jpeg"
    }

    val contentValues = ContentValues().apply {
      put(MediaStore.MediaColumns.DISPLAY_NAME, photoPath)

      put(MediaStore.MediaColumns.MIME_TYPE, mimeType)
      put(MediaStore.MediaColumns.RELATIVE_PATH, "Pictures/")
      put(MediaStore.Images.Media.IS_PENDING, 1)
    }

    val collection = MediaStore.Images.Media.getContentUri(MediaStore.VOLUME_EXTERNAL_PRIMARY)
    val imageUri = context.contentResolver.insert(collection, contentValues)
    imageUri?.let {
      context.contentResolver.openOutputStream(imageUri)?.use { out ->
        BitmapFactory.decodeFile(photoPath).compress(compressionFormat, 100, out)
      }
      contentValues.clear()
      contentValues.put(MediaStore.Images.Media.IS_PENDING, 0)
      context.contentResolver.update(imageUri, contentValues, null, null)
    }
  } catch (exception: Exception) {
    HyprMXLog.e("Exception when trying to store a picture (Q and Above)", exception)
  }
}

/** Returns true if the provided permission has been granted by the user, false otherwise.  */
internal fun isPermissionGranted(
  context: Context,
  permission: String,
): Boolean {
  return ContextCompat.checkSelfPermission(
    context,
    permission,
  ) == PackageManager.PERMISSION_GRANTED
}

/**
 * @return True if build version is less then api 26.
 */
internal fun isBuildVersionUnsupported(): Boolean {
  return Build.VERSION.SDK_INT < MINIMUM_SUPPORTED_VERSION
}

/**
 * Returns true if there is an internet connection, false otherwise.
 *
 * @param context activity or application context
 * @return true if there is an internet connection, false otherwise.
 */
internal fun isNetworkAvailable(context: Context): Boolean {
  val connectivityManager =
    context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager
  val networkInfo = connectivityManager.activeNetworkInfo
  return networkInfo != null && networkInfo.isConnected
}

/**
 * Gets the placementName using the provided attributes
 *
 * @param attrs Attribute set to pull the placement name from
 * @return The placement name or null if the attribute cannot be parsed or does not exist
 */
internal fun getPlacementNameAttributeSet(context: Context, attrs: AttributeSet): String? {
  val styledAttributes = context.theme.obtainStyledAttributes(
    attrs,
    R.styleable.HyprMXView,
    0,
    0,
  )

  val placementName = styledAttributes.getString(R.styleable.HyprMXView_hyprMXPlacementName)

  if (placementName == null) {
    HyprMXLog.d("HyprMXPlacementName not defined in XML")
  }

  return placementName
}

/**
 * Create the adSize object using the provided attributes
 *
 * @param attrs Attribute set to pull the size from
 * @return The size object or null if the attribute cannot be parsed or does not exist
 */
internal fun getAdSizeFromAttributeSet(context: Context, attrs: AttributeSet): HyprMXBannerSize? {
  val styledAttributes = context.theme.obtainStyledAttributes(
    attrs,
    R.styleable.HyprMXView,
    0,
    0,
  )

  val index = styledAttributes.getInteger(R.styleable.HyprMXView_hyprMXAdSize, Int.MIN_VALUE)

  if (index == Int.MIN_VALUE) {
    HyprMXLog.d("HyprMXAdSize not defined in XML")
    return null
  }

  return when (index) {
    0 -> HyprMXBannerSize.HyprMXAdSizeShort
    1 -> HyprMXBannerSize.HyprMXAdSizeBanner
    2 -> HyprMXBannerSize.HyprMXAdSizeMediumRectangle
    3 -> HyprMXBannerSize.HyprMXAdSizeLeaderboard
    4 -> HyprMXBannerSize.HyprMXAdSizeSkyScraper
    5 -> HyprMXBannerSize.HyprMXAdSizeCustom(0, 0)
    else -> {
      HyprMXLog.d("Could not determine HyprMXAdSize from attributes")
      null
    }
  }
}

internal fun JSONArray.toArrayList(): ArrayList<String> {
  val list = arrayListOf<String>()
  for (i in 0 until this.length()) {
    list.add(this.getString(i))
  }
  return list
}

/**
 * This method converts device specific pixels to density independent pixels.
 *
 * @param px A value in px (pixels) unit. Which we need to convert into db
 * @param context Context to get resources and device specific display metrics
 * @return A float value to represent dp equivalent to px value
 */
fun convertPixelsToDp(px: Float, context: Context): Int {
  return ceil(px / (context.resources.displayMetrics.densityDpi.toFloat() / DisplayMetrics.DENSITY_DEFAULT)).toInt()
}

internal fun File.mimeType(): String? {
  var type: String? = null
  val extension = MimeTypeMap.getFileExtensionFromUrl(toUri().toString())
  if (extension != null) {
    type = MimeTypeMap.getSingleton().getMimeTypeFromExtension(extension)
  }
  return type
}
