package com.hyprmx.android.sdk.webview

import android.graphics.Bitmap
import android.net.Uri
import android.os.Message
import android.webkit.JsResult
import android.webkit.PermissionRequest
import android.webkit.ValueCallback
import android.webkit.WebChromeClient
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.annotation.VisibleForTesting
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.presentation.PresentationFactory

internal interface HyprMXWebChromeClient {
  fun onPermissionRequest(request: PermissionRequest)
  fun onCreateWindow(url: String): String?
  fun onJSDialog(showCancel: Boolean, url: String, message: String, jsResult: JsResult): Boolean
  fun onShowFileChooser(webView: WebView, filePathCallback: ValueCallback<Array<Uri>>, fileChooserParams: WebChromeClient.FileChooserParams): Boolean
}

internal class HyprMXWebViewChromeAdapter(private val client: HyprMXWebChromeClient, private val webViewFactory: WebViewFactoryIf? = hyprmxDelegate.hyprMXController?.webViewFactory) : WebChromeClient() {
  @VisibleForTesting
  var presenterFactory: PresentationFactory? = null
    get() = field ?: hyprmxDelegate.hyprMXController?.presenterFactory

  override fun onCreateWindow(view: WebView, isDialog: Boolean, isUserGesture: Boolean, resultMsg: Message): Boolean {
    /**
     * We need to determine the URL that we want to present before we present it.
     * In Android, they don't let you know the url until you pass it to a webview.
     * I don't know why.. that's just how they do it.
     *
     * This code creates a webview and captures the first url load, blocks it, and sends it off
     * to the shared code to evaluate it and decide if we should open up a sharing overlay
     */
    val targetWebView = WebView(view.context) // pass a context
    targetWebView.webViewClient = object : WebViewClient() {
      override fun shouldOverrideUrlLoading(view: WebView, url: String): Boolean {
        targetWebView.webViewClient = WebViewClient()
        client.onCreateWindow(url) // you can get your target url here
        return true // return false if you want the load to continue
      }
    }
    val transport = resultMsg.obj as WebView.WebViewTransport
    transport.webView = targetWebView
    resultMsg.sendToTarget()
    return true
  }

  override fun onShowFileChooser(
    webView: WebView,
    filePathCallback: ValueCallback<Array<Uri>>,
    fileChooserParams: FileChooserParams,
  ): Boolean {
    return client.onShowFileChooser(webView, filePathCallback, fileChooserParams)
  }

  override fun onJsAlert(view: WebView, url: String, message: String, result: JsResult): Boolean {
    return client.onJSDialog(true, url, message, result)
  }

  override fun onJsConfirm(view: WebView, url: String, message: String, result: JsResult): Boolean {
    return client.onJSDialog(false, url, message, result)
  }

  override fun onPermissionRequest(request: PermissionRequest) {
    client.onPermissionRequest(request)
  }

  /**
   * If default video poster is null then a new transparent bitmap of size 1x1 is created and returned.
   * This will replace android showing the filmstrip icon with a black background right before playing html5 video.
   *
   * @return Bitmap to show before playing video.
   */
  override fun getDefaultVideoPoster(): Bitmap = super.getDefaultVideoPoster() ?: Bitmap.createBitmap(10, 10, Bitmap.Config.ARGB_8888)
}
