package com.hyprmx.android.sdk.webview

import android.annotation.SuppressLint
import android.app.Activity
import android.app.AlertDialog
import android.content.Context
import android.graphics.Color
import android.os.Build
import android.util.AttributeSet
import android.view.MotionEvent
import android.view.ViewGroup
import android.webkit.CookieManager
import android.webkit.JsResult
import android.webkit.WebSettings
import android.webkit.WebView
import android.webkit.WebViewClient
import android.widget.FrameLayout
import androidx.annotation.RequiresApi
import com.hyprmx.android.R
import com.hyprmx.android.sdk.core.hyprmxDelegate
import com.hyprmx.android.sdk.jsinterface.HyprMXAppJSInterfaceHandler
import com.hyprmx.android.sdk.jsinterface.HyprMXMraidJSInterfaceHandler
import com.hyprmx.android.sdk.mvp.LifecycleEvent
import com.hyprmx.android.sdk.presentation.PresentationFactory
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.convertPixelsToDp

@SuppressLint("SetJavaScriptEnabled", "ClickableViewAccessibility")
internal class HyprMXWebView @JvmOverloads constructor(
  context: Context,
  attrs: AttributeSet? = null,
  defStyleAttr: Int = 0,
  private var userAgent: String? = null,
  var webView: WebView = createNewWebView(context),
) : FrameLayout(context, attrs, defStyleAttr),
  HyprMXWebViewContract.View {

  private var presenter: HyprMXWebViewContract.Presenter? = null

  private var presenterFactory: PresentationFactory? = null
    get() = field ?: hyprmxDelegate.hyprMXController?.presenterFactory

  override var containingActivity: Activity? = null

  override val currentUrl: String? = webView.url

  override val progress: Int = webView.progress

  init {
    addWebViewToLayout(webView)
  }

  override fun initialize(viewModelIdentifier: String, userAgent: String?) {
    this.userAgent = userAgent
    if (this.userAgent?.isBlank() == false) {
      this.webView.settings.userAgentString = userAgent
    }

    if (presenter != null) {
      presenter?.rebind(viewModelIdentifier)
      return
    }

    presenterFactory?.let { factory ->
      presenter = factory.makeWebViewPresenter(
        this,
        viewModelIdentifier = viewModelIdentifier,
      )
        .also {
          HyprMXLog.d("Updating webview with chrome, client, download, js listeners")
          webView.webViewClient = HyprMXWebViewClientAdapter(it)
          webView.webChromeClient = HyprMXWebViewChromeAdapter(it)
          webView.setDownloadListener(it)
        }
    }

    addJSInterfaces()
  }

  override fun postUrl(url: String, postData: ByteArray) {
    webView.postUrl(url, postData)
  }

  private fun addWebViewToLayout(webViewToAdd: WebView) {
    webViewToAdd.layoutParams = LayoutParams(
      ViewGroup.LayoutParams.MATCH_PARENT,
      ViewGroup.LayoutParams.MATCH_PARENT,
    )

    addView(webViewToAdd)
  }

  override fun loadUrl(url: String, userAgent: String?) {
    addJSInterfaces()
    HyprMXLog.d("loadUrl($url) with userAgent = $userAgent")
    if (userAgent?.isNotBlank() == true) {
      webView.settings.userAgentString = userAgent
    }

    webView.loadUrl(url)
  }

  override fun loadData(url: String, data: String, mimeType: String, encoding: String) {
    HyprMXLog.d("loadData $data")
    webView.loadDataWithBaseURL(
      url,
      data,
      mimeType,
      encoding,
      null,
    )
  }

  @SuppressLint("AddJavascriptInterface")
  override fun addJSInterfaces() {
    HyprMXLog.d("Attaching JS Interfaces")
    presenter?.let {
      webView.addJavascriptInterface(HyprMXAppJSInterfaceHandler(it), HyprMXAppJSInterfaceHandler.JS_INTERFACE)
      webView.addJavascriptInterface(HyprMXMraidJSInterfaceHandler(it), HyprMXMraidJSInterfaceHandler.JS_INTERFACE)
    }
  }

  override fun removeJSInterfaces() {
    HyprMXLog.d("Removing JS Interfaces")
    presenter?.let {
      webView.removeJavascriptInterface(HyprMXAppJSInterfaceHandler.JS_INTERFACE)
      webView.removeJavascriptInterface(HyprMXMraidJSInterfaceHandler.JS_INTERFACE)
    }
  }

  @RequiresApi(Build.VERSION_CODES.KITKAT)
  override fun executeJS(script: String) {
    webView.evaluateJavascript(script, null)
  }

  override fun navigateBack() {
    webView.goBack()
  }

  override fun navigateForward() {
    webView.goForward()
  }

  override fun pauseJSExecution() {
    webView.onPause()
  }

  override fun resumeJSExecution() {
    webView.onResume()
  }

  override fun clearHistory() = webView.clearHistory()

  private var currentConfiguration: WebViewConfiguration? = null

  override fun updateWebViewConfiguration(
    scrollable: Boolean,
    bounceEnable: Boolean,
    allowPinchGesture: Boolean,
    linkPreview: Boolean,
    javascriptEnabled: Boolean,
    domStorageEnabled: Boolean,
    loadWithOverviewMode: Boolean,
    useWideViewPort: Boolean,
    displayZoomControls: Boolean,
    builtInZoomControls: Boolean,
    supportMultiWindow: Boolean,
    backgroundColor: String,
    customUserAgent: String?,
    playbackRequiresUserAction: Boolean,
  ) {
    currentConfiguration = WebViewConfiguration(
      scrollable,
      bounceEnable,
      allowPinchGesture,
      linkPreview,
      javascriptEnabled,
      domStorageEnabled,
      loadWithOverviewMode,
      useWideViewPort,
      displayZoomControls,
      builtInZoomControls,
      supportMultiWindow,
      backgroundColor,
      customUserAgent,
      playbackRequiresUserAction,
    )
    updateWebViewWithConfiguration()
  }

  override fun setUserAgent(userAgent: String) {
    webView.settings.userAgentString = userAgent
  }

  private fun updateWebViewWithConfiguration() {
    currentConfiguration?.apply {
      enableScrolling(scrollable)

      webView.apply {
        setBackgroundColor(Color.parseColor("#$backgroundColor"))
        overScrollMode = if (bounceEnable) OVER_SCROLL_ALWAYS else OVER_SCROLL_NEVER

        if (customUserAgent?.isNotEmpty() == true) {
          settings.userAgentString = customUserAgent
        }

        settings.javaScriptEnabled = javascriptEnabled
        settings.domStorageEnabled = domStorageEnabled
        settings.loadWithOverviewMode = loadWithOverviewMode
        settings.useWideViewPort = useWideViewPort

        settings.setSupportZoom(allowPinchGesture)
        settings.displayZoomControls = displayZoomControls
        settings.builtInZoomControls = builtInZoomControls

        settings.setSupportMultipleWindows(supportMultiWindow)

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
          settings.mediaPlaybackRequiresUserGesture = playbackRequiresUserAction
        }
      }
    }
  }

  override fun enableScrolling(enable: Boolean) {
    if (enable) {
      webView.setOnTouchListener(null)
    } else {
      webView.setOnTouchListener { _, event ->
        // We're not handling events if the current action is ACTION_MOVE
        event.action == MotionEvent.ACTION_MOVE
      }
      isHorizontalScrollBarEnabled = false
      isVerticalScrollBarEnabled = false
    }
  }

  override fun onAttachedToWindow() {
    HyprMXLog.d("onAttachedToWindow ${presenter?.viewModelIdentifier}")
    presenter?.onLifecycleEvent(LifecycleEvent.onAttachedToWindow)
    super.onAttachedToWindow()
  }

  override fun onDetachedFromWindow() {
    HyprMXLog.d("onDetachedFromWindow ${presenter?.viewModelIdentifier}")
    super.onDetachedFromWindow()
    presenter?.onLifecycleEvent(LifecycleEvent.onDetachedFromWindow)
  }

  override fun onSizeChanged(w: Int, h: Int, oldw: Int, oldh: Int) {
    super.onSizeChanged(w, h, oldw, oldh)
    presenter?.onSizeChanged(w.convertPixelsToDp(context), h.convertPixelsToDp(context))
  }

  fun cleanup() {
    if (presenter?.shouldLoadAboutBlank() == true) {
      webView.loadUrl("about:blank")
    }
    presenter?.cleanup()
    presenter = null
    containingActivity = null
    cleanupWebview()
    webView.destroy()
  }

  private fun cleanupWebview() {
    removeJSInterfaces()
    webView.webChromeClient = null
    webView.webViewClient = WebViewClient()
  }

  @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
  override fun showAlertDialog(showCancel: Boolean, message: String, jsResult: JsResult) {
    containingActivity?.let {
      val alertDialogBuilder = AlertDialog.Builder(it)
        .setMessage(message)
        .setPositiveButton(R.string.hyprmx_ok) { dialogInterface, _ ->
          dialogInterface.dismiss()
          jsResult.confirm()
        }

      if (showCancel) {
        alertDialogBuilder
          .setNegativeButton(R.string.hyprmx_cancel) { dialogInterface, _ ->
            dialogInterface.dismiss()
            jsResult.cancel()
          }
          .setCancelable(true)
          .setOnCancelListener { jsResult.cancel() }
      }

      val alertDialog = alertDialogBuilder.create()
      alertDialog.setCanceledOnTouchOutside(true)
      alertDialog.show()
    }
  }

  override fun removeWebView(replace: Boolean) {
    HyprMXLog.d("Removing webview {${webView.hashCode()}")
    removeAllViews()

    if (replace) {
      webView = createNewWebView(context)
      updateWebViewWithConfiguration()
      presenter?.let {
        webView.webViewClient = HyprMXWebViewClientAdapter(it)
        webView.webChromeClient = HyprMXWebViewChromeAdapter(it)
        webView.setDownloadListener(it)
      }
      addWebViewToLayout(webView)
    }
  }

  override fun canGoBack(): Boolean = webView.canGoBack()
  override fun canGoForward(): Boolean = webView.canGoForward()
  override fun stopLoading() = webView.stopLoading()
  fun shouldTakeFocus(): Boolean {
    return presenter?.shouldTakeFocus() == true
  }
}

/**
 * factory method to create a new webview with the required settings
 */
@SuppressLint("SetJavaScriptEnabled")
internal fun createNewWebView(context: Context): WebView {
  val newWebView = WebView(context)

  newWebView.apply {
    id = R.id.hyprmx_webview

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
      settings.mixedContentMode = WebSettings.MIXED_CONTENT_COMPATIBILITY_MODE
      CookieManager.getInstance().setAcceptThirdPartyCookies(this, true)
    } else {
      CookieManager.getInstance().setAcceptCookie(true)
    }
  }
  return newWebView
}

private data class WebViewConfiguration(
  val scrollable: Boolean,
  val bounceEnable: Boolean,
  val allowPinchGesture: Boolean,
  val linkPreview: Boolean,
  val javascriptEnabled: Boolean,
  val domStorageEnabled: Boolean,
  val loadWithOverviewMode: Boolean,
  val useWideViewPort: Boolean,
  val displayZoomControls: Boolean,
  val builtInZoomControls: Boolean,
  val supportMultiWindow: Boolean,
  val backgroundColor: String,
  val customUserAgent: String?,
  val playbackRequiresUserAction: Boolean,
)
