package com.hyprmx.android.sdk.webview

import android.graphics.Bitmap
import android.os.Build
import android.webkit.RenderProcessGoneDetail
import android.webkit.WebResourceError
import android.webkit.WebResourceRequest
import android.webkit.WebResourceResponse
import android.webkit.WebView
import android.webkit.WebViewClient
import androidx.annotation.RequiresApi
import com.hyprmx.android.sdk.utility.HyprMXLog
import java.net.MalformedURLException
import java.net.URL

internal interface HyprMXWebViewClient {
  fun onPageStarted(url: String)
  fun onPageFinished(url: String)
  fun onReceivedError(description: String, errorCode: String, url: String)
  fun shouldOverrideUrlLoading(url: String, isMainFrame: Boolean): Boolean
  fun shouldInterceptRequest(url: String, scheme: String?, isMainFrame: Boolean): WebResourceResponse?
  fun onRenderProcessGone(): Boolean
  fun onHistoryChanged(canNavigateBack: Boolean, canNavigateForward: Boolean, currentIndex: Int, currentItem: String?, currentHost: String?, currentTitle: String?, history: List<String>)
}

internal class HyprMXWebViewClientAdapter(private val client: HyprMXWebViewClient) : WebViewClient() {

  @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
  override fun shouldOverrideUrlLoading(view: WebView, request: WebResourceRequest): Boolean {
    return client.shouldOverrideUrlLoading(request.url.toString(), request.isForMainFrame)
  }

  override fun onPageStarted(view: WebView, url: String, favicon: Bitmap?) {
    client.onPageStarted(url)
    onHistoryChanged(view)
  }

  override fun onPageFinished(view: WebView, url: String) {
    client.onPageFinished(url)
    onHistoryChanged(view)
  }

  private fun onHistoryChanged(view: WebView) {
    val backForwardList = view.copyBackForwardList()
    val history = mutableListOf<String>()
    for (index in 0 until backForwardList.size) {
      history.add(backForwardList.getItemAtIndex(index).url)
    }
    val currentHost = try {
      URL(view.url).host
    } catch (ex: MalformedURLException) {
      view.url
    }
    client.onHistoryChanged(canNavigateBack = view.canGoBack(), canNavigateForward = view.canGoForward(), currentIndex = backForwardList.currentIndex, currentItem = backForwardList.currentItem?.url, currentHost = currentHost, currentTitle = view.title, history = history)
  }

  @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
  override fun shouldInterceptRequest(view: WebView, request: WebResourceRequest): WebResourceResponse? {
    return client.shouldInterceptRequest(request.url.toString(), request.url.scheme, request.isForMainFrame)
  }

  override fun onReceivedError(view: WebView, errorCode: Int, description: String, failingUrl: String) {
    client.onReceivedError(description, errorCode.toString(), failingUrl)
  }

  @RequiresApi(Build.VERSION_CODES.M)
  override fun onReceivedError(view: WebView, request: WebResourceRequest, error: WebResourceError) {
    client.onReceivedError(error.description.toString(), error.errorCode.toString(), view.url.toString())
  }

  override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail?): Boolean {
    HyprMXLog.d("onRenderProcessGone for ${view?.hashCode()}")
    client.onRenderProcessGone()
    return true
  }
}
