package com.hyprmx.android.sdk.webview

import android.app.Activity
import android.webkit.DownloadListener
import android.webkit.JsResult
import com.hyprmx.android.sdk.core.BasePresenter
import com.hyprmx.android.sdk.core.BaseView

internal interface HyprMXWebViewContract {
  interface View : BaseView, ExecuteJavaScript, InternalView {

    /**
     * Initialize the webview
     *
     * @param viewModelIdentifier Optional. The parent VM identifier to connect to the shared code with.  If not provided assumes preloaded.
     * @param userAgent Optional user agent to have the webview use
     */
    fun initialize(viewModelIdentifier: String = "", userAgent: String? = null)

    /**
     * Loads a url with post data
     * @param url The url to load
     * @param postData The data to send
     */
    fun postUrl(url: String, postData: ByteArray)

    /**
     * Load the url in the webview
     */
    fun loadUrl(url: String, userAgent: String? = null)

    /**
     * Load the data in the webview
     */
    fun loadData(url: String, data: String, mimeType: String, encoding: String)

    /**
     * Add all the JS interfaces attached to the webview
     */
    fun addJSInterfaces()

    /**
     * Remove all the JS interfaces attached to the webview
     */
    fun removeJSInterfaces()

    /**
     * Execute a script on the webview
     */
    override fun executeJS(script: String)

    /**
     * Have the webview navigate one page back
     */
    fun navigateBack()

    /**
     * Have the webview navigate one page forward
     */
    fun navigateForward()

    /**
     * Pauses the JS Execution in the webview
     */
    fun pauseJSExecution()

    /**
     * Resumes the JS Execution in the webview
     */
    fun resumeJSExecution()

    /**
     * Clears the webview history
     */
    fun clearHistory()

    /**
     * Updates the webview configuration
     */
    fun updateWebViewConfiguration(
      scrollable: Boolean,
      bounceEnable: Boolean,
      allowPinchGesture: Boolean,
      linkPreview: Boolean,
      javascriptEnabled: Boolean,
      domStorageEnabled: Boolean,
      loadWithOverviewMode: Boolean,
      useWideViewPort: Boolean,
      displayZoomControls: Boolean,
      builtInZoomControls: Boolean,
      supportMultiWindow: Boolean,
      backgroundColor: String,
      customUserAgent: String?,
      playbackRequiresUserAction: Boolean,
    )

    fun setUserAgent(userAgent: String)

    /**
     * Enables/Disables scrolling of the webview
     */
    fun enableScrolling(enable: Boolean)

    var containingActivity: Activity?

    fun showAlertDialog(showCancel: Boolean, message: String, jsResult: JsResult)

    /**
     * Remove the native webview.
     * @param replace Indicates if the webview should be replaced
     */
    fun removeWebView(replace: Boolean)
  }

  interface Presenter : BasePresenter<View>, HyprMXWebViewClient, HyprMXWebChromeClient, DownloadListener, HyprMXJSMessageHandler, WebViewSharedInterface {
    fun rebind(viewModelIdentifier: String)
  }

  /** TODO: Eventually remove this when we no longer need to access to the webview for fullscreen ads */
  interface InternalView : ExecuteJavaScript {
    val currentUrl: String?
    val progress: Int
    fun canGoBack(): Boolean
    fun canGoForward(): Boolean
    fun stopLoading()
  }
}

internal interface ExecuteJavaScript {
  fun executeJS(script: String)
}

internal interface HyprMXJSMessageHandler {
  fun onJSMessage(methodName: String, argument: String? = null)
}

internal interface CleanUpHandler {
  // Cleanup references and stop collecting messages.
  fun cleanup()
}
