package com.hyprmx.android.sdk.webview

import android.annotation.SuppressLint
import android.net.Uri
import android.os.Build
import android.webkit.JsResult
import android.webkit.PermissionRequest
import android.webkit.ValueCallback
import android.webkit.WebChromeClient
import android.webkit.WebView
import androidx.annotation.RequiresApi
import androidx.core.net.toUri
import com.hyprmx.android.sdk.core.js.JSEngine
import com.hyprmx.android.sdk.mvp.LifecycleEventAdapter
import com.hyprmx.android.sdk.mvp.LifecycleEventHandler
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.presentation.WebViewPresentationEventPublisher
import com.hyprmx.android.sdk.utility.DefaultURLFilter
import com.hyprmx.android.sdk.utility.HyprMXLog
import com.hyprmx.android.sdk.utility.NavigationDecision
import com.hyprmx.android.sdk.utility.URLFilter
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.launch
import org.json.JSONObject
import java.lang.Error
import java.lang.Exception

@RequiresApi(Build.VERSION_CODES.LOLLIPOP)
internal class HyprMXWebViewPresenter(
  override var view: HyprMXWebViewContract.View?,
  viewModelIdentifier: String,
  private val jsEngine: JSEngine,
  private val scope: CoroutineScope,
  eventPublisher: PresentationEventPublisher = WebViewPresentationEventPublisher(
    jsEngine,
    viewModelIdentifier,
  ),
  private val urlFilter: URLFilter = DefaultURLFilter(eventPublisher, scope),
  lifecycleHandler: LifecycleEventHandler = LifecycleEventAdapter(eventPublisher, scope),
  val sharedInterface: WebViewSharedInterface = WebViewSharedConnector(eventPublisher),
) : HyprMXWebViewContract.Presenter,
  URLFilter by urlFilter,
  LifecycleEventHandler by lifecycleHandler,
  CoroutineScope by scope,
  WebViewNativeInterface,
  WebViewSharedInterface by sharedInterface {
  init {
    attach(this)
    updateWebViewConfiguration(getWebViewConfiguration())
  }

  private var permissionRequests = mutableMapOf<Int, PermissionRequest>()
  private var permissionRequestIds: Int = 0

  private var webviewFilePathCallback: ValueCallback<Array<Uri>>? = null

  override fun shouldOverrideUrlLoading(url: String, isMainFrame: Boolean): Boolean {
    return when (val decision = urlNavigationAttempt(url, isMainFrame)) {
      NavigationDecision.NavigationAllowed -> false
      NavigationDecision.NavigationBlocked -> true
      NavigationDecision.NavigationOpenOutsideApplication -> true
      is NavigationDecision.NavigationRedirected -> {
        loadUrl(decision.url)
        true
      }
    }
  }

  override fun onRenderProcessGone(): Boolean {
    HyprMXLog.e("onRenderProcessGone - The webview has reported a crash")
    val response = sharedInterface.onRenderProcessGone()
    view?.removeWebView(response)
    return response
  }

  @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
  override fun onPermissionRequest(request: PermissionRequest) {
    permissionRequests[permissionRequestIds] = request
    sharedInterface.onPermissionRequest(request, permissionRequestIds)
    permissionRequestIds++
  }

  override fun onCreateWindow(url: String): String? {
    HyprMXLog.d("onCreateWindow $url")
    return windowOpenAttempt(url)
  }

  /**
   * Determine in the shared code if we should display a custom dialog.
   *
   * Always return true because we are handling the event.
   */
  override fun onJSDialog(
    showCancel: Boolean,
    url: String,
    message: String,
    jsResult: JsResult,
  ): Boolean {
    val result = sharedInterface.onJSDialog(showCancel, url, message, jsResult)

    when {
      result -> view?.showAlertDialog(showCancel, message, jsResult)
      showCancel -> jsResult.cancel()
      else -> jsResult.confirm()
    }
    return true
  }

  @SuppressLint("NewApi")
  override fun onShowFileChooser(
    webView: WebView,
    filePathCallback: ValueCallback<Array<Uri>>,
    fileChooserParams: WebChromeClient.FileChooserParams,
  ): Boolean {
    HyprMXLog.d("onShowFileChooser")
    // Cleanup existing callbacks
    webviewFilePathCallback?.onReceiveValue(null)

    webviewFilePathCallback = filePathCallback
    return onShowFileChooser(fileChooserParams)
  }

  /**
   * We get this event and we use it to redirect requests download mime-type pdf to the online google
   * doc viewer.
   */
  override fun onDownloadStart(
    url: String,
    userAgent: String,
    contentDisposition: String,
    mimetype: String,
    contentLength: Long,
  ) {
    HyprMXLog.d("onDownloadStart $url with type $mimetype")
    val result = shouldRedirectURL(url, mimetype)
    if (result is NavigationDecision.NavigationRedirected) {
      loadUrl(result.url)
    }
  }

  private fun loadUrl(url: String) {
    view?.loadUrl(url)
  }

  override fun rebind(viewModelIdentifier: String) {
    if (this.viewModelIdentifier != viewModelIdentifier) {
      HyprMXLog.d("Rebind webview from ${this.viewModelIdentifier} to $viewModelIdentifier")
      // Do not destroy the webview VM since its the responsibility of the Browser/Fullscreen/Preloader
      this.viewModelIdentifier = viewModelIdentifier
      attach(this)
    }
  }

  override fun cleanup() {
    // Do not destroy the webview VM since its the responsibility of the Browser/Fullscreen/Preloader
    view = null
  }

  override fun postUrl(url: String, postParams: String) {
    launch {
      view?.postUrl(url, postParams.toByteArray())
    }
  }

  override fun runScript(script: String) {
    launch {
      view?.executeJS(script)
    }
  }

  override fun navigateBack() {
    launch {
      view?.navigateBack()
    }
  }

  override fun navigateForward() {
    launch {
      view?.navigateForward()
    }
  }

  override fun pauseJSExecution() {
    launch {
      view?.pauseJSExecution()
    }
  }

  override fun resumeJSExecution() {
    launch {
      view?.resumeJSExecution()
    }
  }

  override fun removeJavascriptInterface() {
    launch {
      view?.removeJSInterfaces()
    }
  }

  override fun addJavascriptInterface() {
    launch {
      view?.addJSInterfaces()
    }
  }

  override fun imageCaptured(url: String) {
    launch {
      if (url.isBlank()) {
        HyprMXLog.d("Image capture returned with empty path.")
        webviewFilePathCallback?.onReceiveValue(null)
      } else {
        webviewFilePathCallback?.onReceiveValue(arrayOf(url.toUri()))
      }
      webviewFilePathCallback = null
    }
  }

  override fun permissionResponse(data: String) {
    val jsonObject = JSONObject(data)
    val permissionId = jsonObject.getInt(WebViewEvents.PERMISSION_RESPONSE_ARG_PERMISSION_ID)
    val permissionList = jsonObject.getJSONArray(
      WebViewEvents.PERMISSION_RESPONSE_ARG_PERMISSIONS,
    )
    val permissions = HashMap<String, Boolean>()
    for (i in 0 until permissionList.length()) {
      val entry = permissionList.getJSONObject(i)
      permissions[entry.getString(WebViewEvents.PERMISSION_RESPONSE_ARG_PERMISSION)] =
        entry.getBoolean(
          WebViewEvents.PERMISSION_RESPONSE_ARG_GRANTED,
        )
    }

    val permissionRequest = permissionRequests.remove(permissionId)
    try {
      // If the permission already has been granted/denied and
      // you try again, the Request will throw an IllegalStateException.
      // https://chromium.googlesource.com/chromium/src.git/+/2e7314ec42493e43b93327e0dc72446cff068732/android_webview/java/src/org/chromium/android_webview/permission/AwPermissionRequest.java#84
      //  Only grant permissions that were granted.  If no permissions were granted, deny the request
      val grantedResources = permissions.filter { it.value }.map { it.key }
      if (grantedResources.isNotEmpty()) {
        permissionRequest?.grant(grantedResources.toTypedArray())
      } else {
        permissionRequest?.deny()
      }
    } catch (exception: Exception) {
      HyprMXLog.e("Exception while adjusting permissions: ${exception.localizedMessage}")
    } catch (error: Error) {
      HyprMXLog.e("Error while adjusting permissions: ${error.localizedMessage}")
    }
  }

  override fun updateWebViewConfiguration(data: String) {
    val event = createWebViewConfigEvent(data)
    if (event != null) {
      view?.updateWebViewConfiguration(
        scrollable = event.scrollable,
        bounceEnable = event.bounceEnable,
        allowPinchGesture = event.allowPinchGesture,
        linkPreview = event.linkPreview,
        javascriptEnabled = event.javascriptEnabled,
        domStorageEnabled = event.domStorageEnabled,
        loadWithOverviewMode = event.loadWithOverviewMode,
        displayZoomControls = event.displayZoomControls,
        builtInZoomControls = event.builtInZoomControls,
        supportMultiWindow = event.supportMultiWindow,
        useWideViewPort = event.useWideViewPort,
        backgroundColor = event.backgroundColor,
        customUserAgent = event.customUserAgent,
        playbackRequiresUserAction = event.playbackRequiresUserAction,
      )
    } else {
      HyprMXLog.d("Error parsing webview configuration update event")
    }
  }

  override fun setUrl(url: String) {
    launch {
      view?.loadUrl(url)
    }
  }

  override fun setAdHtml(data: String) {
    launch {
      AdHtml.fromJSON(data).let { adHtml ->
        view?.loadData(adHtml.baseUrl, adHtml.data, adHtml.mimeType, adHtml.encoding)
        onLoadData()
      }
    }
  }

  override fun setCatalogFramePost(data: String) {
    launch {
      CatalogFramePost.fromJSON(data).let {
        view?.postUrl(it.url, it.params.toByteArray())
      }
    }
  }
}

internal data class AdHtml(
  val data: String,
  val baseUrl: String,
  val mimeType: String,
  val encoding: String,
) {
  companion object {
    private const val DATA_KEY = "data"
    private const val BASE_URL_KEY = "baseUrl"
    private const val MIME_TYPE_KEY = "mimeType"
    private const val ENCODING_KEY = "encoding"
    fun fromJSON(data: String): AdHtml {
      JSONObject(data).apply {
        return AdHtml(
          this.optString(DATA_KEY),
          this.optString(BASE_URL_KEY),
          this.optString(MIME_TYPE_KEY),
          this.optString(ENCODING_KEY),
        )
      }
    }
  }
}

internal data class CatalogFramePost(
  val url: String,
  val params: String,
  val query: String,
) {
  companion object {
    private const val URL_KEY = "url"
    private const val PARAMS_KEY = "params"
    private const val QUERY_KEY = "query"
    fun fromJSON(data: String): CatalogFramePost {
      JSONObject(data).apply {
        return CatalogFramePost(
          this.optString(URL_KEY, ""),
          this.optString(PARAMS_KEY, ""),
          this.optString(QUERY_KEY, ""),
        )
      }
    }
  }
}
