package com.hyprmx.android.sdk.webview

import android.os.Build
import android.webkit.JsResult
import android.webkit.PermissionRequest
import android.webkit.WebChromeClient
import android.webkit.WebResourceResponse
import androidx.annotation.RequiresApi
import com.hyprmx.android.sdk.core.SharedInterface
import com.hyprmx.android.sdk.presentation.ArgumentKey
import com.hyprmx.android.sdk.presentation.PresentationEventPublisher
import com.hyprmx.android.sdk.presentation.PublishingEvent
import com.hyprmx.android.sdk.utility.HyprMXLog

internal interface WebViewSharedInterface : HyprMXJSMessageHandler, SharedInterface {
  var adBaseUrl: String
  var adBaseEncoding: String
  var adBaseMime: String
  var userAgent: String
  var queryParams: String

  fun onSizeChanged(currentWidth: Float, currentHeight: Float)
  fun onPermissionRequest(request: PermissionRequest, permissionRequestIds: Int)
  fun onLoadData()
  fun onJSDialog(showCancel: Boolean, url: String, message: String, jsResult: JsResult): Boolean
  fun onShowFileChooser(fileChooserParams: WebChromeClient.FileChooserParams): Boolean
  fun onPageStarted(url: String)
  fun onPageFinished(url: String)
  fun onReceivedError(description: String, errorCode: String, url: String)
  fun shouldInterceptRequest(url: String, scheme: String?, isMainFrame: Boolean): WebResourceResponse?
  fun onHistoryChanged(canNavigateBack: Boolean, canNavigateForward: Boolean, currentIndex: Int, currentItem: String?, currentHost: String?, currentTitle: String?, history: List<String>)
  fun onRenderProcessGone(): Boolean
  fun getWebViewConfiguration(): String
  fun shouldTakeFocus(): Boolean
  fun shouldLoadAboutBlank(): Boolean
}

internal class WebViewSharedConnector(private val eventPublisher: PresentationEventPublisher) :
  WebViewSharedInterface {
  override var adBaseUrl: String
    get() = eventPublisher.getProperty(PublishingEvent.AD_BASE_URL) as String
    set(_) {}
  override var adBaseEncoding: String
    get() = eventPublisher.getProperty(PublishingEvent.AD_BASE_ENCODING) as String
    set(_) {}
  override var adBaseMime: String
    get() = eventPublisher.getProperty(PublishingEvent.AD_BASE_MIME) as String
    set(_) {}
  override var userAgent: String
    get() = eventPublisher.getProperty(PublishingEvent.USER_AGENT) as String
    set(_) {}
  override var queryParams: String
    get() = eventPublisher.getProperty(PublishingEvent.QUERY_PARAMS) as String
    set(_) {}

  override fun onSizeChanged(currentWidth: Float, currentHeight: Float) {
    eventPublisher.publishEvent(
      PublishingEvent.WEB_VIEW_SIZE_CHANGED,
      mapOf(ArgumentKey.CONTAINER_HEIGHT to currentHeight, ArgumentKey.CONTAINER_WIDTH to currentWidth),
    )
  }

  @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
  override fun onPermissionRequest(request: PermissionRequest, permissionRequestIds: Int) {
    eventPublisher.publishEvent(
      PublishingEvent.PERMISSION_REQUEST,
      mapOf(ArgumentKey.PERMISSIONS to request.resources, ArgumentKey.PERMISSION_ID to permissionRequestIds),
    )
  }

  override fun onLoadData() {
    eventPublisher.publishEvent(PublishingEvent.ON_LOAD_DATA)
  }

  override fun onPageStarted(url: String) {
    eventPublisher.publishEvent(PublishingEvent.ON_PAGE_STARTED, mapOf(ArgumentKey.URL to url))
  }

  override fun onPageFinished(url: String) {
    eventPublisher.publishEvent(PublishingEvent.ON_PAGE_FINISHED, mapOf(ArgumentKey.URL to url))
  }

  override fun onReceivedError(description: String, errorCode: String, url: String) {
    eventPublisher.publishEvent(
      PublishingEvent.ON_RECEIVED_ERROR,
      mapOf(ArgumentKey.ERROR_MESSAGE to description, ArgumentKey.ERROR_CODE to errorCode, ArgumentKey.URL to url),
    )
  }

  override fun shouldInterceptRequest(url: String, scheme: String?, isMainFrame: Boolean): WebResourceResponse? {
    eventPublisher.publishEvent(
      PublishingEvent.SHOULD_INTERCEPT_REQUEST,
      mapOf(ArgumentKey.URL to url, ArgumentKey.IS_MAIN_FRAME to isMainFrame, ArgumentKey.SCHEME to scheme),
    )
    return null
  }

  override fun onHistoryChanged(
    canNavigateBack: Boolean,
    canNavigateForward: Boolean,
    currentIndex: Int,
    currentItem: String?,
    currentHost: String?,
    currentTitle: String?,
    history: List<String>,
  ) {
    eventPublisher.publishEvent(
      PublishingEvent.ON_HISTORY_CHANGED,
      mapOf(
        ArgumentKey.CAN_NAVIGATE_BACK to canNavigateBack,
        ArgumentKey.CAN_NAVIGATE_FORWARD to canNavigateForward,
        ArgumentKey.CURRENT_INDEX to currentIndex,
        ArgumentKey.CURRENT_URL to currentItem,
        ArgumentKey.CURRENT_HOST to currentHost,
        ArgumentKey.CURRENT_TITLE to currentTitle,
        ArgumentKey.HISTORY to history.toTypedArray(),
      ),
    )
  }

  override fun onRenderProcessGone(): Boolean {
    return eventPublisher.publishEvent(PublishingEvent.ON_WEBVIEW_CRASH).toString().toBoolean()
  }

  override fun getWebViewConfiguration(): String {
    return eventPublisher.publishEvent(PublishingEvent.WEB_VIEW_CONFIGURATION) as String
  }

  override fun shouldTakeFocus(): Boolean {
    return eventPublisher.publishEvent(PublishingEvent.SHOULD_TAKE_FOCUS) as? Boolean == true
  }

  override fun shouldLoadAboutBlank(): Boolean {
    return eventPublisher.publishEvent(PublishingEvent.SHOULD_LOAD_ABOUT_BLANK) as? Boolean == true
  }

  override fun onJSMessage(methodName: String, argument: String?) {
    HyprMXLog.d("onJSMessage($methodName, $argument")
    eventPublisher.publishEvent(PublishingEvent.ON_JS_MESSAGE, mapOf(ArgumentKey.METHOD_NAME to methodName, ArgumentKey.BODY to argument))
  }

  override var viewModelIdentifier: String
    get() = eventPublisher.viewModelIdentifier
    set(value) {
      eventPublisher.viewModelIdentifier = value
    }

  override fun attach(nativeObject: Any) {
    eventPublisher.setWebViewPresenter(nativeObject)
  }

  override fun destroy() {
    eventPublisher.destroy()
  }

  /**
   * Determine in the shared code if we should display a custom dialog.
   *
   * Always return true because we are handling the event.
   */
  override fun onJSDialog(
    showCancel: Boolean,
    url: String,
    message: String,
    jsResult: JsResult,
  ): Boolean {
    return eventPublisher.publishEvent(
      PublishingEvent.JAVASCRIPT_ALERT_ATTEMPTED,
      mapOf(ArgumentKey.URL to url, ArgumentKey.MESSAGE to message, ArgumentKey.SHOW_CANCEL to showCancel),
    ) as Boolean
  }

  @RequiresApi(Build.VERSION_CODES.LOLLIPOP)
  override fun onShowFileChooser(
    fileChooserParams: WebChromeClient.FileChooserParams,
  ): Boolean {
    return eventPublisher.publishEvent(
      PublishingEvent.ON_SHOW_FILE_CHOOSER,
      mapOf(ArgumentKey.ACCEPT_TYPES to fileChooserParams.acceptTypes),
    ) as Boolean
  }
}
