/*
 * Copyright (c) 2014-2022 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package com.moengage.firebase

import android.content.Context
import android.os.Bundle
import com.moengage.core.LogLevel
import com.moengage.core.internal.SdkInstanceManager
import com.moengage.core.internal.logger.Logger
import com.moengage.core.internal.model.SdkInstance
import com.moengage.firebase.internal.FcmCache
import com.moengage.firebase.internal.FcmInstanceProvider
import com.moengage.firebase.internal.MODULE_TAG
import com.moengage.firebase.internal.TokenRegistrationHandler
import com.moengage.firebase.internal.fetchPushToken
import com.moengage.firebase.listener.NonMoEngagePushListener
import com.moengage.firebase.listener.OnFcmRegistrationCompleteListener
import com.moengage.pushbase.internal.ATTRIBUTE_TOKEN_REGISTERED_APP
import com.moengage.pushbase.internal.PushHelper
import com.moengage.pushbase.listener.TokenAvailableListener

/**
 * Helper class for MoEngage Firebase Module.
 *
 * @author Umang Chamaria
 * @since 9.4.00
 */
public class MoEFireBaseHelper private constructor() {

    private val tag = "${MODULE_TAG}MoEFireBaseHelper"

    /**
     * Helper API to pass the push token to the MoEngage platform.
     *
     * @param context instance of [Context]
     * @param token Generated Push Token.
     * @since 10.3.00
     */
    public fun passPushToken(context: Context, token: String) {
        val instance = SdkInstanceManager.defaultInstance ?: run {
            Logger.print { "$tag passPushToken() : Instance not initialised, cannot process further" }
            return
        }
        passPushToken(context, instance, token)
    }

    /**
     * Pass the payload received from FCM to the MoEngage SDK for showing Push Notification.
     * @param context instance of [Context]
     * @param payload instance of [Bundle]
     */
    public fun passPushPayload(context: Context, payload: Bundle) {
        try {
            PushHelper.getInstance().handlePushPayload(context, payload)
        } catch (e: Exception) {
            Logger.print(LogLevel.ERROR, e) { "$tag passPushPayload() : " }
        }
    }

    /**
     * Pass the payload received from FCM to the MoEngage SDK for showing Push Notification.
     * @param context instance of [Context]
     * @param payload instance of [Map]
     */
    public fun passPushPayload(context: Context, payload: Map<String, String>) {
        try {
            PushHelper.getInstance().handlePushPayload(context, payload)
        } catch (e: Exception) {
            Logger.print(LogLevel.ERROR, e) { "$tag passPushPayload() : " }
        }
    }

    /**
     * API to register for FCM Token. Call this API only if you are delaying the SDK
     * initialisation else this is not required.
     *
     * **Note:** If the application has opted out of token registration this API will not
     * register a push token.
     *
     * @param context instance of [Context]
     * @since 10.3.00
     */
    public fun registerForToken(context: Context) {
        TokenRegistrationHandler.registerForPush(context)
    }

    public companion object {
        private var instance: MoEFireBaseHelper? = null

        @JvmStatic
        public fun getInstance(): MoEFireBaseHelper {
            if (instance == null) {
                synchronized(MoEFireBaseHelper::class.java) {
                    if (instance == null) instance = MoEFireBaseHelper()
                }
            }

            return instance as MoEFireBaseHelper
        }
    }

    /**
     * Helper API to pass the push token to the MoEngage platform.
     *
     * @param context instance of [Context]
     * @param token Generated Push Token.
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     * @since 10.3.00
     */
    public fun passPushToken(context: Context, token: String, appId: String) {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: run {
            Logger.print { "$tag passPushToken() : Instance not initialised, cannot process further" }
            return
        }
        passPushToken(context, instance, token)
    }

    private fun passPushToken(context: Context, sdkInstance: SdkInstance, token: String) {
        FcmInstanceProvider.getControllerForInstance(sdkInstance)
            .processPushToken(context, token, ATTRIBUTE_TOKEN_REGISTERED_APP)
    }

    /**
     * Register a listener to receive a callback whenever a push is received which isn't from
     * MoEngage.
     *
     * @param listener instance of [NonMoEngagePushListener]
     * @since 12.0.00
     */
    public fun addNonMoEngagePushListener(listener: NonMoEngagePushListener) {
        FcmCache.nonMoEngagePushListeners.add(listener)
    }

    /**
     * Remove a registered [NonMoEngagePushListener]
     *
     * @param listener instance of [NonMoEngagePushListener]
     * @since 12.0.00
     */
    public fun removeNonMoEngagePushListener(listener: NonMoEngagePushListener) {
        FcmCache.nonMoEngagePushListeners.remove(listener)
    }

    /**
     * Register a listener to receive a callbacks whenever a token is generated or updated.
     *
     * @param listener instance of [TokenAvailableListener]
     * @since 12.0.00
     */
    public fun addTokenListener(listener: TokenAvailableListener) {
        FcmCache.tokenListeners.add(listener)
    }

    /**
     * Remove registered [TokenAvailableListener]
     *
     * @param listener instance of [TokenAvailableListener]
     * @since 12.0.00
     */
    public fun removeTokenListener(listener: TokenAvailableListener) {
        FcmCache.tokenListeners.remove(listener)
    }

    /**
     * Get saved Push Token(if any) for the account configured as default.
     *
     * @param context Instance of [Context]
     *
     * @return Saved Push token, else null
     *
     * @since 12.0.00
     */
    public fun getPushToken(context: Context): String? {
        val instance = SdkInstanceManager.defaultInstance ?: return null
        return getPushToken(context, instance)
    }

    /**
     * Get saved Push Token(if any) for the given account.
     *
     * @param context Instance of [Context]
     * @param appId Account identifier, APP ID on the MoEngage Dashboard.
     *
     * @return Saved Push token, else null
     *
     * @since 12.0.00
     */
    public fun getPushToken(context: Context, appId: String): String? {
        val instance = SdkInstanceManager.getInstanceForAppId(appId) ?: return null
        return getPushToken(context, instance)
    }

    private fun getPushToken(context: Context, sdkInstance: SdkInstance): String? {
        val token =
            FcmInstanceProvider.getRepositoryForInstance(context, sdkInstance).getPushToken()
        return token.ifBlank { null }
    }

    /**
     * Generates Push Token.
     *
     * **NOTE** The generated token will not be saved.
     *
     * @param listener instance of [OnFcmRegistrationCompleteListener]
     *
     * @since 12.7.00
     */
    public fun generatePushToken(listener: OnFcmRegistrationCompleteListener) {
        fetchPushToken(listener)
    }
}