/*
 * Copyright (c) 2014-2023 MoEngage Inc.
 *
 * All rights reserved.
 *
 *  Use of source code or binaries contained within MoEngage SDK is permitted only to enable use of the MoEngage platform by customers of MoEngage.
 *  Modification of source code and inclusion in mobile apps is explicitly allowed provided that all other conditions are met.
 *  Neither the name of MoEngage nor the names of its contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 *  Redistribution of source code or binaries is disallowed except with specific prior written permission. Any such redistribution must retain the above copyright notice, this list of conditions and the following disclaimer.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package com.moengage.firebase.internal

import com.google.firebase.messaging.FirebaseMessaging
import com.moengage.core.LogLevel
import com.moengage.core.internal.global.GlobalResources
import com.moengage.core.internal.logger.Logger
import com.moengage.firebase.listener.OnFcmRegistrationCompleteListener
import com.moengage.pushbase.model.TokenRegistrationResult

/**
 * @author Arshiya Khanum
 */

private val TAG = "FcmUtils"
private const val ID_PREFIX = "|ID|"

/**
 * Removes multiplexing extras from the given string
 * @since 6.3.0
 */
internal fun ripMultiplexingExtras(token: String): String {
    return if (token.isNotBlank() && token.startsWith(ID_PREFIX)) {
        token.substring(7)
    } else {
        token
    }
}

/**
 * Fetches push token, triggers callback with the if available else with null
 * @since 6.3.0
 */
internal fun fetchPushToken(listener: OnFcmRegistrationCompleteListener) {
    try {
        Logger.print { "$TAG fetchPushToken() : Will try to fetch push token." }
        FirebaseMessaging.getInstance().token.addOnCompleteListener { task ->
            val result = if (task.isSuccessful) {
                TokenRegistrationResult(true, ripMultiplexingExtras(task.result))
            } else {
                TokenRegistrationResult(false, null)
            }
            GlobalResources.mainThread.post {
                try {
                    listener.onComplete(result)
                } catch (t: Throwable) {
                    Logger.print(LogLevel.ERROR, t) { "$TAG fetchPushToken() : " }
                }
            }
        }
    } catch (t: Throwable) {
        Logger.print(LogLevel.ERROR, t) { "$TAG fetchPushToken() : " }
    }
}